% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scp.R
\name{scp}
\alias{scp}
\alias{print.scp}
\alias{summary.scp}
\alias{print.summary.scp}
\title{Classical split conformal prediction method}
\usage{
scp(
  object,
  alpha = 1 - 0.01 * object$level,
  symmetric = FALSE,
  ncal = 10,
  rolling = FALSE,
  quantiletype = 1,
  weightfun = NULL,
  kess = FALSE,
  update = FALSE,
  na.rm = TRUE,
  ...
)
}
\arguments{
\item{object}{An object of class \code{"cvforecast"}. It must have an argument
\code{x} for original univariate time series, an argument \code{MEAN} for
point forecasts and \code{ERROR} for forecast errors on validation set.
See the results of a call to \code{\link{cvforecast}}.}

\item{alpha}{A numeric vector of significance levels to achieve a desired
coverage level \eqn{1-\alpha}.}

\item{symmetric}{If \code{TRUE}, symmetric nonconformity scores (i.e. \eqn{|e_{t+h|t}|})
are used. If \code{FALSE}, asymmetric nonconformity scores (i.e. \eqn{e_{t+h|t}})
are used, and then upper bounds and lower bounds are produced separately.}

\item{ncal}{Length of the calibration set. If \code{rolling = FALSE}, it denotes
the initial period of calibration sets. Otherwise, it indicates
the period of every rolling calibration set.}

\item{rolling}{If \code{TRUE}, a rolling window strategy will be adopted to
form the calibration set. Otherwise, expanding window strategy will be used.}

\item{quantiletype}{An integer between 1 and 9 determining the type of
quantile estimator to be used. Types 1 to 3 are for discontinuous quantiles,
types 4 to 9 are for continuous quantiles. See the
\code{\link[ggdist]{weighted_quantile}} function in the ggdist package.}

\item{weightfun}{Function to return a vector of weights used for sample quantile
computation. Its first argument must be an integer indicating the number of
observations for which weights are generated. If \code{NULL}, equal weights
will be used for sample quantile computation. Currently, only non-data-dependent
weights are supported.}

\item{kess}{If \code{TRUE}, Kish's effective sample size is used for sample
quantile computation.}

\item{update}{If \code{TRUE}, the function will be compatible with the
\code{update}(\link{update.cpforecast}) function, allowing for easy updates of conformal prediction.}

\item{na.rm}{If \code{TRUE}, corresponding entries in sample values and weights
are removed if either is \code{NA} when calculating sample quantile.}

\item{...}{Other arguments are passed to \code{weightfun}.}
}
\value{
A list of class \code{c("scp", "cvforecast", "forecast")}
with the following components:
\item{x}{The original time series.}
\item{series}{The name of the series \code{x}.}
\item{xreg}{Exogenous predictor variables used, if applicable.}
\item{method}{A character string "scp".}
\item{cp_times}{The number of times the conformal prediction is performed in
cross-validation.}
\item{MEAN}{Point forecasts as a multivariate time series, where the \eqn{h}th column
holds the point forecasts for forecast horizon \eqn{h}. The time index
corresponds to the period for which the forecast is produced.}
\item{ERROR}{Forecast errors given by
\eqn{e_{t+h|t} = y_{t+h}-\hat{y}_{t+h|t}}{e[t+h] = y[t+h]-f[t+h]}.}
\item{LOWER}{A list containing lower bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{UPPER}{A list containing upper bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{level}{The confidence values associated with the prediction intervals.}
\item{call}{The matched call.}
\item{model}{A list containing detailed information about the \code{cvforecast} and \code{conformal} models.}
If \code{mean} is included in the \code{object}, the components \code{mean},
\code{lower}, and \code{upper} will also be returned, showing the information
about the test set forecasts generated using all available observations.
}
\description{
Compute prediction intervals and other information by
applying the classical split conformal prediction (SCP) method.
}
\details{
Consider a vector \eqn{s_{t+h|t}} that contains the nonconformity scores for the
\eqn{h}-step-ahead forecasts.

If \code{symmetric} is \code{TRUE}, \eqn{s_{t+h|t}=|e_{t+h|t}|}.
When \code{rolling} is \code{FALSE}, the \eqn{(1-\alpha)}-quantile
\eqn{\hat{q}_{t+h|t}} are computed successively on expanding calibration sets
\eqn{s_{1+h|1},\dots,s_{t|t-h}}, for \eqn{t=\mathrm{ncal}+h,\dots,T}. Then the
prediction intervals will be
\eqn{[\hat{y}_{t+h|t}-\hat{q}_{t+h|t}, \hat{y}_{t+h|t}+\hat{q}_{t+h|t}]}.
When \code{rolling} is \code{TRUE}, the calibration sets will be of same length
\code{ncal}.

If \code{symmetric} is \code{FALSE}, \eqn{s_{t+h|t}^{u}=e_{t+h|t}} for upper
interval bounds and \eqn{s_{t+h|t}^{l} = -e_{t+h|t}} for lower bounds.
Instead of computing \eqn{(1-\alpha)}-quantile, \eqn{(1-\alpha/2)}-quantiles for lower
bound (\eqn{\hat{q}_{t+h|t}^{l}}) and upper bound (\eqn{\hat{q}_{t+h|t}^{u}})
are calculated based on their nonconformity scores, respectively.
Then the prediction intervals will be
\eqn{[\hat{y}_{t+h|t}-\hat{q}_{t+h|t}^{l}, \hat{y}_{t+h|t}+\hat{q}_{t+h|t}^{u}]}.
}
\examples{
# Simulate time series from an AR(2) model
library(forecast)
series <- arima.sim(n = 200, list(ar = c(0.8, -0.5)), sd = sqrt(1))

# Cross-validation forecasting
far2 <- function(x, h, level) {
  Arima(x, order = c(2, 0, 0)) |>
    forecast(h = h, level)
}
fc <- cvforecast(series, forecastfun = far2, h = 3, level = 95,
                 forward = TRUE, initial = 1, window = 50)

# Classical conformal prediction with equal weights
scpfc <- scp(fc, symmetric = FALSE, ncal = 50, rolling = TRUE)
print(scpfc)
summary(scpfc)

# Classical conformal prediction with exponential weights
expweight <- function(n) {
  0.99^{n+1-(1:n)}
}
scpfc_exp <- scp(fc, symmetric = FALSE, ncal = 50, rolling = TRUE,
                 weightfun = expweight, kess = TRUE)

}
\seealso{
\code{\link[ggdist]{weighted_quantile}}
}
