# barchart ----------------------------------------------------------------

#' Bar chart with horizontal bar and data label.
#'
#' Axis-x has the frequency and data label has the percent value.
#'
#' @param data a dataframe object
#' @param variable variable name
#' @param axis_title axis y label. Defaults to "".
#' @param fill bar color. Character scale with hexidecimal color ou named color. Defaults to "blue".
#' @param axis_title_axis_size integer scalar. Size of axis labels. Defaults to 22.
#' @param axis_text_axis_size integer scalar. Size of values annotated in axis. Defaults to 22.
#' @param data_label_size integer scalar. Size of data labels. Defaults to 15
#' @param sort logical scalar. If TRUE, the are ordered using frequency. Defaults to TRUE.
#'
#' @returns a ggplot2 object
#' @export
#' @import ggplot2 dplyr forcats scales rlang
#'
#' @examples
#' df <- data.frame(var = sample(c("A", "B"),
#'  size = 100, replace = TRUE))
#' bar_chart(df, var)
bar_chart <- function (data, variable, axis_title = "",
                       fill = "blue",
                       axis_title_axis_size = 22,
                       axis_text_axis_size = 22,
                       data_label_size = 15,
                       sort = TRUE) {
  p <- ggplot(data, aes(y = {{ variable }}))

  if (sort) {
    p <- ggplot(data, aes(y = fct_rev(fct_infreq({{ variable }}))))
  }

  p +
    geom_bar(fill = fill) +
    geom_text(aes(x = after_stat(count),
                  label = scales::label_percent(accuracy = 0.1, big.mark = ".", decimal.mark = ",")(after_stat(count / sum(count)))),
              hjust = -0.1,
              stat = "count", size = data_label_size) +
    labs(x = "Contagem", y = axis_title) +
    theme_minimal() +
    theme(
      axis.title = element_text(size = axis_title_axis_size),
      axis.text = element_text(size = axis_text_axis_size)
    )
}


# bar chart for checkbox field --------------------------------------------

#' Bar chart with frequency and data label with percent.
#'
#' This function builds a bar chart with ggplot2 for a field with checkbox (user can select 2 or more options). Each option is one separated column.
#'
#' @param data dataframe object
#' @param columns character vector. Columns to count.
#' @param sucess character scalar. Category indicating the sucess. Defaults to "checked".
#' @param labels character vector.Label of each category. Defaults to NULL.
#' @param axis_title axis y label. Defaults to "".
#' @param fill bar color. Character scale with hexidecimal color ou named color. Defaults to "blue".
#' @param axis_title_axis_size integer scalar. Size of axis labels. Defaults to 22.
#' @param axis_text_axis_size integer scalar. Size of values annotated in axis. Defaults to 22.
#' @param data_label_size integer scalar. Size of data labels. Defaults to 15
#' @param sort logical scalar. If TRUE, the are ordered using frequency. Defaults to TRUE.
#'
#' @returns a ggplot2 object
#' @import ggplot2 dplyr forcats scales rlang
#' @export
#'
#' @examples
#' df <- data.frame(
#'  x1 = c("checked", "checked", "unchecked"),
#'  x2 = c("checked", "unchecked", "checked")
#')
#' bar_chart_checkbox(df, c("x1", "x2"))
bar_chart_checkbox <- function (data, columns, sucess = "checked", labels = NULL, axis_title = "",
                                fill = "blue",
                                axis_title_axis_size = 22,
                                axis_text_axis_size = 22,
                                data_label_size = 15,
                                sort = TRUE) {
  sample_size <- nrow(data)

  res <- columns |>
    lapply(\(column) {
      c(
        "n" = sum(dplyr::pull(data, {{ column }}) == sucess, na.rm = T),
        "percent" = sum(dplyr::pull(data, {{ column }}) == sucess, na.rm = T) / sample_size
      )
    }) |>
    bind_rows() |>
    mutate(coluna = columns) |>
    relocate(coluna, .before = n)

  if (!is.null(labels)) {
    res <- res |>
      mutate(coluna = labels)
  }

  p <- res |>
    ggplot(aes(y = coluna, x = n))

  if (sort) {
    p <- res |>
      arrange(n) |>
      ggplot(aes(y = fct(coluna), x = n))
  }

  p +
    geom_bar(fill = fill, stat = "identity") +
    geom_text(aes(x = n, label = scales::label_percent(accuracy = 0.1, big.mark = ".", decimal.mark = ",")(percent)),
              hjust = -0.1,
              stat = "identity", size = data_label_size) +
    labs(x = "Contagem", y = axis_title) +
    theme_minimal() +
    theme(
      axis.title = element_text(size = axis_title_axis_size),
      axis.text = element_text(size = axis_text_axis_size)
    )
}
