#' Get DRI from a Group of Participants
#'
#' `get_dri` calculates the deliberation reasoning index (DRI) for a group of
#' deliberation participants
#'
#' @param ic dataframe generated by `get_dri_ic(data)`
#' @param adjusted a logical indicating whether to use the original or adjusted
#' DRI calculation formula
#'
#' @returns the group-level DRI value
#' @export
#'
#' @seealso [get_dri_ic()] to generate `ic` parameter
#' @family IC methods
#'
#' @examples
#'
#' # get pre-deliberation (stage_id == 1) data from BEP case
#' data <- human_data[human_data$stage_id == 1 & human_data$case == "BEP", ]
#'
#' # calculate IC
#' ic <- get_dri_ic(data)
#'
#' # generate DRI
#' get_dri(ic)
#'
#' # same as the mean of individual DRIs
#' mean(get_dri_ind(ic)$dri)
#'
get_dri <- function(ic, adjusted = TRUE){

  # scalar penalty based on strength of signal (|ccor| and |pcor|)
  if (adjusted) {
    max_cor <- pmax(abs(ic$ccor), abs(ic$pcor))
    penalty <- ifelse(max_cor <= 0.2, max_cor / 0.2, 1)
    dist <- 1 - mean((1 - ic$dj) * penalty)
  } else {
    dist <- mean(ic$dj)
  }

  lambda <- sqrt(2) / 2
  dri <- ((-2 * dist) + lambda) / lambda
  return(dri)
}
