#' @title Validate a constructed GP, DGP, or linked (D)GP emulator
#'
#' @description This function calculates Leave-One-Out (LOO) cross validation or Out-Of-Sample (OOS) validation statistics for a constructed GP, DGP, or linked (D)GP emulator.
#'
#' @param object can be one of the following:
#' * the S3 class `gp`.
#' * the S3 class `dgp`.
#' * the S3 class `lgp`.
#' @param x_test OOS testing input data:
#' * if `object` is an instance of the `gp` or `dgp` class, `x_test` is a matrix where each row is a new input location to be used for validating the emulator and each column is an input dimension.
#' * if `object` is an instance of the `lgp` class, `x_test` must be a matrix representing the global input, where each row corresponds to a test data point and each column represents a global input dimension.
#'   The column indices in `x_test` must align with the indices specified in the `From_Output` column of the `struc` data frame (used in [lgp()]),
#'   corresponding to rows where the `From_Emulator` column is `"Global"`.
#'
#' `x_test` must be provided if `object` is an instance of the `lgp`. `x_test` must also be provided if `y_test` is provided. Defaults to `NULL`, in which case LOO validation is performed.
#' @param y_test the OOS output data corresponding to `x_test`:
#' * if `object` is an instance of the `gp` class, `y_test` is a matrix with only one column where each row represents the output corresponding to the matching row of `x_test`.
#' * if `object` is an instance of the `dgp` class, `y_test` is a matrix where each row represents the output corresponding to the matching row of `x_test` and with columns representing output dimensions.
#' * if `object` is an instance of the `lgp` class, `y_test` can be a single matrix or a list of matrices:
#'   - if `y_test` is a single matrix, then there should be only one emulator in the final layer of the linked emulator system and `y_test`
#'     represents the emulator's output with rows being testing positions and columns being output dimensions.
#'   - if `y_test` is a list, then `y_test` should have *L* matrices, where *L* is the number of emulators in the final layer of the system.
#'     Each matrix has its rows corresponding to testing positions and columns corresponding to output dimensions of the associated emulator
#'     in the final layer.
#'
#' `y_test` must be provided if `object` is an instance of the `lgp`. `y_test` must also be provided if `x_test` is provided. Defaults to `NULL`, in which case LOO validation is performed.
#' @param method `r new_badge("updated")` the prediction approach to use for validation: either the mean-variance approach (`"mean_var"`) or the sampling approach (`"sampling"`). For details see [predict()].
#'      Defaults to `"mean_var"`.
#' @param sample_size the number of samples to draw for each given imputation if `method = "sampling"`. Defaults to `50`.
#' @param verb a bool indicating if trace information for validation should be printed during function execution.
#'     Defaults to `TRUE`.
#' @param M the size of the conditioning set for the Vecchia approximation in emulator validation. This argument is only used if the emulator `object`
#'     was constructed under the Vecchia approximation. Defaults to `50`.
#' @param force a bool indicating whether to force LOO or OOS re-evaluation when the `loo` or `oos` slot already exists in `object`. When `force = FALSE`,
#'     [validate()] will only re-evaluate the emulators if the `x_test` and `y_test` are not identical to the values in the `oos` slot. If the existing `loo` or `oos` validation used a different `M` in a Vecchia approximation or a different `method` to the one prescribed in this call, the emulator will be re-evaluated. Set `force` to `TRUE` when LOO or OOS re-evaluation
#'     is required. Defaults to `FALSE`.
#' @param cores the number of processes to be used for validation. If set to `NULL`, the number of processes is set to `max physical cores available %/% 2`.
#'     Defaults to `1`.
#' @param ... N/A.
#'
#' @return
#' * If `object` is an instance of the `gp` class, an updated `object` is returned with an additional slot called `loo` (for LOO cross validation) or
#'   `oos` (for OOS validation) that contains:
#'   - two slots called `x_train` (or `x_test`) and `y_train` (or `y_test`) that contain the validation data points for LOO (or OOS).
#'   - a column matrix called `mean`, if `method = "mean_var"`, or `median`, if `method = "sampling"`, that contains the predictive means or medians of the
#'     GP emulator at validation positions.
#'   - three column matrices called `std`, `lower`, and `upper` that contain the predictive standard deviations and credible intervals of the
#'     GP emulator at validation positions. If `method = "mean_var"`, the upper and lower bounds of a credible interval are two standard deviations above
#'     and below the predictive mean. If `method = "sampling"`, the upper and lower bounds of a credible interval are 2.5th and 97.5th percentiles.
#'   - a numeric value called `rmse` that contains the root mean/median squared error of the GP emulator.
#'   - a numeric value called `nrmse` that contains the (max-min) normalized root mean/median squared error of the GP emulator. The max-min normalization
#'     uses the maximum and minimum values of the validation outputs contained in `y_train` (or `y_test`).
#'   - an integer called `M` that contains the size of the conditioning set used for the Vecchia approximation, if used, for emulator validation.
#'   - an integer called `sample_size` that contains the number of samples used for validation if `method = "sampling"`.
#'
#'   The rows of matrices (`mean`, `median`, `std`, `lower`, and `upper`) correspond to the validation positions.
#' * If `object` is an instance of the `dgp` class, an updated `object` is returned with an additional slot called `loo` (for LOO cross validation) or
#'   `oos` (for OOS validation) that contains:
#'   - two slots called `x_train` (or `x_test`) and `y_train` (or `y_test`) that contain the validation data points for LOO (or OOS).
#'   - a matrix called `mean`, if `method = "mean_var"`, or `median`, if `method = "sampling"`, that contains the predictive means or medians of the
#'     DGP emulator at validation positions.
#'   - three matrices called `std`, `lower`, and `upper` that contain the predictive standard deviations and credible intervals of the
#'     DGP emulator at validation positions. If `method = "mean_var"`, the upper and lower bounds of a credible interval are two standard deviations above
#'     and below the predictive mean. If `method = "sampling"`, the upper and lower bounds of a credible interval are 2.5th and 97.5th percentiles.
#'   - a vector called `rmse` that contains the root mean/median squared errors of the DGP emulator across different output
#'     dimensions.
#'   - a vector called `nrmse` that contains the (max-min) normalized root mean/median squared errors of the DGP emulator across different output
#'     dimensions. The max-min normalization uses the maximum and minimum values of the validation outputs contained in `y_train` (or `y_test`).
#'   - an integer called `M` that contains size of the conditioning set used for the Vecchia approximation, if used, for emulator validation.
#'   - an integer called `sample_size` that contains the number of samples used for validation if `method = "sampling"`.
#'
#'   The rows and columns of matrices (`mean`, `median`, `std`, `lower`, and `upper`) correspond to the validation positions and DGP emulator output
#' dimensions, respectively.
#' * `r new_badge("updated")` If `object` is an instance of the `dgp` class with a categorical likelihood, an updated `object` is returned with an additional slot called `loo`
#'   (for LOO cross validation) or `oos` (for OOS validation) that contains:
#'   - two slots called `x_train` (or `x_test`) and `y_train` (or `y_test`) that contain the validation data points for LOO (or OOS).
#'   - a vector called `label` that contains predictive labels from the DGP emulator at validation positions.
#'   - a matrix called `probability` that contains mean predictive probabilities for each class from the DGP emulator at validation positions. The matrix has its rows corresponding
#'     to validation positions and columns corresponding to different classes.
#'   - a scalar called `log_loss` that represents the log loss of the trained DGP classifier. Log loss measures the
#'     accuracy of probabilistic predictions, with lower values indicating better classification performance. `log_loss` ranges from `0` to positive infinity, where a
#'     value closer to `0` suggests more confident and accurate predictions.
#'   - a scalar called `accuracy` that represents the accuracy of the trained DGP classifier. Accuracy measures the proportion of correctly classified instances among
#'     all predictions, with higher values indicating better classification performance. accuracy ranges from `0` to `1`, where a value closer to `1` suggests more
#'     reliable and precise predictions.
#'   - a slot named `method` indicating whether the matrix in the `probability` slot were obtained using the `"mean-var"` method or the `"sampling"` method.
#'   - an integer called `M` that contains size of the conditioning set used for the Vecchia approximation, if used, in emulator validation.
#'   - an integer called `sample_size` that contains the number of samples used for validation.
#' * If `object` is an instance of the `lgp` class, an updated `object` is returned with an additional slot called `oos` (for OOS validation) that contains:
#'   - two slots called `x_test` and `y_test` that contain the validation data points for OOS.
#'   - a list called `mean`, if `method = "mean_var"`, or `median`, if `method = "sampling"`, that contains the predictive means or medians of
#'     the linked (D)GP emulator at validation positions.
#'   - three lists called `std`, `lower`, and `upper` that contain the predictive standard deviations and credible intervals of
#'     the linked (D)GP emulator at validation positions. If `method = "mean_var"`, the upper and lower bounds of a credible interval are two standard
#'     deviations above and below the predictive mean. If `method = "sampling"`, the upper and lower bounds of a credible interval are 2.5th and 97.5th percentiles.
#'   - a list called `rmse` that contains the root mean/median squared errors of the linked (D)GP emulator.
#'   - a list called `nrmse` that contains the (max-min) normalized root mean/median squared errors of the linked (D)GP emulator. The max-min normalization
#'     uses the maximum and minimum values of the validation outputs contained in `y_test`.
#'   - an integer called `M` that contains size of the conditioning set used for the Vecchia approximation, if used, in emulator validation.
#'   - an integer called `sample_size` that contains the number of samples used for validation if `method = "sampling"`.
#'
#'   Each element in `mean`, `median`, `std`, `lower`, `upper`, `rmse`, and `nrmse` corresponds to a (D)GP emulator in the final layer of the linked (D)GP
#' emulator.
#'
#' @note
#' * When both `x_test` and `y_test` are `NULL`, LOO cross validation will be implemented. Otherwise, OOS validation will
#'   be implemented. LOO validation is only applicable to a GP or DGP emulator (i.e., `object` is an instance of the `gp` or `dgp`
#'   class). If a linked (D)GP emulator (i.e., `object` is an instance of the `lgp` class) is provided, `x_test` and `y_test` must
#'   also be provided for OOS validation.
#' @details See further examples and tutorials at <`r get_docs_url()`>.
#' @examples
#' \dontrun{
#'
#' # See gp(), dgp(), or lgp() for an example.
#' }
#' @md
#' @name validate
#' @export
validate <- function(object, x_test, y_test, method, sample_size, verb, M, force, cores, ...){
  UseMethod("validate")
}

#' @rdname validate
#' @method validate gp
#' @export
validate.gp <- function(object, x_test = NULL, y_test = NULL, method = "mean_var", sample_size = 50, verb = TRUE, M = 50, force = FALSE, cores = 1, ...) {
  if ( is.null(pkg.env$dgpsi) ) {
    init_py(verb = F)
    if (pkg.env$restart) return(invisible(NULL))
  }
  #check class
  if ( !inherits(object,"gp") ) stop("'object' must be an instance of the 'gp' class.", call. = FALSE)
  if ( reticulate::py_is_null_xptr(object$constructor_obj) ) stop("The Python session originally associated with 'object' is no longer active. Please rebuild the emulator or, if it was saved using dgpsi::write(), load it into the R session with dgpsi::read().", call. = FALSE)
  #check core number
  if( !is.null(cores) ) {
    cores <- as.integer(cores)
    if ( cores < 1 ) stop("The core number must be >= 1.", call. = FALSE)
  }
  M <- as.integer(M)

  if ( method!='mean_var' & method!='sampling' ) stop("'method' can only be either 'mean_var' or 'sampling'.", call. = FALSE)

  sample_size <- as.integer(sample_size)
  #For LOO
  if (is.null(x_test) & is.null(y_test)){
    #check existing LOO
    if ( isFALSE(force) ){
      if ( "loo" %in% names(object) ){
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( isTRUE(verb) ) message(" LOO results found in the gp object.")
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( (method == 'mean_var')&("mean" %in% names(object$loo)) & (M == object$loo$M) ){
          if ( isTRUE(verb) ) message(" LOO re-evaluation not needed.")
          if ( isTRUE(verb) ) message("Exporting gp object without re-evaluation ...", appendLF = FALSE)
          if ( isTRUE(verb) ) Sys.sleep(0.5)
          if ( isTRUE(verb) ) message(" done")
          return(object)
        } else if ( (method == 'sampling')&("median" %in% names(object$loo)) & (M == object$loo$M) ){
          if (sample_size == object$loo$sample_size){
            if ( isTRUE(verb) ) message(" LOO re-evaluation not needed.")
            if ( isTRUE(verb) ) message("Exporting gp object without re-evaluation ...", appendLF = FALSE)
            if ( isTRUE(verb) ) Sys.sleep(0.5)
            if ( isTRUE(verb) ) message(" done")
            return(object)
          } else {
            if ( isTRUE(verb) ) message(" LOO re-evaluation needed.")
            if ( isTRUE(verb) ) message("Start re-evaluation: ")
          }
        } else {
          if ( isTRUE(verb) ) message(" LOO re-evaluation needed.")
          if ( isTRUE(verb) ) message("Start re-evaluation: ")
        }
      }
    }

    if ( isTRUE(verb) ) message("Initializing the LOO ...", appendLF = FALSE)
    x_train <- object$data$X
    y_train <- object$data$Y
    dat <- list('x_train' = x_train, 'y_train' = y_train)
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Calculating the LOO ...", appendLF = FALSE)
    res <- object$emulator_obj$loo(method = method, sample_size = sample_size, m = M)
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Saving results to the slot 'loo' in the gp object ...", appendLF = FALSE)
    if ( method=='mean_var' ){
      dat[["mean"]] <- res[[1]]
      dat[["std"]] <- sqrt(res[[2]])
      dat[["lower"]] <- dat$mean-2*dat$std
      dat[["upper"]] <- dat$mean+2*dat$std
      dat[["rmse"]] <- sqrt(mean((dat$mean-dat$y_train)^2))
      dat[["nrmse"]] <- dat$rmse/(max(dat$y_train)-min(dat$y_train))
    } else if ( method=='sampling' ){
      quant <- t(pkg.env$np$quantile(res, c(0.025, 0.5, 0.975), axis=1L))
      std <- pkg.env$np$std(res, axis=1L, keepdims=TRUE)
      dat[["median"]] <- quant[,2,drop=F]
      dat[["std"]] <- std
      dat[["lower"]] <- quant[,1,drop=F]
      dat[["upper"]] <- quant[,3,drop=F]
      dat[["rmse"]] <- sqrt(mean((dat$median-dat$y_train)^2))
      dat[["nrmse"]] <- dat$rmse/(max(dat$y_train)-min(dat$y_train))
    }
    dat[["M"]] <- M
    if (method == "sampling"){
      dat[["sample_size"]] <- sample_size
    }
    object$loo <- dat
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    return(object)
    #For OOS
  } else if (!is.null(x_test) & !is.null(y_test)) {
    rownames(x_test) <- NULL
    rownames(y_test) <- NULL
    #x_test <- unname(x_test)
    #y_test <- unname(y_test)
    if ( !is.matrix(x_test)&!is.vector(x_test) ) stop("'x_test' must be a vector or a matrix.", call. = FALSE)
    if ( !is.matrix(y_test)&!is.vector(y_test) ) stop("'y_test' must be a vector or a matrix.", call. = FALSE)
    if ( is.vector(x_test) ) {
      if ( ncol(object$data$X)!=1 ){
        x_test <- matrix(x_test, nrow = 1)
      } else {
        x_test <- as.matrix(x_test)
      }
    }
    if ( is.vector(y_test) ) y_test <- as.matrix(y_test)
    if ( nrow(x_test)!=nrow(y_test) ) stop("'x_test' and 'y_test' have different number of data points.", call. = FALSE)
    if ( ncol(x_test) != ncol(object$data$X) ) stop("'x_test' must have the same number of dimensions as the training input.", call. = FALSE)
    #check existing OOS
    if ( isFALSE(force) ){
      if ( "oos" %in% names(object) ){
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( isTRUE(verb) ) message(" OOS results found in the gp object.")
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( identical(object$oos$x_test, x_test) & identical(object$oos$y_test, y_test) & (method == 'mean_var')&("mean" %in% names(object$oos)) & (M == object$oos$M) ){
          if ( isTRUE(verb) ) message(" OOS re-evaluation not needed.")
          if ( isTRUE(verb) ) message("Exporting gp object without re-evaluation ...", appendLF = FALSE)
          if ( isTRUE(verb) ) Sys.sleep(0.5)
          if ( isTRUE(verb) ) message(" done")
          return(object)
        } else if ( ( identical(object$oos$x_test, x_test) & identical(object$oos$y_test, y_test) & (method == 'sampling')&("median" %in% names(object$oos)) & (M == object$oos$M) ) ){
          if ( sample_size == object$oos$sample_size ){
            if ( isTRUE(verb) ) message(" OOS re-evaluation not needed.")
            if ( isTRUE(verb) ) message("Exporting gp object without re-evaluation ...", appendLF = FALSE)
            if ( isTRUE(verb) ) Sys.sleep(0.5)
            if ( isTRUE(verb) ) message(" done")
            return(object)
          } else {
            if ( isTRUE(verb) ) message(" OOS re-evaluation needed.")
            if ( isTRUE(verb) ) message("Start re-evaluation: ")
          }
        } else {
          if ( isTRUE(verb) ) message(" OOS re-evaluation needed.")
          if ( isTRUE(verb) ) message("Start re-evaluation: ")
        }
      }
    }

    if ( isTRUE(verb) ) message("Initializing the OOS ...", appendLF = FALSE)
    dat <- list('x_test' = x_test,'y_test' = y_test)
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Calculating the OOS ...", appendLF = FALSE)
    rep_x <- pkg.env$np$unique(x_test, return_inverse=TRUE, axis=0L)
    x_test_unique <- rep_x[[1]]
    rep <- rep_x[[2]] + 1

    if ( identical(cores,as.integer(1)) ){
      res <- object$emulator_obj$predict(x_test_unique, method = method, sample_size = sample_size, m = M)
    } else {
      res <- object$emulator_obj$ppredict(x_test_unique, method = method, sample_size = sample_size, m = M, core_num = cores)
    }
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Saving results to the slot 'oos' in the gp object ...", appendLF = FALSE)
    if ( method == 'mean_var' ){
      dat[["mean"]] <- res[[1]][rep,,drop=FALSE]
      dat[["std"]] <- sqrt(res[[2]][rep,,drop=FALSE])
      dat[["lower"]] <- dat$mean-2*dat$std
      dat[["upper"]] <- dat$mean+2*dat$std
      dat[["rmse"]] <- unname(sqrt(mean((dat$mean-dat$y_test)^2)))
      dat[["nrmse"]] <- unname(dat$rmse/(max(dat$y_test)-min(dat$y_test)))
    } else if ( method == 'sampling' ){
      quant <- t(pkg.env$np$quantile(res, c(0.025, 0.5, 0.975), axis=1L))[rep,,drop=FALSE]
      std <- pkg.env$np$std(res, axis=1L, keepdims=TRUE)[rep,,drop=FALSE]
      dat[["median"]] <- quant[,2,drop=F]
      dat[["std"]] <- std
      dat[["lower"]] <- quant[,1,drop=F]
      dat[["upper"]] <- quant[,3,drop=F]
      dat[["rmse"]] <- unname(sqrt(mean((dat$median-dat$y_test)^2)))
      dat[["nrmse"]] <- unname(dat$rmse/(max(dat$y_test)-min(dat$y_test)))
    }
    dat[["M"]] <- M
    if (method == "sampling"){
      dat[["sample_size"]] <- sample_size
    }
    object$oos <- dat
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    return(object)
    #For other cases
  } else {
    stop("Either 'x_test' or 'y_test' is not given.", call. = FALSE)
  }
}

#' @rdname validate
#' @method validate dgp
#' @export
validate.dgp <- function(object, x_test = NULL, y_test = NULL, method = "mean_var", sample_size = 50, verb = TRUE, M = 50, force = FALSE, cores = 1, ...) {
  if ( is.null(pkg.env$dgpsi) ) {
    init_py(verb = F)
    if (pkg.env$restart) return(invisible(NULL))
  }
  #check class
  if ( !inherits(object,"dgp") ) stop("'object' must be an instance of the 'dgp' class.", call. = FALSE)
  if ( reticulate::py_is_null_xptr(object$constructor_obj) ) stop("The Python session originally associated with 'object' is no longer active. Please rebuild the emulator or, if it was saved using dgpsi::write(), load it into the R session with dgpsi::read().", call. = FALSE)
  #check core number
  if( !is.null(cores) ) {
    cores <- as.integer(cores)
    if ( cores < 1 ) stop("'cores' must be >= 1.", call. = FALSE)
  }
  M <- as.integer(M)

  L = object$constructor_obj$n_layer
  final_node <- object$specs[[paste('layer', L, sep="")]][['node1']]
  if ("type" %in% names(final_node) && final_node$type == "Categorical") {
    is.categorical <- TRUE
  } else {
    is.categorical <- FALSE
  }

  if ( method!='mean_var' & method!='sampling' ) stop("'method' can only be either 'mean_var' or 'sampling'.", call. = FALSE)

  sample_size <- as.integer(sample_size)
  #For LOO
  if (is.null(x_test) & is.null(y_test)){
    #check existing LOO
    if ( isFALSE(force) ){
      if ( "loo" %in% names(object) ){
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( isTRUE(verb) ) message(" LOO results found in the dgp object.")
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( (method == 'mean_var')&(("mean" %in% names(object$loo)) ||
                                     ("method" %in% names(object$loo) && object$loo$method == "mean_var")) & (M == object$loo$M) ){
          if ( isTRUE(verb) ) message(" LOO re-evaluation not needed.")
          if ( isTRUE(verb) ) message("Exporting dgp object without re-evaluation ...", appendLF = FALSE)
          if ( isTRUE(verb) ) Sys.sleep(0.5)
          if ( isTRUE(verb) ) message(" done")
          return(object)
        } else if ( (method == 'sampling') & (("median" %in% names(object$loo)) ||
                                              ("method" %in% names(object$loo) && object$loo$method == "sampling"))  & (M == object$loo$M) ) {
          if ( sample_size == object$loo$sample_size ) {
            if ( isTRUE(verb) ) message(" LOO re-evaluation not needed.")
            if ( isTRUE(verb) ) message("Exporting dgp object without re-evaluation ...", appendLF = FALSE)
            if ( isTRUE(verb) ) Sys.sleep(0.5)
            if ( isTRUE(verb) ) message(" done")
            return(object)
          } else {
            if ( isTRUE(verb) ) message(" LOO re-evaluation needed.")
            if ( isTRUE(verb) ) message("Start re-evaluation: ")
          }
        } else {
          if ( isTRUE(verb) ) message(" LOO re-evaluation needed.")
          if ( isTRUE(verb) ) message("Start re-evaluation: ")
        }
      }
    }

    if ( isTRUE(verb) ) message("Initializing the LOO ...", appendLF = FALSE)
    x_train <- object$constructor_obj$X
    y_train <- object$data$Y
    rep <- object$constructor_obj$indices
    if ( !is.null(rep) ){
      rep <- rep + 1
      x_train <- x_train[rep,,drop=FALSE]
    }
    dat <- list('x_train' = x_train,'y_train' = y_train)
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Calculating the LOO ...", appendLF = FALSE)
    if ( identical(cores,as.integer(1)) ){
      res <- object$emulator_obj$loo(X = reticulate::np_array(x_train), method = method, sample_size = sample_size, m = M)
    } else {
      res <- object$emulator_obj$ploo(X = reticulate::np_array(x_train), method = method, sample_size = sample_size, m = M, core_num = cores)
    }
    if ( isTRUE(verb) ) message(" done")
    if ( isTRUE(verb) ) message("Saving results to the slot 'loo' in the dgp object ...", appendLF = FALSE)
    if ( method == 'sampling' ){
      if ( is.categorical ) {
        encoder <- object$constructor_obj$all_layer[[L]][[1]]$class_encoder
        prob_samp <- pkg.env$np$transpose(pkg.env$np$asarray(res), c(2L,1L,0L))
        prob_samp_mean <- pkg.env$np$mean(prob_samp, 0L)
        if (ncol(prob_samp_mean)==1){
          encode_label <- as.integer(prob_samp_mean>=0.5)
          dat[["probability"]] <- cbind(1-prob_samp_mean, prob_samp_mean)
        } else {
          encode_label <- as.integer(pkg.env$np$argmax(prob_samp_mean, 1L))
          dat[["probability"]] <- prob_samp_mean
        }
        dat[["label"]] <- encoder$inverse_transform(encode_label)
        colnames(dat[["probability"]]) <- as.character(encoder$classes_)
        y_train_encode <- encoder$transform(as.vector(dat$y_train))
        idx <- cbind(1:length(y_train_encode), y_train_encode + 1)
        dat[["log_loss"]] <- -mean(log(dat[["probability"]][idx]))
        dat[["accuracy"]] <- mean(dat[["label"]] ==  as.vector(dat$y_train))
        dat[['method']] <- method
      } else {
        res_np <- pkg.env$np$array(res)
        quant <- pkg.env$np$transpose(pkg.env$np$quantile(res_np, c(0.025, 0.5, 0.975), axis=2L),c(0L,2L,1L))
        std <- pkg.env$np$std(res_np, axis=2L)
        dat[["median"]] <- as.matrix(quant[2,,])
        dat[["std"]] <- t(std)
        dat[["lower"]] <- as.matrix(quant[1,,])
        dat[["upper"]] <- as.matrix(quant[3,,])
        dat[["rmse"]] <- unname(sqrt(colMeans((dat$median-dat$y_train)^2)))
        dat[["nrmse"]] <- dat$rmse/(pkg.env$np$amax(dat$y_train, axis=0L)-pkg.env$np$amin(dat$y_train, axis=0L))
      }
    } else if ( method == 'mean_var' ) {
      if ( is.categorical ) {
        encoder <- object$constructor_obj$all_layer[[L]][[1]]$class_encoder
        if (ncol(res[[1]])==1){
          encode_label <- as.integer(res[[1]]>=0.5)
          dat[["probability"]] <- cbind(1-res[[1]], res[[1]])
        } else {
          encode_label <- as.integer(pkg.env$np$argmax(res[[1]], 1L))
          dat[["probability"]] <- res[[1]]
        }
        dat[["label"]] <- encoder$inverse_transform(encode_label)
        colnames(dat[["probability"]]) <- as.character(encoder$classes_)
        y_train_encode <- encoder$transform(as.vector(dat$y_train))
        idx <- cbind(1:length(y_train_encode), y_train_encode + 1)
        dat[["log_loss"]] <- -mean(log(dat[["probability"]][idx]))
        dat[["accuracy"]] <- mean(dat[["label"]] ==  as.vector(dat$y_train))
        dat[['method']] <- method
      } else {
      dat[["mean"]] <- res[[1]]
      dat[["std"]] <- sqrt(res[[2]])
      dat[["lower"]] <- dat$mean-2*dat$std
      dat[["upper"]] <- dat$mean+2*dat$std
      dat[["rmse"]] <- unname(sqrt(colMeans((dat$mean-dat$y_train)^2)))
      dat[["nrmse"]] <- dat$rmse/(pkg.env$np$amax(dat$y_train, axis=0L)-pkg.env$np$amin(dat$y_train, axis=0L))
      }
    }
    dat[["M"]] <- M
    if (method == "sampling"){
      dat[["sample_size"]] <- sample_size
    }
    object$loo <- dat
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    return(object)
    #For OOS
  } else if (!is.null(x_test) & !is.null(y_test)) {
    rownames(x_test) <- NULL
    rownames(y_test) <- NULL
    #x_test <- unname(x_test)
    #y_test <- unname(y_test)
    if ( !is.matrix(x_test)&!is.vector(x_test) ) stop("'x_test' must be a vector or a matrix.", call. = FALSE)
    if ( !is.matrix(y_test)&!is.vector(y_test) ) stop("'y_test' must be a vector or a matrix.", call. = FALSE)
    if ( is.vector(x_test) ) {
      if ( ncol(object$data$X)!=1 ){
        x_test <- matrix(x_test, nrow = 1)
      } else {
        x_test <- as.matrix(x_test)
      }
    }
    if ( is.vector(y_test) ) {
      if ( ncol(object$data$Y)!=1 ){
        y_test <- matrix(y_test, nrow = 1)
      } else {
        y_test <- as.matrix(y_test)
      }
    }
    if ( nrow(x_test)!=nrow(y_test) ) stop("'x_test' and 'y_test' have different number of data points.", call. = FALSE)
    if ( ncol(x_test) != ncol(object$data$X) ) stop("'x_test' must have the same number of dimensions as the training input.", call. = FALSE)
    #check existing OOS
    if ( isFALSE(force) ){
      if ( "oos" %in% names(object) ){
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( isTRUE(verb) ) message(" OOS results found in the dgp object.")
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( identical(object$oos$x_test, x_test) & identical(object$oos$y_test, y_test) & (method == 'mean_var')&(("mean" %in% names(object$oos)) ||
                                                                                                                   ("method" %in% names(object$oos) && object$oos$method == "mean_var")) & (M == object$oos$M) ){
          if ( isTRUE(verb) ) message(" OOS re-evaluation not needed.")
          if ( isTRUE(verb) ) message("Exporting dgp object without re-evaluation ...", appendLF = FALSE)
          if ( isTRUE(verb) ) Sys.sleep(0.5)
          if ( isTRUE(verb) ) message(" done")
          return(object)
        } else if ( identical(object$oos$x_test, x_test) & identical(object$oos$y_test, y_test) & (method == 'sampling')&(("median" %in% names(object$oos))||
                                                                                                                          ("method" %in% names(object$oos) && object$oos$method == "sampling")) & (M == object$oos$M) ){
          if ( sample_size == object$oos$sample_size ){
            if ( isTRUE(verb) ) message(" OOS re-evaluation not needed.")
            if ( isTRUE(verb) ) message("Exporting dgp object without re-evaluation ...", appendLF = FALSE)
            if ( isTRUE(verb) ) Sys.sleep(0.5)
            if ( isTRUE(verb) ) message(" done")
            return(object)
          } else {
            if ( isTRUE(verb) ) message(" OOS re-evaluation needed.")
            if ( isTRUE(verb) ) message("Start re-evaluation: ")
          }
        } else {
          if ( isTRUE(verb) ) message(" OOS re-evaluation needed.")
          if ( isTRUE(verb) ) message("Start re-evaluation: ")
        }
      }
    }

    if ( isTRUE(verb) ) message("Initializing the OOS ...", appendLF = FALSE)
    dat <- list('x_test' = x_test,'y_test' = y_test)
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Calculating the OOS ...", appendLF = FALSE)
    rep_x <- pkg.env$np$unique(x_test, return_inverse=TRUE, axis=0L)
    x_test_unique <- rep_x[[1]]
    rep <- rep_x[[2]] + 1

    if ( identical(cores,as.integer(1)) ){
        res <- object$emulator_obj$predict(x = x_test_unique, method = method, sample_size = sample_size, m = M)
    } else {
        res <- object$emulator_obj$ppredict(x = x_test_unique, method = method, sample_size = sample_size, m = M, core_num = cores)
    }
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Saving results to the slot 'oos' in the dgp object ...", appendLF = FALSE)
    if ( method == 'sampling' ){
      if ( is.categorical ) {
        encoder <- object$constructor_obj$all_layer[[L]][[1]]$class_encoder
        prob_samp <- pkg.env$np$transpose(pkg.env$np$asarray(res), c(2L,1L,0L))
        prob_samp_mean <- pkg.env$np$mean(prob_samp, 0L)
        if (ncol(prob_samp_mean)==1){
          encode_label <- as.integer(prob_samp_mean>=0.5)
          dat[["probability"]] <- cbind(1-prob_samp_mean, prob_samp_mean)
        } else {
          encode_label <- as.integer(pkg.env$np$argmax(prob_samp_mean, 1L))
          dat[["probability"]] <- prob_samp_mean
        }
        dat[["probability"]] <- dat[["probability"]][rep,,drop=F]
        original_label <- encoder$inverse_transform(encode_label)
        dat[["label"]] <- original_label[rep]
        colnames(dat[["probability"]]) <- as.character(encoder$classes_)
        y_test_encode <- encoder$transform(as.vector(dat$y_test))
        idx <- cbind(1:length(y_test_encode), y_test_encode + 1)
        dat[["log_loss"]] <- -mean(log(dat[["probability"]][idx]))
        dat[["accuracy"]] <- mean(dat[["label"]] ==  as.vector(dat$y_test))
        dat[['method']] <- method
      } else {
        res_np <- pkg.env$np$array(res)
        quant <- pkg.env$np$transpose(pkg.env$np$quantile(res_np, c(0.025, 0.5, 0.975), axis=2L),c(0L,2L,1L))
        std <- pkg.env$np$std(res_np, axis=2L)
        dat[["median"]] <- as.matrix(quant[2,,])[rep,,drop=F]
        dat[["std"]] <- t(std)[rep,,drop=F]
        dat[["lower"]] <- as.matrix(quant[1,,])[rep,,drop=F]
        dat[["upper"]] <- as.matrix(quant[3,,])[rep,,drop=F]
        dat[["rmse"]] <- unname(sqrt(colMeans((dat$median-dat$y_test)^2)))
        dat[["nrmse"]] <- dat$rmse/(pkg.env$np$amax(dat$y_test, axis=0L)-pkg.env$np$amin(dat$y_test, axis=0L))
      }
    } else if ( method == 'mean_var' ) {
      if ( is.categorical ) {
        encoder <- object$constructor_obj$all_layer[[L]][[1]]$class_encoder
        if (ncol(res[[1]])==1){
          encode_label <- as.integer(res[[1]]>=0.5)
          dat[["probability"]] <- cbind(1-res[[1]], res[[1]])
        } else {
          encode_label <- as.integer(pkg.env$np$argmax(res[[1]], 1L))
          dat[["probability"]] <- res[[1]]
        }
        dat[["probability"]] <- dat[["probability"]][rep,,drop=F]
        original_label <- encoder$inverse_transform(encode_label)
        dat[["label"]] <- original_label[rep]
        colnames(dat[["probability"]]) <- as.character(encoder$classes_)
        y_test_encode <- encoder$transform(as.vector(dat$y_test))
        idx <- cbind(1:length(y_test_encode), y_test_encode + 1)
        dat[["log_loss"]] <- -mean(log(dat[["probability"]][idx]))
        dat[["accuracy"]] <- mean(dat[["label"]] == as.vector(dat$y_test))
        dat[['method']] <- method
      } else {
      dat[["mean"]] <- res[[1]][rep,,drop=F]
      dat[["std"]] <- sqrt(res[[2]][rep,,drop=F])
      dat[["lower"]] <- dat$mean-2*dat$std
      dat[["upper"]] <- dat$mean+2*dat$std
      dat[["rmse"]] <- unname(sqrt(colMeans((dat$mean-dat$y_test)^2)))
      dat[["nrmse"]] <- dat$rmse/(pkg.env$np$amax(dat$y_test, axis=0L)-pkg.env$np$amin(dat$y_test, axis=0L))
      }
    }
    dat[["M"]] <- M
    if (method == "sampling"){
      dat[["sample_size"]] <- sample_size
    }
    object$oos <- dat
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    return(object)
    #For other cases
  } else {
    stop("Either 'x_test' or 'y_test' is not given.", call. = FALSE)
  }
}


#' @rdname validate
#' @method validate lgp
#' @export
validate.lgp <- function(object, x_test = NULL, y_test = NULL, method = "mean_var", sample_size = 50, verb = TRUE, M = 50, force = FALSE, cores = 1, ...) {
  if ( is.null(pkg.env$dgpsi) ) {
    init_py(verb = F)
    if (pkg.env$restart) return(invisible(NULL))
  }
  #check class
  if ( !inherits(object,"lgp") ) stop("'object' must be an instance of the 'lgp' class.", call. = FALSE)
    if ( !("emulator_obj" %in% names(object)) ){
      stop("'object' is not activated for predictions. Please set `activate = TRUE` in `lgp()` to activate the emulator.", call. = FALSE)
    }
  if ( reticulate::py_is_null_xptr(object$emulator_obj) ) stop("The Python session originally associated with 'object' is no longer active. Please rebuild the emulator or, if it was saved using dgpsi::write(), load it into the R session with dgpsi::read().", call. = FALSE)
  #check core number
  if( !is.null(cores) ) {
    cores <- as.integer(cores)
    if ( cores < 1 ) stop("'cores' must be >= 1.", call. = FALSE)
  }
  M <- as.integer(M)

  if ( method!='mean_var' & method!='sampling' ) stop("'method' can only be either 'mean_var' or 'sampling'.", call. = FALSE)

  sample_size <- as.integer(sample_size)
  #For OOS
  if (!is.null(x_test) & !is.null(y_test)) {
    #check testing input
      if ( !is.matrix(x_test)&!is.vector(x_test) ) stop("'x_test' must be a vector or a matrix.", call. = FALSE)
      rownames(x_test) <- NULL
      #x_test <- unname(x_test)
      global_dim <- unique(subset(object$specs$struc, object$specs$struc[["From_Emulator"]] == "Global")$From_Output)
      if ( is.vector(x_test) ) {
        if ( global_dim!=1 ){
          x_test <- matrix(x_test, nrow = 1)
          if ( ncol(x_test)<global_dim ) stop(sprintf("'x_test' has missing dimensions. Expected %d, found %d in 'x'.",
                                                 global_dim, ncol(x_test)), call. = FALSE)
        } else {
          x_test <- as.matrix(x_test)
        }
      }
      nrow_x <- nrow(x_test)
      num_layers <- length(object$constructor)
      x_list <- vector("list", num_layers)
      x_list[[1]] <- x_test
      for (l in 2:num_layers) {
        layer_metadata <- subset(object$specs$metadata, object$specs$metadata[["Layer"]] == l)
        layer_metadata <- layer_metadata[order(layer_metadata$Pos_in_Layer), ]
        layer_matrices <- vector("list", nrow(layer_metadata))
        for (k in seq_len(nrow(layer_metadata))) {
          emulator_id <- layer_metadata$Emulator[k]
          input_connections <- subset(object$specs$struc, object$specs$struc[["To_Emulator"]] == emulator_id)
          global_outputs <- input_connections$From_Output[input_connections$From_Emulator == "Global"]
          if ( length(global_outputs)>0 ) {
            layer_matrices[[k]] <- x_test[,global_outputs,drop=F]
          }
        }
        x_list[[l]] <- layer_matrices
      }
      x_test <- x_list


    #check testing output
      if ( !is.list(y_test) ) {
        if ( !is.matrix(y_test)&!is.vector(y_test) ) {
          stop("'y_test' must be a vector or a matrix.", call. = FALSE)
        } else {
          final_layer <- max(object$specs$metadata$Layer)
          # Filter metadata to get emulators in the final layer
          final_layer_emulators <- subset(object$specs$metadata, object$specs$metadata[["Layer"]] == final_layer)
          emu_num <- nrow(final_layer_emulators)
          if ( emu_num!=1 ) stop(sprintf("The linked system contains %d emulators in its final layer. 'y_test' must be a list of %d vectors or matrices.",
                                         emu_num, emu_num), call. = FALSE)
          rownames(y_test) <- NULL
          #y_test <- unname(y_test)
          if ( is.vector(y_test) ) {
            output_dim <- final_layer_emulators$Total_Output_Dims
            if (output_dim==1){
              y_test <- as.matrix(y_test)
            } else {
              y_test <- matrix(y_test, nrow = 1)
            }
          }
        }
        nrow_y <- nrow(y_test)
        if ( nrow_y!=nrow_x ) stop("The number of data points are inconsistent between 'x_test' and 'y_test'.", call. = FALSE)
      } else {
        final_layer <- max(object$specs$metadata$Layer)
        # Filter metadata to get emulators in the final layer
        final_layer_emulators <- subset(object$specs$metadata, object$specs$metadata[["Layer"]] == final_layer)
        emu_num <- nrow(final_layer_emulators)
        if ( emu_num!=length(y_test) ) stop(sprintf("The linked system's final layer contains %d emulators. 'y_test' should contain %d vectors or matrices, but found %d.",
                                                    emu_num, emu_num, length(y_test)), call. = FALSE)
        for ( l in 1:length(y_test) ){
          if ( !is.matrix(y_test[[l]])&!is.vector(y_test[[l]]) ) {
            stop(sprintf("The element %i of 'y_test' must be a vector or a matrix.", l), call. = FALSE)
          } else {
            rownames(y_test[[l]]) <- NULL
            #y_test[[l]] <- unname(y_test[[l]])
            if ( is.vector(y_test[[l]]) ) {
              emu_cont <- subset(final_layer_emulators, final_layer_emulators[["Pos_in_Layer"]] == l)
              emu_cont_output_dim <- emu_cont$Total_Output_Dims
              if (emu_cont_output_dim==1){
                y_test[[l]] <- as.matrix(y_test[[l]])
              } else {
                y_test[[l]] <- matrix(y_test[[l]], nrow = 1)
              }
            }
            nrow_y <- nrow(y_test[[l]])
            if ( nrow_y!=nrow_x ) stop(sprintf("The number of data points is inconsistent between 'x_test' and the element %i of 'y_test'.", l), call. = FALSE)
          }
        }
      }

    #check existing OOS
    if ( isFALSE(force) ){
      if ( "oos" %in% names(object) ){
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( isTRUE(verb) ) message(" OOS results found in the lgp object.")
        if ( isTRUE(verb) ) message("Checking ...", appendLF = FALSE)
        if ( isTRUE(verb) ) Sys.sleep(0.5)
        if ( identical(object$oos$x_test, x_test[[1]]) & identical(object$oos$y_test, y_test) & (method == 'mean_var')&("mean" %in% names(object$oos)) & (M == object$oos$M) ){
          if ( isTRUE(verb) ) message(" OOS re-evaluation not needed.")
          if ( isTRUE(verb) ) message("Exporting lgp object without re-evaluation ...", appendLF = FALSE)
          if ( isTRUE(verb) ) Sys.sleep(0.5)
          if ( isTRUE(verb) ) message(" done")
          return(object)
        } else if ( identical(object$oos$x_test, x_test[[1]]) & identical(object$oos$y_test, y_test) & (method == 'sampling')&("median" %in% names(object$oos)) & (M == object$oos$M) ){
          if ( sample_size == object$oos$sample_size ) {
            if ( isTRUE(verb) ) message(" OOS re-evaluation not needed.")
            if ( isTRUE(verb) ) message("Exporting lgp object without re-evaluation ...", appendLF = FALSE)
            if ( isTRUE(verb) ) Sys.sleep(0.5)
            if ( isTRUE(verb) ) message(" done")
            return(object)
          } else {
            if ( isTRUE(verb) ) message(" OOS re-evaluation needed.")
            if ( isTRUE(verb) ) message("Start re-evaluation: ")
          }
        } else {
          if ( isTRUE(verb) ) message(" OOS re-evaluation needed.")
          if ( isTRUE(verb) ) message("Start re-evaluation: ")
        }
      }
    }

    if ( isTRUE(verb) ) message("Initializing the OOS ...", appendLF = FALSE)
      dat <- list('x_test' = x_test[[1]],'y_test' = y_test)
    if ( !is.list(y_test) ) y_test <- list(y_test)
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Calculating the OOS ...", appendLF = FALSE)
    if ( identical(cores,as.integer(1)) ){
      res <- object$emulator_obj$predict(x = x_test, method = method, sample_size = sample_size, m = M)
    } else {
      res <- object$emulator_obj$ppredict(x = x_test, method = method, sample_size = sample_size, m = M, core_num = cores)
    }
    if ( isTRUE(verb) ) message(" done")

    if ( isTRUE(verb) ) message("Saving results to the slot 'oos' in the lgp object ...", appendLF = FALSE)
    if ( method == 'sampling' ){
      median_lst <- list()
      std_lst <- list()
      lower_lst <- list()
      upper_lst <- list()
      rmse_lst <- list()
      nrmse_lst <- list()
      for ( l in 1:length(res) ) {
        quant <- pkg.env$np$transpose(pkg.env$np$quantile(res[[l]], c(0.025, 0.5, 0.975), axis=2L),c(0L,2L,1L))
        std <- pkg.env$np$std(res[[l]], axis=2L)
        median_lst[[l]] <- as.matrix(quant[2,,])
        std_lst[[l]] <- t(std)
        lower_lst[[l]] <- as.matrix(quant[1,,])
        upper_lst[[l]] <- as.matrix(quant[3,,])
        rmse_lst[[l]] <- unname(sqrt(colMeans((median_lst[[l]]-y_test[[l]])^2)))
        nrmse_lst[[l]] <- rmse_lst[[l]]/(pkg.env$np$amax(y_test[[l]], axis=0L)-pkg.env$np$amin(y_test[[l]], axis=0L))
      }
      dat[["median"]] <- median_lst
      dat[["std"]] <- std_lst
      dat[["lower"]] <- lower_lst
      dat[["upper"]] <- upper_lst
      dat[["rmse"]] <- rmse_lst
      dat[["nrmse"]] <- nrmse_lst
    } else if ( method == 'mean_var' ) {
      mean_lst <- list()
      std_lst <- list()
      lower_lst <- list()
      upper_lst <- list()
      rmse_lst <- list()
      nrmse_lst <- list()
      for ( l in 1:length(res[[1]]) ) {
        mean_lst[[l]] <- res[[1]][[l]]
        std_lst[[l]] <- sqrt(res[[2]][[l]])
        lower_lst[[l]] <- mean_lst[[l]]-2*std_lst[[l]]
        upper_lst[[l]] <- mean_lst[[l]]+2*std_lst[[l]]
        rmse_lst[[l]] <- unname(sqrt(colMeans((mean_lst[[l]]-y_test[[l]])^2)))
        nrmse_lst[[l]] <- rmse_lst[[l]]/(pkg.env$np$amax(y_test[[l]], axis=0L)-pkg.env$np$amin(y_test[[l]], axis=0L))
      }
      dat[["mean"]] <- mean_lst
      dat[["std"]] <- std_lst
      dat[["lower"]] <- lower_lst
      dat[["upper"]] <- upper_lst
      dat[["rmse"]] <- rmse_lst
      dat[["nrmse"]] <- nrmse_lst
    }
    dat[["M"]] <- M
    if (method == "sampling"){
      dat[["sample_size"]] <- sample_size
    }
    object$oos <- dat
    if ( isTRUE(verb) ) Sys.sleep(0.5)
    if ( isTRUE(verb) ) message(" done")

    return(object)
    #For other cases
  } else {
    stop("Both 'x_test' and 'y_test' must be provided for validation of a linked (D)GP emulator.", call. = FALSE)
  }
}
