% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rast_by_polys.R
\name{rast.by.polys}
\alias{rast.by.polys}
\title{Summarise raster values by polygons}
\usage{
rast.by.polys(
  x,
  polygons,
  id_col = NULL,
  fun = function(v, ...) mean(v, na.rm = TRUE),
  na.rm = TRUE
)
}
\arguments{
\item{x}{A \code{SpatRaster} object containing one or more layers.}

\item{polygons}{A \code{SpatVector} with polygon geometries used to
summarise raster values.}

\item{id_col}{Optional character string giving the name of a column in
\code{polygons} to be used as an identifier (for example, \code{"PA_ID"}).
If \code{NULL} (default), all non-geometry attributes from
\code{polygons} are joined to the summary table.}

\item{fun}{A function applied to the vector of raster values extracted
for each polygon. The function must return a named vector. It should
accept \code{...} so that arguments such as \code{na.rm = TRUE} can be
passed through. The default is
\code{function(v, ...) mean(v, na.rm = TRUE)}.}

\item{na.rm}{Logical; if \code{TRUE}, missing values are removed before
applying \code{fun}. Passed to \code{fun} via \code{...}.}
}
\value{
A \code{data.frame} with one row per polygon. If \code{id_col} is not
\code{NULL}, the first column is the specified identifier; otherwise,
all attribute columns from \code{polygons} are included. Additional
columns contain the summary statistics returned by \code{fun} for each
raster layer.
}
\description{
Extracts values from a \code{SpatRaster} for each polygon in a
\code{SpatVector} and returns a data frame that combines polygon
identifiers with user-defined summary statistics of the raster values.
}
\details{
This function is a convenience wrapper around \code{terra::extract()},
combining extraction, summarisation and binding of polygon attributes
into a single step. It supports multilayer rasters; in that case the
summary statistics are returned for each layer.
}
\examples{
\donttest{
library(terra)

# Example SpatRaster and SpatVector
r <- rast(system.file("ex/elev.tif", package = "terra"))
v <- as.polygons(r > 500, dissolve = TRUE)
v$PA_ID <- paste0("PA_", seq_len(nrow(v)))

# Mean elevation per polygon
pa_stats <- rast.by.polys(
  x        = r,
  polygons = v,
  id_col   = "PA_ID"
)

# Multiple statistics per polygon
pa_stats_multi <- rast.by.polys(
  x        = r,
  polygons = v,
  id_col   = "PA_ID",
  fun      = function(v, ...) c(
    mean = mean(v, ...),
    min  = min(v, ...),
    max  = max(v, ...)
  ),
  na.rm    = TRUE
)
}

}
