#' Calculate Split Statistic
#'
#' @description
#' Computes the split statistic for a candidate partition based on the
#' overlap of factor spaces between two child nodes.
#'
#' @param Y_L Matrix of time series in the proposed left node.
#' @param Y_R Matrix of time series in the proposed right node.
#' @param r_a Number of singular vectors to compute.
#' @param r_b Number of eigenvalues to sum for the statistic.
#' @return The calculated split statistic. Returns `Inf` if a node is too small.
#' @keywords internal
#' @importFrom irlba irlba
#' @importFrom utils head tail
calculate_split_statistic <- function(Y_L, Y_R, r_a, r_b) {
  if (ncol(Y_L) < r_a || ncol(Y_R) < r_a) {
    return(Inf)
  }
  # Use standard svd for smaller matrices as irlba can be slower
  if (min(nrow(Y_L), ncol(Y_L)) < 30) {
    svd_L <- svd(Y_L, nu = r_a, nv = 0)
  } else {
    svd_L <- irlba(Y_L, nv = 0, nu = r_a, work = 10 + r_a)
  }
  U_L <- svd_L$u

  if (min(nrow(Y_R), ncol(Y_R)) < 30) {
    svd_R <- svd(Y_R, nu = r_a, nv = 0)
  } else {
    svd_R <- irlba(Y_R, nv = 0, nu = r_a, work = 10 + r_a)
  }
  U_R <- svd_R$u
  lambdas <- 1/sqrt(2)*svd(cbind(U_L, U_R))$d
  stat <- sum(head(lambdas, r_b))
  return(stat)
}


#' Find the Best Split for a Node
#'
#' Internal function to iterate through all possible split points in all
#' covariates to find the split that minimizes the split statistic.
#'
#' @param X_node Covariate matrix for the current node.
#' @param Y_node Time series matrix for the current node.
#' @param indices Original indices of the observations in the current node.
#' @param r_a Parameter passed to `calculate_split_statistic`.
#' @param r_b Parameter passed to `calculate_split_statistic`.
#' @param control The parameters used in split.
#' @return A list containing the best split found, including the statistic,
#'   split variable, split value, and indices for the left and right children.
#' @keywords internal
#' @importFrom utils head tail
#' @importFrom foreach foreach %dopar%
find_best_split <- function(X_node, Y_node, indices, r_a, r_b, control) {
  p_features <- ncol(X_node)
  n_node <- ncol(Y_node)
  j <- NULL
  # Internal function containing core logic to avoid code duplication
  find_best_for_feature <- function(j) {
    x_feature <- X_node[, j]
    unique_vals <- sort(unique(x_feature))

    if (length(unique_vals) < 2) return(NULL)
    feature_best <- list(stat = Inf) # Initialize for minimization
    # Define split candidates and subsample if needed
    split_candidates <- (head(unique_vals, -1) + tail(unique_vals, -1)) / 2
    if (is.character(control$sep) && control$sep == "auto" && n_node > 800) {
      split_candidates <- split_candidates[seq(1, length(split_candidates), by = ceiling(n_node/200))]
    } else if (is.numeric(control$sep)) {
      split_candidates <- split_candidates[seq(1, length(split_candidates), by = control$sep)]
    }
    current_minbucket <- max(control$minbucket, ceiling(n_node * 0.1))
    for (c_val in split_candidates) {
      idx_L <- which(x_feature < c_val)
      idx_R <- which(x_feature >= c_val)
      if (length(idx_L) < current_minbucket || length(idx_R) < current_minbucket) next
      current_stat <- calculate_split_statistic(
        Y_node[, idx_L, drop = FALSE], Y_node[, idx_R, drop = FALSE],
        r_a, r_b
      )

      # Find minimum statistic
      if (current_stat < feature_best$stat) {
        feature_best <- list(
          stat = current_stat,
          split_val = c_val,
          left_indices = indices[idx_L],
          right_indices = indices[idx_R],
          var_idx = j
        )
      }
    }
    return(if(is.finite(feature_best$stat)) feature_best else NULL)
  }
  # --- Execution Logic (Parallel over features for threshold method) ---
  if (control$parallel) {
    results <- foreach::foreach(
      j = 1:p_features, .export = "calculate_split_statistic", .packages = "irlba"
    ) %dopar% {
      find_best_for_feature(j)
    }
  } else {
    results <- lapply(1:p_features, find_best_for_feature)
  }
  # --- Aggregate Results ---
  valid_results <- Filter(Negate(is.null), results)
  if (length(valid_results) == 0) {
    return(list(stat = Inf)) # Default if no valid split found
  }

  stats <- sapply(valid_results, `[[`, "stat")
  best_idx <- which.min(stats)

  return(valid_results[[best_idx]])
}

