% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fungibleExtrema.R
\name{fungibleExtrema}
\alias{fungibleExtrema}
\title{Locate Extrema of Fungible Regression Weights}
\usage{
fungibleExtrema(
  R.X,
  rxy,
  r.yhata.yhatb,
  Nstarts = 100,
  MaxMin = "Max",
  Seed = NULL,
  maxGrad = 1e-05,
  PrintLevel = 1
)
}
\arguments{
\item{R.X}{p x p Predictor variable correlation matrix.}

\item{rxy}{p x 1 Vector of predictor-criterion correlations.}

\item{r.yhata.yhatb}{Correlation between least squares (yhatb) and
alternate-weight (yhata) composites.}

\item{Nstarts}{Maximum number of (max) minimizations from random starting
configurations.}

\item{MaxMin}{Character: "Max" = maximize cos(a,b); "Min" = minimize
cos(a,b).}

\item{Seed}{Starting seed for the random number generator. If Seed = NULL 
then the program will sample a random integer in the (0, 100,000) interval.
Default (Seed = NULL).}

\item{maxGrad}{The optimization routine will end when the maximimum of
the (absolute value of the ) function gradient falls below the value specified in 
maxGrad. Default (maxGrad = 1E-05).}

\item{PrintLevel}{(integer). If PrintLevel = 1 then the program will print 
additional output during function convergence. Default (PrintLevel = 1).}
}
\value{
\item{cos.ab}{cosine between OLS and alternate weights.}
\item{a}{extrema of fungible weights.} \item{k}{k weights.} \item{z}{z
weights: a normalized random vector.} \item{b}{OLS weights.} \item{u}{p x 1
vector of u weights.} \item{r.yhata.yhatb}{Correlation between yhata and
yhatb.} \item{r.y.yhatb}{Correlation between y and yhatb.}
\item{gradient}{Gradient of converged solution.}
}
\description{
Locate extrema of fungible regression weights.
}
\examples{
\dontrun{  
## Example 
## This is Koopman's Table 2 Example


R.X <- matrix(c(1.00,  .69,  .49,  .39,
                .69, 1.00,  .38,  .19,
                .49,  .38, 1.00,  .27,
                .39,  .19,  .27, 1.00),4,4)


b <- c(.39, .22, .02, .43)
rxy <- R.X \%*\% b

OLSRSQ <- t(b) \%*\% R.X \%*\% b

theta <- .02
r.yhata.yhatb <- sqrt( 1 - (theta)/OLSRSQ)

Converged = FALSE
SEED = 1234
MaxTries = 100 
iter = 1

while( iter <= MaxTries){
   SEED <- SEED + 1
  
   cat("\nCurrent Seed = ", SEED, "\n")
   output <- fungibleExtrema(R.X, rxy, 
                             r.yhata.yhatb, 
                             Nstarts = 5,
                             MaxMin = "Min", 
                             Seed = SEED,
                             maxGrad = 1E-05,
                             PrintLevel = 1)
  
   Converged <- output$converged
   if(Converged) break
   iter = iter + 1
}  

print( output )

## Scale to replicate Koopman
a <- output$a
a.old <- a
aRa <- t(a) \%*\% R.X \%*\% a

## Scale a such that a' R a = .68659
## vc = variance of composite
vc <- aRa
## sf = scale factor
sf <- .68659/vc
a <- as.numeric(sqrt(sf)) * a
cat("\nKoopman Scaling\n")
print(round(a,2))
}
}
\references{
Koopman, R. F.  (1988).  On the sensitivity of a composite to
its weights.  \emph{Psychometrika, 53(4)}, 547--552.

Waller, N. & Jones, J. (2009). Locating the extrema of fungible regression
weights in multiple regression. \emph{Psychometrika, 74}, 589--602.
}
\author{
Niels Waller and Jeff Jones
}
\keyword{fungible}
