#' Function to calculate the number of necessary individuals and generations as the crossing cost for all the crossing schemes.
#' This is the wrapper function of calCost.
#'
#' @param line_comb_lis A list of combinations of parents.
#' @param gene_df1 Data frame of one set of haplotype. Values take 1 (target allele) or 0 (non-target).
#' @param gene_df2 Data frame of the other set of haplotype. Values take 1 or 0.
#' @param recom_mat Matrix of recombination rate among genes.
#' @param prob_total Probability of success.
#' @param last_cross Whether or not to conduct the last cross to a cultivar without target alleles.
#' @param last_selfing Whether or not to conduct the last selfing.
#'
#' @importFrom ape node.depth
#' @importFrom dplyr as_tibble
#'
#' @returns calcCostAll function returns a `gpyramid_all` object. This is a named list with the following components:
#' * `cost_all` (`data frame`): Data frame showing the number of necessary generations (n_generation) and
#' the number of individuals (N_total) for each crossing scheme. The crossing schemes are identified by cross_id.
#' * Other components mirror the input parameters for downstream analysis.
#'
#' @export

calcCostAll <- function(line_comb_lis, gene_df1, gene_df2, recom_mat, prob_total, last_cross = FALSE, last_selfing = FALSE){

  if(last_cross + last_selfing == 1){
    stop("The current version only supports scenarios where last_cross and last_selfing are both TRUE or both FALSE.")
  }

  cost_all <- data.frame(cross_id = numeric(0),
                         n_generation = numeric(0),
                         N_total = numeric(0),
                         n_parent = numeric(0),
                         line_comb_id = numeric(0),
                         TR_id = numeric(0))

  # Calculation for each parental set
  k <- 1
  for(i in 1:length(line_comb_lis)){

    # Generate crossing procedures
    line_comb <- line_comb_lis[[i]]
    n_line <- length(line_comb)
    TR <- allCrosses(n_line, line_comb)
    n_TR <- length(TR)

    # prepare genotypes of parents
    gene_df1_sel <- gene_df1[, line_comb]
    gene_df2_sel <- gene_df2[, line_comb]

    for(j in 1:n_TR){

      # Calculate cost
      topolo <- TR[[j]]
      cost_onecrossing <- calcCost(topolo, gene_df1_sel, gene_df2_sel, recom_mat, prob_total, last_cross = last_cross, last_selfing = last_selfing)

      # Summarize cost index
      n_generation <- max(node.depth(topolo, method = 2)) - 1
      N_total <- cost_onecrossing$n_plant_df$n_plant |> ceiling() |> sum()
      cost_all[k, ] <- c(k, n_generation, N_total, n_line, i, j)

      k <- k + 1
    }
  }

  output <- list(cost_all = as_tibble(cost_all),
                 line_comb_lis = line_comb_lis,
                 gene_df1 = gene_df1,
                 gene_df2 = gene_df2,
                 recom_mat = recom_mat,
                 prob_total = prob_total,
                 last_cross = last_cross,
                 last_selfing = last_selfing)

  class(output) <- "gpyramid_all"
  output
}

#' @method print gpyramid_all
#'
#' @importFrom utils head
#'
#' @export

print.gpyramid_all <- function(x, ...) {
  cat("# Head of the summary table\n")
  print(head(x$cost_all))
  cat("\n# To extract summary table, check the list element named as cost_all.\n")

}




