% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simKHCE.R
\name{simKHCE}
\alias{simKHCE}
\title{Simulate a kidney disease \code{hce} dataset}
\usage{
simKHCE(
  n,
  CM_A,
  CM_P = -4,
  n0 = n,
  TTE_A = 1000,
  TTE_P = TTE_A,
  fixedfy = 2,
  Emin = 20,
  Emax = 100,
  sigma = NULL,
  Sigma = 3,
  m = 10,
  theta = -0.4605,
  phi = 0,
  two_meas = c("no", "base", "postbase", "both")
)
}
\arguments{
\item{n}{sample size in the active treatment group.}

\item{CM_A}{annualized eGFR slope in the active group.}

\item{CM_P}{annualized eGFR slope in the control group.}

\item{n0}{sample size in the control treatment group.}

\item{TTE_A}{event rate per year in the active group for KFRT.}

\item{TTE_P}{event rate per year in the placebo group for KFRT.}

\item{fixedfy}{length of follow-up in years.}

\item{Emin}{lower limit of eGFR at baseline.}

\item{Emax}{upper limit of eGFR at baseline.}

\item{sigma}{within-patient standard deviation.}

\item{Sigma}{between-patient standard deviation.}

\item{m}{number of equidistant visits.}

\item{theta}{coefficient of dependence of eGFR values and the risk of KFRT.}

\item{phi}{coefficient of proportionality (between 0 and 1) of the treatment effect. The case of 0 corresponds to the uniform treatment effect.}

\item{two_meas}{determines whether to use duplicate measurements at baseline and/or at \code{fixedfy}. The default is to use a single measurement.}
}
\value{
a list containing the dataset \code{GFR} for longitudinal measurements of
eGFR and the competing KFRT events, the dataset \code{ADET} for the time-to-event
kidney outcomes (sustained declines or sustained low levels of eGFR),
and the combined \code{HCE} dataset for the kidney hierarchical composite endpoint.
}
\description{
Simulate a kidney disease \code{hce} dataset, capturing eGFR (Estimated Glomerular Filtration Rate) progression over time, along with
a competing and dependent terminal event: KFRT (Kidney Failure Replacement Therapy)
}
\details{
The default setting is \code{TTE_A = TTE_P} because, conditional on eGFR level,
the treatment effect does not influence the event rate of KFRT. In this model,
the effect of treatment on KFRT operates entirely through its impact on eGFR decline.

The parameters \code{TTE_A} and \code{theta} are chosen so that when GFR is 15, the event rate
is 1 per patient per year, and when GFR is 25, the event rate is 0.01 per patient per year. These
parameter values are obtained by solving the equation \code{rate0*exp(GFR*theta) = rate} for \code{rate0}
and \code{theta}. When the observed eGFR is above 30, the event rate is set to a very low value (0.001),
while when the observed eGFR is below 10, the event rate is set to a very high value (10000). This ensures that patients with observed low eGFR values
always experience KFRT, while those with high eGFR values do not.

By default, the standard deviation for within-patient variability, \code{sigma}, is set to \code{NULL.} When left as \code{NULL}, \code{sigma}
is calculated as \verb{sqrt(0.67*predicted eGFR)}. This approach results in time-dependent variability for measurements,
where lower predicted eGFR values lead to reduced variability.

When \code{phi = 0}, the treatment effect is fully additive - the same average treatment effect
applies to all patients, regardless of their baseline disease progression rate (\code{CM_P}).
When \code{phi = 1}, the treatment effect is fully proportional - there is no additive component
(the value of \code{CM_A} is irrelevant).
The more relativistic intermediate treatment effect (half proportional and half additive)
can be obtained by setting \verb{phi = abs(CM_A - CM_P)/(2*abs(CM_P)).}

The kidney hierarchical composite endpoint is defined in the following order:
(1) Kidney Failure Replacement Therapy (KFRT); (2) Sustained eGFR < 15;
(3) Sustained 57 percent or more decline in eGFR; (4) Sustained 50 percent or more  decline in eGFR; (5) Sustained 40 percent or more decline in eGFR;
and (6) Change in eGFR. In practice, because KFRT is frequently initiated when true eGFR is very low,
sustained eGFR < 15 events are rarely observed.
}
\examples{
# Example 1 - minimal example
set.seed(2022)
L <- simKHCE(n = 1000, CM_A = -3.25)
dat <- L$HCE
calcWO(dat)
# Example 2 - using the most important variables
set.seed(2022)
CM_A <- -4
CM_P <- - 4.5
phi <- abs(CM_A - CM_P) / (2*abs(CM_P))
L <- simKHCE(n = 1000, CM_A = CM_A, CM_P = CM_P, 
fixedfy = 4, Emin = 25, Emax = 75, phi = phi)
dat <- L$HCE
calcWO(dat)
}
\seealso{
\code{\link[=simHCE]{simHCE()}} for a general function of simulating \code{hce} datasets.
}
