/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HLprivate2.h"

#include "H5DOpublic.h"

#ifndef H5_NO_DEPRECATED_SYMBOLS

herr_t
H5DOwrite_chunk(hid_t dset_id, hid_t dxpl_id, uint32_t filters, const hsize_t *offset, size_t data_size,
                const void *buf)
{
    
    if (H5Dwrite_chunk(dset_id, dxpl_id, filters, offset, data_size, buf) < 0)
        return FAIL;
    else
        return SUCCEED;

} 

herr_t
H5DOread_chunk(hid_t dset_id, hid_t dxpl_id, const hsize_t *offset, uint32_t *filters, void *buf)
{
    
    if (H5Dread_chunk1(dset_id, dxpl_id, offset, filters, buf) < 0)
        return FAIL;
    else
        return SUCCEED;
} 

#endif 

herr_t
H5DOappend(hid_t dset_id, hid_t dxpl_id, unsigned axis, size_t extension, hid_t memtype, const void *buf)
{
    hsize_t  size[H5S_MAX_RANK];  
    hsize_t  old_size = 0;        
    int      sndims;              
    unsigned ndims;               
    hid_t    space_id     = FAIL; 
    hid_t    new_space_id = FAIL; 
    hid_t    mem_space_id = FAIL; 
    hssize_t snelmts;             
    hsize_t  nelmts;              
    hid_t    dapl = FAIL;         

    hsize_t start[H5S_MAX_RANK];  
    hsize_t count[H5S_MAX_RANK];  
    hsize_t stride[H5S_MAX_RANK]; 
    hsize_t block[H5S_MAX_RANK];  

    hsize_t        *boundary = NULL;  
    H5D_append_cb_t append_cb;        
    void           *udata;            
    bool            hit = false;      
    hsize_t         k;                
    unsigned        u;                
    herr_t          ret_value = FAIL; 

    
    if (H5I_DATASET != H5Iget_type(dset_id))
        goto done;

    
    if (H5P_DEFAULT != dxpl_id)
        if (true != H5Pisa_class(dxpl_id, H5P_DATASET_XFER))
            goto done;

    
    if (FAIL == (space_id = H5Dget_space(dset_id)))
        goto done;

    
    if ((sndims = H5Sget_simple_extent_ndims(space_id)) < 0)
        goto done;
    ndims = (unsigned)sndims;

    
    if (axis >= ndims)
        goto done;

    
    if (H5Sget_simple_extent_dims(space_id, size, NULL) < 0)
        goto done;

    
    old_size = size[axis];
    size[axis] += extension;
    if (size[axis] < old_size)
        goto done;

    
    if (H5Dset_extent(dset_id, size) < 0)
        goto done;

    
    if (FAIL == (new_space_id = H5Dget_space(dset_id)))
        goto done;

    
    for (u = 0; u < ndims; u++) {
        start[u]  = 0;
        stride[u] = 1;
        count[u]  = size[u];
        block[u]  = 1;
        if (u == axis) {
            count[u] = extension;
            start[u] = old_size;
        } 
    }     
    if (FAIL == H5Sselect_hyperslab(new_space_id, H5S_SELECT_SET, start, stride, count, block))
        goto done;

    
    if ((snelmts = H5Sget_select_npoints(new_space_id)) < 0)
        goto done;
    nelmts = (hsize_t)snelmts;

    
    if (FAIL == (mem_space_id = H5Screate_simple(1, &nelmts, NULL)))
        goto done;

    
    if (H5Dwrite(dset_id, memtype, mem_space_id, new_space_id, dxpl_id, buf) < 0)
        goto done;

    
    if ((dapl = H5Dget_access_plist(dset_id)) < 0)
        goto done;

    
    boundary = (hsize_t *)malloc(ndims * sizeof(hsize_t));

    
    if (H5Pget_append_flush(dapl, ndims, boundary, &append_cb, &udata) < 0)
        goto done;

    
    if (boundary[axis] != 0) {

        
        for (k = start[axis]; k < size[axis]; k++)
            if (!((k + 1) % boundary[axis])) {
                hit = true;
                break;
            }

        if (hit) { 
            
            if (append_cb && append_cb(dset_id, size, udata) < 0)
                goto done;

            
            if (H5Dflush(dset_id) < 0)
                goto done;
        } 
    }     

    
    ret_value = SUCCEED;

done:
    
    if (space_id != FAIL && H5Sclose(space_id) < 0)
        ret_value = FAIL;

    
    if (new_space_id != FAIL && H5Sclose(new_space_id) < 0)
        ret_value = FAIL;

    
    if (mem_space_id != FAIL && H5Sclose(mem_space_id) < 0)
        ret_value = FAIL;

    
    if (dapl != FAIL && H5Pclose(dapl) < 0)
        ret_value = FAIL;

    if (boundary)
        free(boundary);

    return ret_value;
} 
