/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5B2module.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5B2pkg.h"     
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

#define H5B2_HDR_VERSION  0 
#define H5B2_INT_VERSION  0 
#define H5B2_LEAF_VERSION 0 

static herr_t H5B2__cache_hdr_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5B2__cache_hdr_verify_chksum(const void *image_ptr, size_t len, void *udata);
static void  *H5B2__cache_hdr_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5B2__cache_hdr_image_len(const void *thing, size_t *image_len);
static herr_t H5B2__cache_hdr_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5B2__cache_hdr_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5B2__cache_hdr_free_icr(void *thing);

static herr_t H5B2__cache_int_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5B2__cache_int_verify_chksum(const void *image_ptr, size_t len, void *udata);
static void  *H5B2__cache_int_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5B2__cache_int_image_len(const void *thing, size_t *image_len);
static herr_t H5B2__cache_int_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5B2__cache_int_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5B2__cache_int_free_icr(void *thing);

static herr_t H5B2__cache_leaf_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5B2__cache_leaf_verify_chksum(const void *image_ptr, size_t len, void *udata);
static void  *H5B2__cache_leaf_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5B2__cache_leaf_image_len(const void *thing, size_t *image_len);
static herr_t H5B2__cache_leaf_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5B2__cache_leaf_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5B2__cache_leaf_free_icr(void *thing);

const H5AC_class_t H5AC_BT2_HDR[1] = {{
    H5AC_BT2_HDR_ID,                       
    "v2 B-tree header",                    
    H5FD_MEM_BTREE,                        
    H5AC__CLASS_NO_FLAGS_SET,              
    H5B2__cache_hdr_get_initial_load_size, 
    NULL,                                  
    H5B2__cache_hdr_verify_chksum,         
    H5B2__cache_hdr_deserialize,           
    H5B2__cache_hdr_image_len,             
    NULL,                                  
    H5B2__cache_hdr_serialize,             
    H5B2__cache_hdr_notify,                
    H5B2__cache_hdr_free_icr,              
    NULL,                                  
}};

const H5AC_class_t H5AC_BT2_INT[1] = {{
    H5AC_BT2_INT_ID,                       
    "v2 B-tree internal node",             
    H5FD_MEM_BTREE,                        
    H5AC__CLASS_NO_FLAGS_SET,              
    H5B2__cache_int_get_initial_load_size, 
    NULL,                                  
    H5B2__cache_int_verify_chksum,         
    H5B2__cache_int_deserialize,           
    H5B2__cache_int_image_len,             
    NULL,                                  
    H5B2__cache_int_serialize,             
    H5B2__cache_int_notify,                
    H5B2__cache_int_free_icr,              
    NULL,                                  
}};

const H5AC_class_t H5AC_BT2_LEAF[1] = {{
    H5AC_BT2_LEAF_ID,                       
    "v2 B-tree leaf node",                  
    H5FD_MEM_BTREE,                         
    H5AC__CLASS_NO_FLAGS_SET,               
    H5B2__cache_leaf_get_initial_load_size, 
    NULL,                                   
    H5B2__cache_leaf_verify_chksum,         
    H5B2__cache_leaf_deserialize,           
    H5B2__cache_leaf_image_len,             
    NULL,                                   
    H5B2__cache_leaf_serialize,             
    H5B2__cache_leaf_notify,                
    H5B2__cache_leaf_free_icr,              
    NULL,                                   
}};

static herr_t
H5B2__cache_hdr_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5B2_hdr_cache_ud_t *udata = (H5B2_hdr_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->f);
    assert(image_len);

    
    *image_len = H5B2_HEADER_SIZE_FILE(udata->f);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5B2__cache_hdr_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;                

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5B2__cache_hdr_deserialize(const void *_image, size_t H5_ATTR_UNUSED len, void *_udata,
                            bool H5_ATTR_UNUSED *dirty)
{
    H5B2_hdr_t          *hdr   = NULL; 
    H5B2_hdr_cache_ud_t *udata = (H5B2_hdr_cache_ud_t *)_udata;
    H5B2_create_t        cparam;                              
    H5B2_subid_t         id;                                  
    uint16_t             depth;                               
    uint32_t             stored_chksum;                       
    const uint8_t       *image     = (const uint8_t *)_image; 
    H5B2_hdr_t          *ret_value = NULL;                    

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    if (NULL == (hdr = H5B2__hdr_alloc(udata->f)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "allocation failed for B-tree header");

    
    if (memcmp(image, H5B2_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "wrong B-tree header signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5B2_HDR_VERSION)
        HGOTO_ERROR(H5E_BTREE, H5E_BADRANGE, NULL, "wrong B-tree header version");

    
    id = (H5B2_subid_t)*image++;
    if (id >= H5B2_NUM_BTREE_ID)
        HGOTO_ERROR(H5E_BTREE, H5E_BADTYPE, NULL, "incorrect B-tree type");

    
    UINT32DECODE(image, cparam.node_size);

    
    UINT16DECODE(image, cparam.rrec_size);

    
    UINT16DECODE(image, depth);

    
    cparam.split_percent = *image++;
    cparam.merge_percent = *image++;

    
    H5F_addr_decode(udata->f, (const uint8_t **)&image, &(hdr->root.addr));
    UINT16DECODE(image, hdr->root.node_nrec);
    H5F_DECODE_LENGTH(udata->f, image, hdr->root.all_nrec);

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == hdr->hdr_size);

    
    cparam.cls = H5B2_client_class_g[id];
    if (H5B2__hdr_init(hdr, &cparam, udata->ctx_udata, depth) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, NULL, "can't initialize B-tree header info");

    
    hdr->addr = udata->addr;

    
    assert((size_t)(image - (const uint8_t *)_image) <= len);

    
    ret_value = hdr;

done:
    if (!ret_value && hdr)
        if (H5B2__hdr_free(hdr) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTRELEASE, NULL, "can't release v2 B-tree header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_hdr_image_len(const void *_thing, size_t *image_len)
{
    const H5B2_hdr_t *hdr = (const H5B2_hdr_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(image_len);

    
    *image_len = hdr->hdr_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__cache_hdr_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5B2_hdr_t *hdr   = (H5B2_hdr_t *)_thing; 
    uint8_t    *image = (uint8_t *)_image;    
    uint32_t    metadata_chksum;              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(hdr);

    
    H5MM_memcpy(image, H5B2_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5B2_HDR_VERSION;

    
    assert(hdr->cls->id <= 255);
    *image++ = (uint8_t)hdr->cls->id;

    
    UINT32ENCODE(image, hdr->node_size);

    
    UINT16ENCODE(image, hdr->rrec_size);

    
    UINT16ENCODE(image, hdr->depth);

    
    H5_CHECK_OVERFLOW(hdr->split_percent,  unsigned,  uint8_t);
    *image++ = (uint8_t)hdr->split_percent;
    H5_CHECK_OVERFLOW(hdr->merge_percent,  unsigned,  uint8_t);
    *image++ = (uint8_t)hdr->merge_percent;

    
    H5F_addr_encode(f, &image, hdr->root.addr);
    UINT16ENCODE(image, hdr->root.node_nrec);
    H5F_ENCODE_LENGTH(f, image, hdr->root.all_nrec);

    
    metadata_chksum = H5_checksum_metadata(_image, (hdr->hdr_size - H5B2_SIZEOF_CHKSUM), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__cache_hdr_notify(H5AC_notify_action_t action, void *_thing)
{
    H5B2_hdr_t *hdr       = (H5B2_hdr_t *)_thing;
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->swmr_write) {
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
                
                break;

            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
                
                hdr->shadow_epoch++;
                break;

            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (hdr->parent) {
                    
                    assert(hdr->top_proxy);

                    
                    if (H5AC_proxy_entry_remove_child((H5AC_proxy_entry_t *)hdr->parent,
                                                      (void *)hdr->top_proxy) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNDEPEND, FAIL,
                                    "unable to destroy flush dependency between v2 B-tree and proxy");
                    hdr->parent = NULL;
                } 

                
                if (hdr->top_proxy) {
                    if (H5AC_proxy_entry_remove_child(hdr->top_proxy, hdr) < 0)
                        HGOTO_ERROR(
                            H5E_BTREE, H5E_CANTUNDEPEND, FAIL,
                            "unable to destroy flush dependency between header and v2 B-tree 'top' proxy");
                    
                } 
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else     
                assert(0 && "Unknown action?!?");
#endif    
        } 
    }     
    else
        assert(NULL == hdr->parent);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_hdr_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5B2__hdr_free((H5B2_hdr_t *)thing) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to free v2 B-tree header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_int_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5B2_internal_cache_ud_t *udata = (H5B2_internal_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(image_len);

    
    *image_len = udata->hdr->node_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5B2__cache_int_verify_chksum(const void *_image, size_t H5_ATTR_UNUSED len, void *_udata)
{
    const uint8_t            *image = (const uint8_t *)_image;            
    H5B2_internal_cache_ud_t *udata = (H5B2_internal_cache_ud_t *)_udata; 
    size_t                    chk_size;         
    uint32_t                  stored_chksum;    
    uint32_t                  computed_chksum;  
    htri_t                    ret_value = true; 

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    chk_size = H5B2_INT_PREFIX_SIZE + (udata->nrec * udata->hdr->rrec_size) +
               ((size_t)(udata->nrec + 1) * H5B2_INT_POINTER_SIZE(udata->hdr, udata->depth));

    
    if (H5F_get_checksums(image, chk_size, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5B2__cache_int_deserialize(const void *_image, size_t H5_ATTR_UNUSED len, void *_udata,
                            bool H5_ATTR_UNUSED *dirty)
{
    H5B2_internal_cache_ud_t *udata    = (H5B2_internal_cache_ud_t *)_udata; 
    H5B2_internal_t          *internal = NULL;                               
    const uint8_t            *image    = (const uint8_t *)_image; 
    uint8_t                  *native;                             
    H5B2_node_ptr_t          *int_node_ptr;                       
    uint32_t                  stored_chksum;                      
    unsigned                  u;                                  
    H5B2_internal_t          *ret_value = NULL;                   
    int                       node_nrec = 0;

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    if (NULL == (internal = H5FL_CALLOC(H5B2_internal_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5B2__hdr_incr(udata->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, NULL, "can't increment ref. count on B-tree header");

    
    internal->hdr          = udata->hdr;
    internal->parent       = udata->parent;
    internal->shadow_epoch = udata->hdr->shadow_epoch;

    
    if (memcmp(image, H5B2_INT_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "wrong B-tree internal node signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5B2_INT_VERSION)
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "wrong B-tree internal node version");

    
    if (*image++ != (uint8_t)udata->hdr->cls->id)
        HGOTO_ERROR(H5E_BTREE, H5E_BADTYPE, NULL, "incorrect B-tree type");

    
    if (NULL ==
        (internal->int_native = (uint8_t *)H5FL_FAC_MALLOC(udata->hdr->node_info[udata->depth].nat_rec_fac)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for B-tree internal native keys");

    
    if (NULL == (internal->node_ptrs =
                     (H5B2_node_ptr_t *)H5FL_FAC_MALLOC(udata->hdr->node_info[udata->depth].node_ptr_fac)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for B-tree internal node pointers");

    
    internal->nrec  = udata->nrec;
    internal->depth = udata->depth;

    
    native = internal->int_native;
    for (u = 0; u < internal->nrec; u++) {
        
        if ((udata->hdr->cls->decode)(image, native, udata->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDECODE, NULL, "unable to decode B-tree record");

        
        image += udata->hdr->rrec_size;
        native += udata->hdr->cls->nrec_size;
    } 

    
    int_node_ptr = internal->node_ptrs;
    for (u = 0; u < (unsigned)(internal->nrec + 1); u++) {
        
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &(int_node_ptr->addr));
        UINT64DECODE_VAR(image, node_nrec, udata->hdr->max_nrec_size);
        H5_CHECKED_ASSIGN(int_node_ptr->node_nrec, uint16_t, node_nrec, int);
        if (udata->depth > 1)
            UINT64DECODE_VAR(image, int_node_ptr->all_nrec,
                             udata->hdr->node_info[udata->depth - 1].cum_max_nrec_size);
        else
            int_node_ptr->all_nrec = int_node_ptr->node_nrec;

        
        int_node_ptr++;
    } 

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) <= len);

    
    ret_value = internal;

done:
    if (!ret_value && internal)
        if (H5B2__internal_free(internal) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, NULL, "unable to destroy B-tree internal node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_int_image_len(const void *_thing, size_t *image_len)
{
    const H5B2_internal_t *internal =
        (const H5B2_internal_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(internal);
    assert(internal->hdr);
    assert(image_len);

    
    *image_len = internal->hdr->node_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__cache_int_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5B2_internal_t *internal = (H5B2_internal_t *)_thing; 
    uint8_t         *image    = (uint8_t *)_image;         
    uint8_t         *native;                               
    H5B2_node_ptr_t *int_node_ptr;                         
    uint32_t         metadata_chksum;                      
    unsigned         u;                                    
    herr_t           ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(internal);
    assert(internal->hdr);

    
    H5MM_memcpy(image, H5B2_INT_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5B2_INT_VERSION;

    
    assert(internal->hdr->cls->id <= 255);
    *image++ = (uint8_t)internal->hdr->cls->id;
    assert((size_t)(image - (uint8_t *)_image) == (H5B2_INT_PREFIX_SIZE - H5B2_SIZEOF_CHKSUM));

    
    native = internal->int_native;
    for (u = 0; u < internal->nrec; u++) {
        
        if ((internal->hdr->cls->encode)(image, native, internal->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTENCODE, FAIL, "unable to encode B-tree record");

        
        image += internal->hdr->rrec_size;
        native += internal->hdr->cls->nrec_size;
    } 

    
    int_node_ptr = internal->node_ptrs;
    for (u = 0; u < (unsigned)(internal->nrec + 1); u++) {
        
        H5F_addr_encode(f, &image, int_node_ptr->addr);
        UINT64ENCODE_VAR(image, int_node_ptr->node_nrec, internal->hdr->max_nrec_size);
        if (internal->depth > 1)
            UINT64ENCODE_VAR(image, int_node_ptr->all_nrec,
                             internal->hdr->node_info[internal->depth - 1].cum_max_nrec_size);

        
        int_node_ptr++;
    } 

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) <= len);

    
    memset(image, 0, len - (size_t)(image - (uint8_t *)_image));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_int_notify(H5AC_notify_action_t action, void *_thing)
{
    H5B2_internal_t *internal  = (H5B2_internal_t *)_thing;
    herr_t           ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(internal);
    assert(internal->hdr);

    
    if (internal->hdr->swmr_write) {
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
                
                if (H5B2__create_flush_depend((H5AC_info_t *)internal->parent, (H5AC_info_t *)internal) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTDEPEND, FAIL, "unable to create flush dependency");
                break;

            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (H5B2__destroy_flush_depend((H5AC_info_t *)internal->parent, (H5AC_info_t *)internal) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");

                
                if (internal->top_proxy) {
                    if (H5AC_proxy_entry_remove_child(internal->top_proxy, internal) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNDEPEND, FAIL,
                                    "unable to destroy flush dependency between internal node and v2 B-tree "
                                    "'top' proxy");
                    internal->top_proxy = NULL;
                } 
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else     
                assert(0 && "Unknown action?!?");
#endif    
        } 
    }     
    else
        assert(NULL == internal->top_proxy);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_int_free_icr(void *_thing)
{
    H5B2_internal_t *internal  = (H5B2_internal_t *)_thing;
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(internal);

    
    if (H5B2__internal_free(internal) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to release v2 B-tree internal node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_leaf_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5B2_leaf_cache_ud_t *udata = (H5B2_leaf_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(image_len);

    
    *image_len = udata->hdr->node_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5B2__cache_leaf_verify_chksum(const void *_image, size_t H5_ATTR_UNUSED len, void *_udata)
{
    const uint8_t            *image = (const uint8_t *)_image;            
    H5B2_internal_cache_ud_t *udata = (H5B2_internal_cache_ud_t *)_udata; 
    size_t                    chk_size;         
    uint32_t                  stored_chksum;    
    uint32_t                  computed_chksum;  
    htri_t                    ret_value = true; 

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    chk_size = H5B2_LEAF_PREFIX_SIZE + (udata->nrec * udata->hdr->rrec_size);

    
    if (H5F_get_checksums(image, chk_size, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5B2__cache_leaf_deserialize(const void *_image, size_t H5_ATTR_UNUSED len, void *_udata,
                             bool H5_ATTR_UNUSED *dirty)
{
    H5B2_leaf_cache_ud_t *udata = (H5B2_leaf_cache_ud_t *)_udata;
    H5B2_leaf_t          *leaf  = NULL;                    
    const uint8_t        *image = (const uint8_t *)_image; 
    uint8_t              *native;                          
    uint32_t              stored_chksum;                   
    unsigned              u;                               
    H5B2_leaf_t          *ret_value = NULL;                

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    if (NULL == (leaf = H5FL_CALLOC(H5B2_leaf_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    if (H5B2__hdr_incr(udata->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, NULL, "can't increment ref. count on B-tree header");

    
    leaf->hdr          = udata->hdr;
    leaf->parent       = udata->parent;
    leaf->shadow_epoch = udata->hdr->shadow_epoch;

    
    if (memcmp(image, H5B2_LEAF_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "wrong B-tree leaf node signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5B2_LEAF_VERSION)
        HGOTO_ERROR(H5E_BTREE, H5E_BADRANGE, NULL, "wrong B-tree leaf node version");

    
    if (*image++ != (uint8_t)udata->hdr->cls->id)
        HGOTO_ERROR(H5E_BTREE, H5E_BADTYPE, NULL, "incorrect B-tree type");

    
    if (NULL == (leaf->leaf_native = (uint8_t *)H5FL_FAC_MALLOC(udata->hdr->node_info[0].nat_rec_fac)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for B-tree leaf native keys");

    
    leaf->nrec = udata->nrec;

    
    native = leaf->leaf_native;
    for (u = 0; u < leaf->nrec; u++) {
        
        if ((udata->hdr->cls->decode)(image, native, udata->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTENCODE, NULL, "unable to decode B-tree record");

        
        image += udata->hdr->rrec_size;
        native += udata->hdr->cls->nrec_size;
    } 

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) <= udata->hdr->node_size);

    
    assert((size_t)(image - (const uint8_t *)_image) <= len);

    
    ret_value = leaf;

done:
    if (!ret_value && leaf)
        if (H5B2__leaf_free(leaf) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, NULL, "unable to destroy B-tree leaf node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_leaf_image_len(const void *_thing, size_t *image_len)
{
    const H5B2_leaf_t *leaf = (const H5B2_leaf_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(leaf);
    assert(leaf->hdr);
    assert(image_len);

    
    *image_len = leaf->hdr->node_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__cache_leaf_serialize(const H5F_t H5_ATTR_UNUSED *f, void *_image, size_t H5_ATTR_UNUSED len,
                           void *_thing)
{
    H5B2_leaf_t *leaf  = (H5B2_leaf_t *)_thing; 
    uint8_t     *image = (uint8_t *)_image;     
    uint8_t     *native;                        
    uint32_t     metadata_chksum;               
    unsigned     u;                             
    herr_t       ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(leaf);
    assert(leaf->hdr);

    
    H5MM_memcpy(image, H5B2_LEAF_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5B2_LEAF_VERSION;

    
    assert(leaf->hdr->cls->id <= 255);
    *image++ = (uint8_t)leaf->hdr->cls->id;
    assert((size_t)(image - (uint8_t *)_image) == (H5B2_LEAF_PREFIX_SIZE - H5B2_SIZEOF_CHKSUM));

    
    native = leaf->leaf_native;
    for (u = 0; u < leaf->nrec; u++) {
        
        if ((leaf->hdr->cls->encode)(image, native, leaf->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTENCODE, FAIL, "unable to encode B-tree record");

        
        image += leaf->hdr->rrec_size;
        native += leaf->hdr->cls->nrec_size;
    } 

    
    metadata_chksum =
        H5_checksum_metadata(_image, (size_t)((const uint8_t *)image - (const uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) <= len);

    
    memset(image, 0, len - (size_t)(image - (uint8_t *)_image));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_leaf_notify(H5AC_notify_action_t action, void *_thing)
{
    H5B2_leaf_t *leaf      = (H5B2_leaf_t *)_thing;
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(leaf);
    assert(leaf->hdr);

    
    if (leaf->hdr->swmr_write) {
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
                
                if (H5B2__create_flush_depend((H5AC_info_t *)leaf->parent, (H5AC_info_t *)leaf) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTDEPEND, FAIL, "unable to create flush dependency");
                break;

            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (H5B2__destroy_flush_depend((H5AC_info_t *)leaf->parent, (H5AC_info_t *)leaf) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");

                
                if (leaf->top_proxy) {
                    if (H5AC_proxy_entry_remove_child(leaf->top_proxy, leaf) < 0)
                        HGOTO_ERROR(
                            H5E_BTREE, H5E_CANTUNDEPEND, FAIL,
                            "unable to destroy flush dependency between leaf node and v2 B-tree 'top' proxy");
                    leaf->top_proxy = NULL;
                } 
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else     
                assert(0 && "Unknown action?!?");
#endif    
        } 
    }     
    else
        assert(NULL == leaf->top_proxy);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__cache_leaf_free_icr(void *_thing)
{
    H5B2_leaf_t *leaf      = (H5B2_leaf_t *)_thing;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(leaf);

    
    if (H5B2__leaf_free(leaf) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to destroy B-tree leaf node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
