/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5MMprivate.h" 

#define H5FA_HDR_VERSION    0 
#define H5FA_DBLOCK_VERSION 0 

static herr_t H5FA__cache_hdr_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5FA__cache_hdr_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5FA__cache_hdr_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5FA__cache_hdr_image_len(const void *thing, size_t *image_len);
static herr_t H5FA__cache_hdr_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5FA__cache_hdr_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5FA__cache_hdr_free_icr(void *thing);

static herr_t H5FA__cache_dblock_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5FA__cache_dblock_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5FA__cache_dblock_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5FA__cache_dblock_image_len(const void *thing, size_t *image_len);
static herr_t H5FA__cache_dblock_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5FA__cache_dblock_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5FA__cache_dblock_free_icr(void *thing);
static herr_t H5FA__cache_dblock_fsf_size(const void *thing, hsize_t *fsf_size);

static herr_t H5FA__cache_dblk_page_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5FA__cache_dblk_page_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5FA__cache_dblk_page_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5FA__cache_dblk_page_image_len(const void *thing, size_t *image_len);
static herr_t H5FA__cache_dblk_page_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5FA__cache_dblk_page_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5FA__cache_dblk_page_free_icr(void *thing);

const H5AC_class_t H5AC_FARRAY_HDR[1] = {{
    H5AC_FARRAY_HDR_ID,                    
    "Fixed-array Header",                  
    H5FD_MEM_FARRAY_HDR,                   
    H5AC__CLASS_NO_FLAGS_SET,              
    H5FA__cache_hdr_get_initial_load_size, 
    NULL,                                  
    H5FA__cache_hdr_verify_chksum,         
    H5FA__cache_hdr_deserialize,           
    H5FA__cache_hdr_image_len,             
    NULL,                                  
    H5FA__cache_hdr_serialize,             
    H5FA__cache_hdr_notify,                
    H5FA__cache_hdr_free_icr,              
    NULL,                                  
}};

const H5AC_class_t H5AC_FARRAY_DBLOCK[1] = {{
    H5AC_FARRAY_DBLOCK_ID,                    
    "Fixed Array Data Block",                 
    H5FD_MEM_FARRAY_DBLOCK,                   
    H5AC__CLASS_NO_FLAGS_SET,                 
    H5FA__cache_dblock_get_initial_load_size, 
    NULL,                                     
    H5FA__cache_dblock_verify_chksum,         
    H5FA__cache_dblock_deserialize,           
    H5FA__cache_dblock_image_len,             
    NULL,                                     
    H5FA__cache_dblock_serialize,             
    H5FA__cache_dblock_notify,                
    H5FA__cache_dblock_free_icr,              
    H5FA__cache_dblock_fsf_size,              
}};

const H5AC_class_t H5AC_FARRAY_DBLK_PAGE[1] = {{
    H5AC_FARRAY_DBLK_PAGE_ID,                    
    "Fixed Array Data Block Page",               
    H5FD_MEM_FARRAY_DBLK_PAGE,                   
    H5AC__CLASS_NO_FLAGS_SET,                    
    H5FA__cache_dblk_page_get_initial_load_size, 
    NULL,                                        
    H5FA__cache_dblk_page_verify_chksum,         
    H5FA__cache_dblk_page_deserialize,           
    H5FA__cache_dblk_page_image_len,             
    NULL,                                        
    H5FA__cache_dblk_page_serialize,             
    H5FA__cache_dblk_page_notify,                
    H5FA__cache_dblk_page_free_icr,              
    NULL,                                        
}};

static herr_t
H5FA__cache_hdr_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5FA_hdr_cache_ud_t *udata = (H5FA_hdr_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->f);
    assert(image_len);

    
    *image_len = (size_t)H5FA_HEADER_SIZE_FILE(udata->f);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5FA__cache_hdr_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5FA__cache_hdr_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                            bool H5_ATTR_UNUSED *dirty)
{
    H5FA_cls_id_t        id;           
    H5FA_hdr_t          *hdr   = NULL; 
    H5FA_hdr_cache_ud_t *udata = (H5FA_hdr_cache_ud_t *)_udata;
    const uint8_t       *image = (const uint8_t *)_image; 
    uint32_t             stored_chksum;                   
    void                *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(udata);
    assert(udata->f);
    assert(H5_addr_defined(udata->addr));

    
    if (NULL == (hdr = H5FA__hdr_alloc(udata->f)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for fixed array shared header");

    
    hdr->addr = udata->addr;

    
    if (memcmp(image, H5FA_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, NULL, "wrong fixed array header signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5FA_HDR_VERSION)
        HGOTO_ERROR(H5E_FARRAY, H5E_VERSION, NULL, "wrong fixed array header version");

    
    id = (H5FA_cls_id_t)*image++;
    if (id >= H5FA_NUM_CLS_ID)
        HGOTO_ERROR(H5E_FARRAY, H5E_BADTYPE, NULL, "incorrect fixed array class");
    hdr->cparam.cls = H5FA_client_class_g[id];

    
    hdr->cparam.raw_elmt_size             = *image++; 
    hdr->cparam.max_dblk_page_nelmts_bits = *image++; 

    
    H5F_DECODE_LENGTH(udata->f, image, hdr->cparam.nelmts); 

    
    H5F_addr_decode(udata->f, &image, &hdr->dblk_addr); 

    
    if (H5_addr_defined(hdr->dblk_addr)) {
        H5FA_dblock_t dblock;           
        size_t        dblk_page_nelmts; 

        
        dblock.hdr                 = hdr;
        dblock.dblk_page_init_size = 0;
        dblock.npages              = 0;
        dblk_page_nelmts           = (size_t)1 << hdr->cparam.max_dblk_page_nelmts_bits;
        if (hdr->cparam.nelmts > dblk_page_nelmts) {
            dblock.npages = (size_t)(((hdr->cparam.nelmts + dblk_page_nelmts) - 1) / dblk_page_nelmts);
            dblock.dblk_page_init_size = (dblock.npages + 7) / 8;
        } 

        
        hdr->stats.dblk_size = (size_t)H5FA_DBLOCK_SIZE(&dblock);
    } 

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5FA_SIZEOF_CHKSUM));

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == len);

    
    if (H5FA__hdr_init(hdr, udata->ctx_udata) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINIT, NULL, "initialization failed for fixed array header");
    assert(hdr->size == len);

    
    ret_value = hdr;

done:
    
    if (!ret_value)
        if (hdr && H5FA__hdr_dest(hdr) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, NULL, "unable to destroy fixed array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_hdr_image_len(const void *_thing, size_t *image_len)
{
    const H5FA_hdr_t *hdr = (const H5FA_hdr_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(image_len);

    
    *image_len = hdr->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__cache_hdr_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5FA_hdr_t *hdr   = (H5FA_hdr_t *)_thing; 
    uint8_t    *image = (uint8_t *)_image;    
    uint32_t    metadata_chksum;              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(hdr);

    
    H5MM_memcpy(image, H5FA_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5FA_HDR_VERSION;

    
    assert(hdr->cparam.cls->id <= 255);
    *image++ = (uint8_t)hdr->cparam.cls->id;

    
    *image++ = hdr->cparam.raw_elmt_size; 
    *image++ =
        hdr->cparam.max_dblk_page_nelmts_bits; 

    
    H5F_ENCODE_LENGTH(f, image, hdr->stats.nelmts); 

    
    H5F_addr_encode(f, &image, hdr->dblk_addr); 

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__cache_hdr_notify(H5AC_notify_action_t action, void *_thing)
{
    H5FA_hdr_t *hdr       = (H5FA_hdr_t *)_thing; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->swmr_write) {
        
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (hdr->parent) {
                    
                    assert(hdr->top_proxy);

                    
                    if (H5AC_proxy_entry_remove_child((H5AC_proxy_entry_t *)hdr->parent,
                                                      (void *)hdr->top_proxy) < 0)
                        HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNDEPEND, FAIL,
                                    "unable to destroy flush dependency between fixed array and proxy");
                    hdr->parent = NULL;
                } 

                
                if (hdr->top_proxy) {
                    if (H5AC_proxy_entry_remove_child(hdr->top_proxy, hdr) < 0)
                        HGOTO_ERROR(
                            H5E_FARRAY, H5E_CANTUNDEPEND, FAIL,
                            "unable to destroy flush dependency between header and fixed array 'top' proxy");
                    
                } 
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else     
                assert(0 && "Unknown action?!?");
#endif    
        } 
    }     
    else
        assert(NULL == hdr->parent);

done:

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_hdr_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5FA__hdr_dest((H5FA_hdr_t *)thing) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTFREE, FAIL, "can't free fixed array header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblock_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5FA_dblock_cache_ud_t *udata = (H5FA_dblock_cache_ud_t *)_udata; 
    H5FA_dblock_t           dblock;                                   
    size_t                  dblk_page_nelmts;                         

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(image_len);

    
    
    memset(&dblock, 0, sizeof(dblock));

    
    dblock.hdr       = udata->hdr;
    dblk_page_nelmts = (size_t)1 << udata->hdr->cparam.max_dblk_page_nelmts_bits;
    if (udata->hdr->cparam.nelmts > dblk_page_nelmts) {
        dblock.npages = (size_t)(((udata->hdr->cparam.nelmts + dblk_page_nelmts) - 1) / dblk_page_nelmts);
        dblock.dblk_page_init_size = (dblock.npages + 7) / 8;
    } 

    
    if (!dblock.npages)
        *image_len = (size_t)H5FA_DBLOCK_SIZE(&dblock);
    else
        *image_len = (size_t)H5FA_DBLOCK_PREFIX_SIZE(&dblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5FA__cache_dblock_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5FA__cache_dblock_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                               bool H5_ATTR_UNUSED *dirty)
{
    H5FA_dblock_t          *dblock = NULL;                             
    H5FA_dblock_cache_ud_t *udata  = (H5FA_dblock_cache_ud_t *)_udata; 
    const uint8_t          *image  = (const uint8_t *)_image;          
    uint32_t                stored_chksum;                             
    haddr_t                 arr_addr; 
    void                   *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(udata);
    assert(udata->hdr);

    
    if (NULL == (dblock = H5FA__dblock_alloc(udata->hdr)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL, "memory allocation failed for fixed array data block");

    assert(((!dblock->npages) && (len == (size_t)H5FA_DBLOCK_SIZE(dblock))) ||
           (len == (size_t)H5FA_DBLOCK_PREFIX_SIZE(dblock)));

    
    dblock->addr = udata->dblk_addr;

    
    if (memcmp(image, H5FA_DBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, NULL, "wrong fixed array data block signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5FA_DBLOCK_VERSION)
        HGOTO_ERROR(H5E_FARRAY, H5E_VERSION, NULL, "wrong fixed array data block version");

    
    if (*image++ != (uint8_t)udata->hdr->cparam.cls->id)
        HGOTO_ERROR(H5E_FARRAY, H5E_BADTYPE, NULL, "incorrect fixed array class");

    
    H5F_addr_decode(udata->hdr->f, &image, &arr_addr);
    if (H5_addr_ne(arr_addr, udata->hdr->addr))
        HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, NULL, "wrong fixed array header address");

    
    if (dblock->npages > 0) {
        H5MM_memcpy(dblock->dblk_page_init, image, dblock->dblk_page_init_size);
        image += dblock->dblk_page_init_size;
    }

    
    if (!dblock->npages) {
        
        
        if ((udata->hdr->cparam.cls->decode)(image, dblock->elmts, (size_t)udata->hdr->cparam.nelmts,
                                             udata->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTDECODE, NULL, "can't decode fixed array data elements");
        image += (udata->hdr->cparam.nelmts * udata->hdr->cparam.raw_elmt_size);
    }

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5FA_SIZEOF_CHKSUM));

    
    dblock->size = H5FA_DBLOCK_SIZE(dblock);

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == len);

    
    ret_value = dblock;

done:
    
    if (!ret_value)
        if (dblock && H5FA__dblock_dest(dblock) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, NULL, "unable to destroy fixed array data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblock_image_len(const void *_thing, size_t *image_len)
{
    const H5FA_dblock_t *dblock = (const H5FA_dblock_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblock);
    assert(image_len);

    
    if (!dblock->npages)
        *image_len = (size_t)dblock->size;
    else
        *image_len = H5FA_DBLOCK_PREFIX_SIZE(dblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__cache_dblock_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5FA_dblock_t *dblock = (H5FA_dblock_t *)_thing; 
    uint8_t       *image  = (uint8_t *)_image;       
    uint32_t       metadata_chksum;                  
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(dblock);
    assert(dblock->hdr);

    
    H5MM_memcpy(image, H5FA_DBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5FA_DBLOCK_VERSION;

    
    assert(dblock->hdr->cparam.cls->id <= 255);
    *image++ = (uint8_t)dblock->hdr->cparam.cls->id;

    
    H5F_addr_encode(f, &image, dblock->hdr->addr);

    
    if (dblock->npages > 0) {
        
        H5MM_memcpy(image, dblock->dblk_page_init, dblock->dblk_page_init_size);
        image += dblock->dblk_page_init_size;
    }

    
    if (!dblock->npages) {
        

        
        H5_CHECK_OVERFLOW(dblock->hdr->cparam.nelmts,  hsize_t,  size_t);
        if ((dblock->hdr->cparam.cls->encode)(image, dblock->elmts, (size_t)dblock->hdr->cparam.nelmts,
                                              dblock->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTENCODE, FAIL, "can't encode fixed array data elements");
        image += (dblock->hdr->cparam.nelmts * dblock->hdr->cparam.raw_elmt_size);
    }

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblock_notify(H5AC_notify_action_t action, void *_thing)
{
    H5FA_dblock_t *dblock    = (H5FA_dblock_t *)_thing;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    if (dblock->hdr->swmr_write) {
        
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
                
                if (H5FA__create_flush_depend((H5AC_info_t *)dblock->hdr, (H5AC_info_t *)dblock) < 0)
                    HGOTO_ERROR(
                        H5E_FARRAY, H5E_CANTDEPEND, FAIL,
                        "unable to create flush dependency between data block and header, address = %llu",
                        (unsigned long long)dblock->addr);
                break;

            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (H5FA__destroy_flush_depend((H5AC_info_t *)dblock->hdr, (H5AC_info_t *)dblock) < 0)
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");

                
                if (dblock->top_proxy) {
                    if (H5AC_proxy_entry_remove_child(dblock->top_proxy, dblock) < 0)
                        HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNDEPEND, FAIL,
                                    "unable to destroy flush dependency between data block "
                                    "and fixed array 'top' proxy");
                    dblock->top_proxy = NULL;
                }
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else
                assert(0 && "Unknown action?!?");
#endif
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblock_free_icr(void *_thing)
{
    H5FA_dblock_t *dblock    = (H5FA_dblock_t *)_thing; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    if (H5FA__dblock_dest(dblock) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTFREE, FAIL, "can't free fixed array data block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblock_fsf_size(const void *_thing, hsize_t *fsf_size)
{
    const H5FA_dblock_t *dblock = (const H5FA_dblock_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblock);
    assert(dblock->cache_info.type == H5AC_FARRAY_DBLOCK);
    assert(fsf_size);

    *fsf_size = dblock->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__cache_dblk_page_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5FA_dblk_page_cache_ud_t *udata = (H5FA_dblk_page_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(udata->nelmts > 0);
    assert(image_len);

    
    *image_len = (size_t)H5FA_DBLK_PAGE_SIZE(udata->hdr, udata->nelmts);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5FA__cache_dblk_page_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5FA__cache_dblk_page_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5FA_dblk_page_t          *dblk_page = NULL; 
    H5FA_dblk_page_cache_ud_t *udata =
        (H5FA_dblk_page_cache_ud_t *)_udata;        
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    void          *ret_value = NULL;

    
    FUNC_ENTER_PACKAGE

    assert(udata);
    assert(udata->hdr);
    assert(udata->nelmts > 0);
    assert(H5_addr_defined(udata->dblk_page_addr));

    
    if (NULL == (dblk_page = H5FA__dblk_page_alloc(udata->hdr, udata->nelmts)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for fixed array data block page");

    
    dblk_page->addr = udata->dblk_page_addr;

    

    
    
    if ((udata->hdr->cparam.cls->decode)(image, dblk_page->elmts, udata->nelmts, udata->hdr->cb_ctx) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTDECODE, NULL, "can't decode fixed array data elements");
    image += (udata->nelmts * udata->hdr->cparam.raw_elmt_size);

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5FA_SIZEOF_CHKSUM));

    
    dblk_page->size = len;

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == dblk_page->size);

    
    ret_value = dblk_page;

done:

    
    if (!ret_value)
        if (dblk_page && H5FA__dblk_page_dest(dblk_page) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, NULL, "unable to destroy fixed array data block page");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblk_page_image_len(const void *_thing, size_t *image_len)
{
    const H5FA_dblk_page_t *dblk_page = (const H5FA_dblk_page_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblk_page);
    assert(image_len);

    
    *image_len = dblk_page->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__cache_dblk_page_serialize(const H5F_t H5_ATTR_NDEBUG_UNUSED *f, void *_image, size_t H5_ATTR_UNUSED len,
                                void *_thing)
{
    H5FA_dblk_page_t *dblk_page = (H5FA_dblk_page_t *)_thing; 
    uint8_t          *image     = (uint8_t *)_image;          
    uint32_t          metadata_chksum;                        
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(dblk_page);
    assert(dblk_page->hdr);

    

    

    
    if ((dblk_page->hdr->cparam.cls->encode)(image, dblk_page->elmts, dblk_page->nelmts,
                                             dblk_page->hdr->cb_ctx) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTENCODE, FAIL, "can't encode fixed array data elements");
    image += (dblk_page->nelmts * dblk_page->hdr->cparam.raw_elmt_size);

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblk_page_notify(H5AC_notify_action_t action, void *_thing)
{
    H5FA_dblk_page_t *dblk_page = (H5FA_dblk_page_t *)_thing; 
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblk_page);

    
    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            if (dblk_page->top_proxy) {
                if (H5AC_proxy_entry_remove_child(dblk_page->top_proxy, dblk_page) < 0)
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between data block page "
                                "and fixed array 'top' proxy");
                dblk_page->top_proxy = NULL;
            } 
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__cache_dblk_page_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5FA__dblk_page_dest((H5FA_dblk_page_t *)thing) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTFREE, FAIL, "can't free fixed array data block page");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
