/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDfamily.h"  
#include "H5FDpkg.h"     
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  

#define H5FD_FAM_MEMB_NAME_BUF_SIZE 4096

#define H5FD_FAM_DEF_MEM_SIZE ((hsize_t)(100 * H5_MB))

hid_t H5FD_FAMILY_id_g = H5I_INVALID_HID;

typedef struct H5FD_family_t {
    H5FD_t   pub;          
    hid_t    memb_fapl_id; 
    hsize_t  memb_size;    
    hsize_t  pmem_size;    
    unsigned nmembs;       
    unsigned amembs;       
    H5FD_t **memb;         
    haddr_t  eoa;          
    char    *name;         
    unsigned flags;        

    
    hsize_t mem_newsize; 
    bool repart_members; 
} H5FD_family_t;

typedef struct H5FD_family_fapl_t {
    hsize_t memb_size;    
    hid_t   memb_fapl_id; 
} H5FD_family_fapl_t;

static herr_t H5FD__family_get_default_config(H5FD_family_fapl_t *fa_out);
static char  *H5FD__family_get_default_printf_filename(const char *old_filename);

static void   *H5FD__family_fapl_get(H5FD_t *_file);
static void   *H5FD__family_fapl_copy(const void *_old_fa);
static herr_t  H5FD__family_fapl_free(void *_fa);
static hsize_t H5FD__family_sb_size(H5FD_t *_file);
static herr_t  H5FD__family_sb_encode(H5FD_t *_file, char *name , unsigned char *buf );
static herr_t  H5FD__family_sb_decode(H5FD_t *_file, const char *name, const unsigned char *buf);
static H5FD_t *H5FD__family_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr);
static herr_t  H5FD__family_close(H5FD_t *_file);
static int     H5FD__family_cmp(const H5FD_t *_f1, const H5FD_t *_f2);
static herr_t  H5FD__family_query(const H5FD_t *_f1, unsigned long *flags);
static haddr_t H5FD__family_get_eoa(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__family_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t eoa);
static haddr_t H5FD__family_get_eof(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__family_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle);
static herr_t  H5FD__family_read(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size,
                                 void *_buf );
static herr_t  H5FD__family_write(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size,
                                  const void *_buf);
static herr_t  H5FD__family_flush(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD__family_truncate(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD__family_lock(H5FD_t *_file, bool rw);
static herr_t  H5FD__family_unlock(H5FD_t *_file);
static herr_t  H5FD__family_delete(const char *filename, hid_t fapl_id);

static const H5FD_class_t H5FD_family_g = {
    H5FD_CLASS_VERSION,         
    H5FD_FAMILY_VALUE,          
    "family",                   
    HADDR_MAX,                  
    H5F_CLOSE_WEAK,             
    NULL,                       
    H5FD__family_sb_size,       
    H5FD__family_sb_encode,     
    H5FD__family_sb_decode,     
    sizeof(H5FD_family_fapl_t), 
    H5FD__family_fapl_get,      
    H5FD__family_fapl_copy,     
    H5FD__family_fapl_free,     
    0,                          
    NULL,                       
    NULL,                       
    H5FD__family_open,          
    H5FD__family_close,         
    H5FD__family_cmp,           
    H5FD__family_query,         
    NULL,                       
    NULL,                       
    NULL,                       
    H5FD__family_get_eoa,       
    H5FD__family_set_eoa,       
    H5FD__family_get_eof,       
    H5FD__family_get_handle,    
    H5FD__family_read,          
    H5FD__family_write,         
    NULL,                       
    NULL,                       
    NULL,                       
    NULL,                       
    H5FD__family_flush,         
    H5FD__family_truncate,      
    H5FD__family_lock,          
    H5FD__family_unlock,        
    H5FD__family_delete,        
    NULL,                       
    H5FD_FLMAP_DICHOTOMY        
};

static herr_t
H5FD__family_get_default_config(H5FD_family_fapl_t *fa_out)
{
    H5P_genplist_t *def_plist;
    H5P_genplist_t *plist;
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(fa_out);

    fa_out->memb_size = H5FD_FAM_DEF_MEM_SIZE;

    
    if (NULL == (def_plist = (H5P_genplist_t *)H5I_object(H5P_FILE_ACCESS_DEFAULT)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
    if ((fa_out->memb_fapl_id = H5P_copy_plist(def_plist, false)) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTCOPY, FAIL, "can't copy property list");
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fa_out->memb_fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
    if (H5P_set_driver_by_value(plist, H5_VFD_SEC2, NULL, true) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTSET, FAIL, "can't set default driver on member FAPL");

done:
    if (ret_value < 0 && fa_out->memb_fapl_id >= 0)
        if (H5I_dec_ref(fa_out->memb_fapl_id) < 0)
            HDONE_ERROR(H5E_VFL, H5E_CANTDEC, FAIL, "can't decrement ref. count on member FAPL ID");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static char *
H5FD__family_get_default_printf_filename(const char *old_filename)
{
    const char *suffix           = "-%06d";
    size_t      old_filename_len = 0;
    size_t      new_filename_len = 0;
    char       *file_extension   = NULL;
    char       *tmp_buffer       = NULL;
    char       *ret_value        = NULL;

    FUNC_ENTER_PACKAGE

    assert(old_filename);

    old_filename_len = strlen(old_filename);
    if (0 == old_filename_len)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, NULL, "invalid filename");

    new_filename_len = old_filename_len + strlen(suffix) + 1;
    if (NULL == (tmp_buffer = H5MM_malloc(new_filename_len)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "can't allocate new filename buffer");

    
    file_extension = strstr(old_filename, ".h5");
    if (file_extension) {
        
        intptr_t beginningLength = file_extension - old_filename;

        snprintf(tmp_buffer, new_filename_len, "%.*s%s%s", (int)beginningLength, old_filename, suffix, ".h5");
    }
    else {
        
        file_extension = strrchr(old_filename, '.');
        if (file_extension) {
            intptr_t beginningLength = file_extension - old_filename;

            snprintf(tmp_buffer, new_filename_len, "%.*s%s%s", (int)beginningLength, old_filename, suffix,
                     file_extension);
        }
        else
            
            snprintf(tmp_buffer, new_filename_len, "%s%s", old_filename, suffix);
    }

    ret_value = tmp_buffer;

done:
    if (!ret_value)
        H5MM_xfree(tmp_buffer);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__family_register(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5I_VFL != H5I_get_type(H5FD_FAMILY_id_g))
        if ((H5FD_FAMILY_id_g = H5FD_register(&H5FD_family_g, sizeof(H5FD_class_t), false)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register family driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__family_unregister(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5FD_FAMILY_id_g = H5I_INVALID_HID;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_fapl_family(hid_t fapl_id, hsize_t msize, hid_t memb_fapl_id)
{
    herr_t             ret_value;
    H5FD_family_fapl_t fa = {0, H5I_INVALID_HID};
    H5P_genplist_t    *plist; 

    FUNC_ENTER_API(FAIL)

    
    if (true != H5P_isa_class(fapl_id, H5P_FILE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
    if (H5P_DEFAULT == memb_fapl_id) {
        
        if (H5FD__family_get_default_config(&fa) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get default driver configuration info");
    }
    else if (true != H5P_isa_class(memb_fapl_id, H5P_FILE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access list");

    
    fa.memb_size = msize;
    if (H5P_DEFAULT != memb_fapl_id)
        fa.memb_fapl_id = memb_fapl_id;

    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
    ret_value = H5P_set_driver(plist, H5FD_FAMILY, &fa, NULL);

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Pget_fapl_family(hid_t fapl_id, hsize_t *msize , hid_t *memb_fapl_id )
{
    H5P_genplist_t           *plist; 
    const H5FD_family_fapl_t *fa;
    herr_t                    ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (NULL == (plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, true)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access list");
    if (H5FD_FAMILY != H5P_peek_driver(plist))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "incorrect VFL driver");
    if (NULL == (fa = (const H5FD_family_fapl_t *)H5P_peek_driver_info(plist)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "bad VFL driver info");
    if (msize)
        *msize = fa->memb_size;
    if (memb_fapl_id) {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(fa->memb_fapl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access list");
        *memb_fapl_id = H5P_copy_plist(plist, true);
    } 

done:
    FUNC_LEAVE_API(ret_value)
}

static void *
H5FD__family_fapl_get(H5FD_t *_file)
{
    H5FD_family_t      *file = (H5FD_family_t *)_file;
    H5FD_family_fapl_t *fa   = NULL;
    H5P_genplist_t     *plist;            
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    if (NULL == (fa = (H5FD_family_fapl_t *)H5MM_calloc(sizeof(H5FD_family_fapl_t))))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "memory allocation failed");

    fa->memb_size = file->memb_size;
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(file->memb_fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");
    fa->memb_fapl_id = H5P_copy_plist(plist, false);

    
    ret_value = fa;

done:
    if (ret_value == NULL)
        if (fa != NULL)
            H5MM_xfree(fa);

    FUNC_LEAVE_NOAPI(ret_value)
}

static void *
H5FD__family_fapl_copy(const void *_old_fa)
{
    const H5FD_family_fapl_t *old_fa = (const H5FD_family_fapl_t *)_old_fa;
    H5FD_family_fapl_t       *new_fa = NULL;
    H5P_genplist_t           *plist;            
    void                     *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    if (NULL == (new_fa = (H5FD_family_fapl_t *)H5MM_malloc(sizeof(H5FD_family_fapl_t))))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    H5MM_memcpy(new_fa, old_fa, sizeof(H5FD_family_fapl_t));

    
    if (old_fa->memb_fapl_id == H5P_FILE_ACCESS_DEFAULT) {
        if (H5I_inc_ref(new_fa->memb_fapl_id, false) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINC, NULL, "unable to increment ref count on VFL driver");
    } 
    else {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(old_fa->memb_fapl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");
        new_fa->memb_fapl_id = H5P_copy_plist(plist, false);
    } 

    
    ret_value = new_fa;

done:
    if (ret_value == NULL)
        if (new_fa != NULL)
            H5MM_xfree(new_fa);

    FUNC_LEAVE_NOAPI(ret_value)
}

static herr_t
H5FD__family_fapl_free(void *_fa)
{
    H5FD_family_fapl_t *fa        = (H5FD_family_fapl_t *)_fa;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5I_dec_ref(fa->memb_fapl_id) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDEC, FAIL, "can't close driver ID");
    H5MM_xfree(fa);

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static hsize_t
H5FD__family_sb_size(H5FD_t H5_ATTR_UNUSED *_file)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    FUNC_LEAVE_NOAPI(8)
}

static herr_t
H5FD__family_sb_encode(H5FD_t *_file, char *name , unsigned char *buf )
{
    H5FD_family_t *file = (H5FD_family_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    
    strncpy(name, "NCSAfami", (size_t)9);
    name[8] = '\0';

    
    UINT64ENCODE(buf, (uint64_t)file->pmem_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FD__family_sb_decode(H5FD_t *_file, const char H5_ATTR_UNUSED *name, const unsigned char *buf)
{
    H5FD_family_t *file = (H5FD_family_t *)_file;
    uint64_t       msize;
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    UINT64DECODE(buf, msize);

    
    if (file->mem_newsize)
        file->memb_size = file->pmem_size = file->mem_newsize;
    else {
        
        if (file->pmem_size == H5F_FAMILY_DEFAULT)
            file->pmem_size = msize;

        
        if (msize != file->pmem_size)
            HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL,
                        "Family member size should be %lu.  But the size from file access property is %lu",
                        (unsigned long)msize, (unsigned long)file->pmem_size);

        
        file->memb_size = msize;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5FD_t *
H5FD__family_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    H5FD_family_t *file      = NULL;
    char          *memb_name = NULL, *temp = NULL;
    hsize_t        eof            = HADDR_UNDEF;
    bool           default_config = false;
    unsigned       t_flags        = flags & ~H5F_ACC_CREAT;
    H5FD_t        *ret_value      = NULL;

    FUNC_ENTER_PACKAGE

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "invalid file name");
    if (0 == maxaddr || HADDR_UNDEF == maxaddr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, NULL, "bogus maxaddr");

    
    if (NULL == (file = (H5FD_family_t *)H5MM_calloc(sizeof(H5FD_family_t))))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "unable to allocate file struct");
    if (H5P_FILE_ACCESS_DEFAULT == fapl_id) {
        H5FD_family_fapl_t default_fa;

        
        if (H5FD__family_get_default_config(&default_fa) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get default driver configuration info");

        file->memb_fapl_id = default_fa.memb_fapl_id;
        file->memb_size    = H5FD_FAM_DEF_MEM_SIZE; 
        file->pmem_size    = H5FD_FAM_DEF_MEM_SIZE; 
        file->mem_newsize  = 0;                     

        default_config = true;
    } 
    else {
        H5P_genplist_t           *plist; 
        const H5FD_family_fapl_t *fa;
        H5FD_family_fapl_t        default_fa;

        if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");
        if (NULL == (fa = (const H5FD_family_fapl_t *)H5P_peek_driver_info(plist))) {
            if (H5FD__family_get_default_config(&default_fa) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get default family VFD configuration");
            fa             = &default_fa;
            default_config = true;
        }

        
        if (H5P_exist_plist(plist, H5F_ACS_FAMILY_NEWSIZE_NAME) > 0) {
            
            if (H5P_get(plist, H5F_ACS_FAMILY_NEWSIZE_NAME, &file->mem_newsize) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, NULL, "can't get new family member size");

            
            file->repart_members = true;
        } 

        if (fa->memb_fapl_id == H5P_FILE_ACCESS_DEFAULT) {
            if (H5I_inc_ref(fa->memb_fapl_id, false) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTINC, NULL, "unable to increment ref count on VFL driver");
            file->memb_fapl_id = fa->memb_fapl_id;
        } 
        else {
            if (NULL == (plist = (H5P_genplist_t *)H5I_object(fa->memb_fapl_id)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");
            file->memb_fapl_id = H5P_copy_plist(plist, false);
        }                                
        file->memb_size = fa->memb_size; 
        file->pmem_size = fa->memb_size; 

        if (default_config && H5I_dec_ref(fa->memb_fapl_id) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTDEC, NULL, "can't decrement ref. count on member FAPL");
    } 
    file->name  = H5MM_strdup(name);
    file->flags = flags;

    
    if (NULL == (memb_name = (char *)H5MM_malloc(H5FD_FAM_MEMB_NAME_BUF_SIZE)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "unable to allocate member name");
    if (NULL == (temp = (char *)H5MM_malloc(H5FD_FAM_MEMB_NAME_BUF_SIZE)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "unable to allocate temporary member name");

    
    H5_WARN_FORMAT_NONLITERAL_OFF
    snprintf(memb_name, H5FD_FAM_MEMB_NAME_BUF_SIZE, name, 0);
    snprintf(temp, H5FD_FAM_MEMB_NAME_BUF_SIZE, name, 1);
    H5_WARN_FORMAT_NONLITERAL_ON

    if (!strcmp(memb_name, temp)) {
        if (default_config) {
            temp = H5MM_xfree(temp);
            if (NULL == (temp = H5FD__family_get_default_printf_filename(name)))
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get default Rprintf-style filename");
            name = temp;
        }
        else
            HGOTO_ERROR(H5E_VFL, H5E_FILEEXISTS, NULL,
                        "differing member numbers do not produce unique member file names - try inserting "
                        "\"%%06d\" into the file name string");
    }

    
    while (1) {
        H5_WARN_FORMAT_NONLITERAL_OFF
        snprintf(memb_name, H5FD_FAM_MEMB_NAME_BUF_SIZE, name, file->nmembs);
        H5_WARN_FORMAT_NONLITERAL_ON

        
        if (file->nmembs >= file->amembs) {
            unsigned n = MAX(64, 2 * file->amembs);
            H5FD_t **x;

            assert(n > 0);
            if (NULL == (x = (H5FD_t **)H5MM_realloc(file->memb, n * sizeof(H5FD_t *))))
                HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "unable to reallocate members");
            file->amembs = n;
            file->memb   = x;
        } 

        
        if (0 == file->nmembs) {
            if (H5FD_open(false, &file->memb[file->nmembs], memb_name, (0 == file->nmembs ? flags : t_flags),
                          file->memb_fapl_id, HADDR_UNDEF) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTOPENFILE, NULL, "unable to open member file");
        }
        else {
            if (H5FD_open(true, &file->memb[file->nmembs], memb_name, (0 == file->nmembs ? flags : t_flags),
                          file->memb_fapl_id, HADDR_UNDEF) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTOPENFILE, NULL, "unable to open member file");

            if (!file->memb[file->nmembs])
                break;
        }

        file->nmembs++;
    }

    
    if ((eof = H5FD_get_eof(file->memb[0], H5FD_MEM_DEFAULT)))
        file->memb_size = eof;

    ret_value = (H5FD_t *)file;

done:
    
    if (memb_name)
        H5MM_xfree(memb_name);
    if (temp)
        H5MM_xfree(temp);

    
    if (ret_value == NULL && file != NULL) {
        unsigned nerrors = 0; 
        unsigned u;           

        
        for (u = 0; u < file->nmembs; u++)
            if (file->memb[u])
                if (H5FD_close(file->memb[u]) < 0)
                    nerrors++;
        if (nerrors)
            HDONE_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, NULL, "unable to close member files");

        if (file->memb)
            H5MM_xfree(file->memb);
        if (H5I_dec_ref(file->memb_fapl_id) < 0)
            HDONE_ERROR(H5E_VFL, H5E_CANTDEC, NULL, "can't close driver ID");
        if (file->name)
            H5MM_xfree(file->name);
        H5MM_xfree(file);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__family_close(H5FD_t *_file)
{
    H5FD_family_t *file    = (H5FD_family_t *)_file;
    unsigned       nerrors = 0;         
    unsigned       u;                   
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    for (u = 0; u < file->nmembs; u++) {
        if (file->memb[u]) {
            if (H5FD_close(file->memb[u]) < 0)
                nerrors++;
            else
                file->memb[u] = NULL;
        } 
    }     
    if (nerrors)
        
        HDONE_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "unable to close member files");

    
    if (H5I_dec_ref(file->memb_fapl_id) < 0)
        
        HDONE_ERROR(H5E_VFL, H5E_CANTDEC, FAIL, "can't close driver ID");
    H5MM_xfree(file->memb);
    H5MM_xfree(file->name);
    H5MM_xfree(file);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FD__family_cmp(const H5FD_t *_f1, const H5FD_t *_f2)
{
    const H5FD_family_t *f1        = (const H5FD_family_t *)_f1;
    const H5FD_family_t *f2        = (const H5FD_family_t *)_f2;
    int                  ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    assert(f1->nmembs >= 1 && f1->memb[0]);
    assert(f2->nmembs >= 1 && f2->memb[0]);

    ret_value = H5FD_cmp(f1->memb[0], f2->memb[0]);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__family_query(const H5FD_t *_file, unsigned long *flags )
{
    const H5FD_family_t *file = (const H5FD_family_t *)_file; 

    FUNC_ENTER_PACKAGE_NOERR

    
    if (flags) {
        *flags = 0;
        *flags |= H5FD_FEAT_AGGREGATE_METADATA;  
        *flags |= H5FD_FEAT_ACCUMULATE_METADATA; 
        *flags |= H5FD_FEAT_DATA_SIEVE; 
        *flags |= H5FD_FEAT_AGGREGATE_SMALLDATA; 

        
        if (file && file->repart_members)
            *flags |= H5FD_FEAT_DIRTY_DRVRINFO_LOAD; 
    }                                                

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__family_get_eoa(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_family_t *file = (const H5FD_family_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eoa)
}

static herr_t
H5FD__family_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t abs_eoa)
{
    H5FD_family_t *file      = (H5FD_family_t *)_file;
    haddr_t        addr      = abs_eoa;
    char          *memb_name = NULL;
    unsigned       u;                   
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (memb_name = (char *)H5MM_malloc(H5FD_FAM_MEMB_NAME_BUF_SIZE)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "unable to allocate member name");

    for (u = 0; addr || u < file->nmembs; u++) {

        
        if (u >= file->amembs) {
            unsigned n = MAX(64, 2 * file->amembs);
            H5FD_t **x = (H5FD_t **)H5MM_realloc(file->memb, n * sizeof(H5FD_t *));

            if (!x)
                HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "unable to allocate memory block");
            file->amembs = n;
            file->memb   = x;
            file->nmembs = u;
        } 

        
        if (u >= file->nmembs || !file->memb[u]) {
            file->nmembs = MAX(file->nmembs, u + 1);

            H5_WARN_FORMAT_NONLITERAL_OFF
            snprintf(memb_name, H5FD_FAM_MEMB_NAME_BUF_SIZE, file->name, u);
            H5_WARN_FORMAT_NONLITERAL_ON

            H5_CHECK_OVERFLOW(file->memb_size, hsize_t, haddr_t);
            if (H5FD_open(false, &file->memb[u], memb_name, file->flags | H5F_ACC_CREAT, file->memb_fapl_id,
                          (haddr_t)file->memb_size) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTOPENFILE, FAIL, "unable to open member file");
        } 

        
        H5_CHECK_OVERFLOW(file->memb_size, hsize_t, haddr_t);
        if (addr > (haddr_t)file->memb_size) {
            if (H5FD_set_eoa(file->memb[u], type, (haddr_t)file->memb_size) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "unable to set file eoa");
            addr -= file->memb_size;
        } 
        else {
            if (H5FD_set_eoa(file->memb[u], type, addr) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "unable to set file eoa");
            addr = 0;
        } 
    }     

    file->eoa = abs_eoa;

done:
    
    if (memb_name)
        H5MM_xfree(memb_name);

    FUNC_LEAVE_NOAPI(ret_value)
}

static haddr_t
H5FD__family_get_eof(const H5FD_t *_file, H5FD_mem_t type)
{
    const H5FD_family_t *file = (const H5FD_family_t *)_file;
    haddr_t              eof  = 0;
    int                  i;                       
    haddr_t              ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(file->nmembs > 0);
    for (i = (int)(file->nmembs - 1); i >= 0; --i) {
        if ((eof = H5FD_get_eof(file->memb[i], type)) != 0)
            break;
        if (0 == i)
            break;
    } 

    
    eof += ((unsigned)i) * file->memb_size;

    
    ret_value = eof;

    FUNC_LEAVE_NOAPI(ret_value)
}

static herr_t
H5FD__family_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle)
{
    H5FD_family_t  *file = (H5FD_family_t *)_file;
    H5P_genplist_t *plist;
    hsize_t         offset;
    int             memb;
    herr_t          ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (plist = H5P_object_verify(fapl, H5P_FILE_ACCESS, true)))
        HGOTO_ERROR(H5E_VFL, H5E_BADID, FAIL, "can't find object for ID");
    if (H5P_get(plist, H5F_ACS_FAMILY_OFFSET_NAME, &offset) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get offset for family driver");

    if (offset > (file->memb_size * file->nmembs))
        HGOTO_ERROR(H5E_VFL, H5E_BADID, FAIL, "offset is bigger than file size");
    memb = (int)(offset / file->memb_size);

    ret_value = H5FD_get_vfd_handle(file->memb[memb], fapl, file_handle);

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static herr_t
H5FD__family_read(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size,
                  void *_buf )
{
    H5FD_family_t  *file = (H5FD_family_t *)_file;
    unsigned char  *buf  = (unsigned char *)_buf;
    haddr_t         sub;
    size_t          req;
    hsize_t         tempreq;
    unsigned        u;                   
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(dxpl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    while (size > 0) {
        H5_CHECKED_ASSIGN(u, unsigned, addr / file->memb_size, hsize_t);

        sub = addr % file->memb_size;

        
        tempreq = file->memb_size - sub;
        if (tempreq > SIZE_MAX)
            tempreq = SIZE_MAX;
        req = MIN(size, (size_t)tempreq);

        assert(u < file->nmembs);

        if (H5FD_read(file->memb[u], type, sub, req, buf) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "member file read failed");

        addr += req;
        buf += req;
        size -= req;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static herr_t
H5FD__family_write(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size, const void *_buf)
{
    H5FD_family_t       *file = (H5FD_family_t *)_file;
    const unsigned char *buf  = (const unsigned char *)_buf;
    haddr_t              sub;
    size_t               req;
    hsize_t              tempreq;
    unsigned             u;                   
    H5P_genplist_t      *plist;               
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(dxpl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    while (size > 0) {
        H5_CHECKED_ASSIGN(u, unsigned, addr / file->memb_size, hsize_t);

        sub = addr % file->memb_size;

        
        tempreq = file->memb_size - sub;
        if (tempreq > SIZE_MAX)
            tempreq = SIZE_MAX;
        req = MIN(size, (size_t)tempreq);

        assert(u < file->nmembs);

        if (H5FD_write(file->memb[u], type, sub, req, buf) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "member file write failed");

        addr += req;
        buf += req;
        size -= req;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static herr_t
H5FD__family_flush(H5FD_t *_file, hid_t H5_ATTR_UNUSED dxpl_id, bool closing)
{
    H5FD_family_t *file = (H5FD_family_t *)_file;
    unsigned       u, nerrors = 0;
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    for (u = 0; u < file->nmembs; u++)
        if (file->memb[u] && H5FD_flush(file->memb[u], closing) < 0)
            nerrors++;

    if (nerrors)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "unable to flush member files");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__family_truncate(H5FD_t *_file, hid_t H5_ATTR_UNUSED dxpl_id, bool closing)
{
    H5FD_family_t *file = (H5FD_family_t *)_file;
    unsigned       u, nerrors = 0;
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    for (u = 0; u < file->nmembs; u++)
        if (file->memb[u] && H5FD_truncate(file->memb[u], closing) < 0)
            nerrors++;

    if (nerrors)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "unable to flush member files");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__family_lock(H5FD_t *_file, bool rw)
{
    H5FD_family_t *file = (H5FD_family_t *)_file; 
    unsigned       u;                             
    herr_t         ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    
    for (u = 0; u < file->nmembs; u++)
        if (file->memb[u])
            if (H5FD_lock(file->memb[u], rw) < 0)
                break;

    
    if (u < file->nmembs) {
        unsigned v; 

        for (v = 0; v < u; v++)
            if (H5FD_unlock(file->memb[v]) < 0)
                
                HDONE_ERROR(H5E_IO, H5E_CANTUNLOCKFILE, FAIL, "unable to unlock member files");

        HGOTO_ERROR(H5E_VFL, H5E_CANTLOCKFILE, FAIL, "unable to lock member files");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__family_unlock(H5FD_t *_file)
{
    H5FD_family_t *file = (H5FD_family_t *)_file; 
    unsigned       u;                             
    herr_t         ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    
    for (u = 0; u < file->nmembs; u++)
        if (file->memb[u])
            if (H5FD_unlock(file->memb[u]) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTUNLOCKFILE, FAIL, "unable to unlock member files");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__family_delete(const char *filename, hid_t fapl_id)
{
    H5P_genplist_t           *plist;
    const H5FD_family_fapl_t *fa;
    H5FD_family_fapl_t        default_fa     = {0, H5I_INVALID_HID};
    bool                      default_config = false;
    hid_t                     memb_fapl_id   = H5I_INVALID_HID;
    unsigned                  current_member;
    char                     *member_name = NULL;
    char                     *temp        = NULL;
    herr_t                    ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    if (!filename)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "invalid filename pointer");

    
    if (H5P_FILE_ACCESS_DEFAULT == fapl_id) {
        if (H5FD__family_get_default_config(&default_fa) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get default family VFD configuration");
        memb_fapl_id   = default_fa.memb_fapl_id;
        default_config = true;
    }
    else {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
        if (NULL == (fa = (const H5FD_family_fapl_t *)H5P_peek_driver_info(plist))) {
            if (H5FD__family_get_default_config(&default_fa) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get default family VFD configuration");
            fa             = &default_fa;
            default_config = true;
        }
        memb_fapl_id = fa->memb_fapl_id;
    }

    
    if (NULL == (member_name = (char *)H5MM_malloc(H5FD_FAM_MEMB_NAME_BUF_SIZE)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "unable to allocate member name");
    if (NULL == (temp = (char *)H5MM_malloc(H5FD_FAM_MEMB_NAME_BUF_SIZE)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "unable to allocate temporary member name");

    

    H5_WARN_FORMAT_NONLITERAL_OFF
    snprintf(member_name, H5FD_FAM_MEMB_NAME_BUF_SIZE, filename, 0);
    snprintf(temp, H5FD_FAM_MEMB_NAME_BUF_SIZE, filename, 1);
    H5_WARN_FORMAT_NONLITERAL_ON

    if (!strcmp(member_name, temp)) {
        if (default_config) {
            temp = H5MM_xfree(temp);
            if (NULL == (temp = H5FD__family_get_default_printf_filename(filename)))
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get default Rprintf-style filename");
            filename = temp;
        }
        else
            HGOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL,
                        "provided file name cannot generate unique sub-files");
    }

    
    current_member = 0;
    while (1) {
        
        H5_WARN_FORMAT_NONLITERAL_OFF
        snprintf(member_name, H5FD_FAM_MEMB_NAME_BUF_SIZE, filename, current_member);
        H5_WARN_FORMAT_NONLITERAL_ON

        
        if (0 == current_member) {
            if (H5FD_delete(member_name, memb_fapl_id) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL, "unable to delete member file");
        }
        else {
            herr_t delete_error;

            H5E_PAUSE_ERRORS
                {
                    delete_error = H5FD_delete(member_name, memb_fapl_id);
                }
            H5E_RESUME_ERRORS
            if (delete_error < 0)
                break;
        }

        current_member++;
    } 

done:
    if (member_name)
        H5MM_xfree(member_name);
    if (temp)
        H5MM_xfree(temp);

    
    if (default_fa.memb_fapl_id >= 0 && H5I_dec_ref(default_fa.memb_fapl_id) < 0)
        HDONE_ERROR(H5E_VFL, H5E_CANTDEC, FAIL, "can't decrement ref. count on member FAPL ID");

    FUNC_LEAVE_NOAPI(ret_value)
} 
