/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HLmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HLpkg.h"     
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 

#define H5HL_MIN_HEAP 128 

static H5HL_free_t *H5HL__remove_free(H5HL_t *heap, H5HL_free_t *fl);
static herr_t       H5HL__minimize_heap_space(H5F_t *f, H5HL_t *heap);
static herr_t       H5HL__dirty(H5HL_t *heap);

bool H5_PKG_INIT_VAR = false;

H5FL_DEFINE(H5HL_free_t);

H5FL_BLK_DEFINE(lheap_chunk);

herr_t
H5HL_create(H5F_t *f, size_t size_hint, haddr_t *addr_p )
{
    H5HL_t      *heap       = NULL; 
    H5HL_prfx_t *prfx       = NULL; 
    hsize_t      total_size = 0;    
    herr_t       ret_value  = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(addr_p);

    
    if (size_hint && size_hint < H5HL_SIZEOF_FREE(f))
        size_hint = H5HL_SIZEOF_FREE(f);
    size_hint = H5HL_ALIGN(size_hint);

    
    if (NULL == (heap = H5HL__new(H5F_SIZEOF_SIZE(f), H5F_SIZEOF_ADDR(f), H5HL_SIZEOF_HDR(f))))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't allocate new heap struct");

    
    total_size = heap->prfx_size + size_hint;
    if (HADDR_UNDEF == (heap->prfx_addr = H5MF_alloc(f, H5FD_MEM_LHEAP, total_size)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "unable to allocate file memory");

    
    heap->single_cache_obj = true;
    heap->dblk_addr        = heap->prfx_addr + (hsize_t)heap->prfx_size;
    heap->dblk_size        = size_hint;
    if (size_hint)
        if (NULL == (heap->dblk_image = H5FL_BLK_CALLOC(lheap_chunk, size_hint)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    if (size_hint) {
        if (NULL == (heap->freelist = H5FL_MALLOC(H5HL_free_t)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");
        heap->freelist->offset = 0;
        heap->freelist->size   = size_hint;
        heap->freelist->prev = heap->freelist->next = NULL;
        heap->free_block                            = 0;
    }
    else {
        heap->freelist   = NULL;
        heap->free_block = H5HL_FREE_NULL;
    }

    
    if (NULL == (prfx = H5HL__prfx_new(heap)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    if (FAIL == H5AC_insert_entry(f, H5AC_LHEAP_PRFX, heap->prfx_addr, prfx, H5AC__NO_FLAGS_SET))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "unable to cache local heap prefix");

    
    *addr_p = heap->prfx_addr;

done:
    if (ret_value < 0) {
        *addr_p = HADDR_UNDEF;
        if (prfx) {
            if (FAIL == H5HL__prfx_dest(prfx))
                HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy local heap prefix");
        }
        else {
            if (heap) {
                if (H5_addr_defined(heap->prfx_addr))
                    if (FAIL == H5MF_xfree(f, H5FD_MEM_LHEAP, heap->prfx_addr, total_size))
                        HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "can't release heap data?");
                if (FAIL == H5HL__dest(heap))
                    HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy local heap");
            }
        }
    }
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__minimize_heap_space(H5F_t *f, H5HL_t *heap)
{
    size_t new_heap_size = heap->dblk_size; 
    herr_t ret_value     = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(heap);

    
    if (heap->freelist) {
        H5HL_free_t *tmp_fl;
        H5HL_free_t *last_fl = NULL;

        
        for (tmp_fl = heap->freelist; tmp_fl; tmp_fl = tmp_fl->next)
            
            if (tmp_fl->offset + tmp_fl->size == heap->dblk_size) {
                last_fl = tmp_fl;
                break;
            }

        
        if (last_fl) {
            
            if (last_fl->size >= (heap->dblk_size / 2) && heap->dblk_size > H5HL_MIN_HEAP) {
                
                while (new_heap_size > H5HL_MIN_HEAP &&
                       new_heap_size >= (last_fl->offset + H5HL_SIZEOF_FREE(f)))
                    new_heap_size /= 2;

                
                if (new_heap_size < (last_fl->offset + H5HL_SIZEOF_FREE(f))) {
                    
                    if (last_fl->prev == NULL && last_fl->next == NULL) {
                        
                        new_heap_size *= 2;

                        
                        last_fl->size = H5HL_ALIGN(new_heap_size - last_fl->offset);
                        new_heap_size = last_fl->offset + last_fl->size;
                        assert(last_fl->size >= H5HL_SIZEOF_FREE(f));
                    }
                    else {
                        
                        new_heap_size = last_fl->offset;

                        
                        last_fl = H5HL__remove_free(heap, last_fl);
                    }
                }
                else {
                    
                    last_fl->size = H5HL_ALIGN(new_heap_size - last_fl->offset);
                    new_heap_size = last_fl->offset + last_fl->size;
                    assert(last_fl->size >= H5HL_SIZEOF_FREE(f));
                    assert(last_fl->size == H5HL_ALIGN(last_fl->size));
                }
            }
        }
    }

    
    if (new_heap_size != heap->dblk_size) {
        assert(new_heap_size < heap->dblk_size);

        
        if (NULL == (heap->dblk_image = H5FL_BLK_REALLOC(lheap_chunk, heap->dblk_image, new_heap_size)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");

        
        if (FAIL == H5HL__dblk_realloc(f, heap, new_heap_size))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "reallocating data block failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HL_t *
H5HL_protect(H5F_t *f, haddr_t addr, unsigned flags)
{
    H5HL_cache_prfx_ud_t prfx_udata;       
    H5HL_prfx_t         *prfx      = NULL; 
    H5HL_dblk_t         *dblk      = NULL; 
    H5HL_t              *heap      = NULL; 
    H5HL_t              *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    prfx_udata.sizeof_size = H5F_SIZEOF_SIZE(f);
    prfx_udata.sizeof_addr = H5F_SIZEOF_ADDR(f);
    prfx_udata.prfx_addr   = addr;
    prfx_udata.sizeof_prfx = H5HL_SIZEOF_HDR(f);

    
    if (NULL == (prfx = (H5HL_prfx_t *)H5AC_protect(f, H5AC_LHEAP_PRFX, addr, &prfx_udata, flags)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to load heap prefix");

    
    heap = prfx->heap;

    
    
    if (heap->prots == 0) {
        void *pin_obj; 

        
        if (heap->single_cache_obj)
            
            pin_obj = prfx;
        else {
            
            if (NULL ==
                (dblk = (H5HL_dblk_t *)H5AC_protect(f, H5AC_LHEAP_DBLK, heap->dblk_addr, heap, flags)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to load heap data block");

            
            pin_obj = dblk;
        }

        
        if (H5AC_pin_protected_entry(pin_obj) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPIN, NULL, "unable to pin local heap object");
    }

    
    heap->prots++;

    
    ret_value = heap;

done:
    
    if (prfx && heap && H5AC_unprotect(f, H5AC_LHEAP_PRFX, heap->prfx_addr, prfx, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, NULL, "unable to release local heap prefix");

    
    if (dblk && heap && H5AC_unprotect(f, H5AC_LHEAP_DBLK, heap->dblk_addr, dblk, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, NULL, "unable to release local heap data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5HL_offset_into(const H5HL_t *heap, size_t offset)
{
    void *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(heap);
    if (offset >= heap->dblk_size)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, NULL, "unable to offset into local heap data block");

    ret_value = heap->dblk_image + offset;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL_unprotect(H5HL_t *heap)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(heap);

    
    heap->prots--;

    
    if (heap->prots == 0) {
        
        if (heap->single_cache_obj) {
            
            if (FAIL == H5AC_unpin_entry(heap->prfx))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin local heap data block");
        }
        else {
            
            assert(heap->dblk);

            
            
            if (FAIL == H5AC_unpin_entry(heap->dblk))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin local heap data block");
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HL_free_t *
H5HL__remove_free(H5HL_t *heap, H5HL_free_t *fl)
{
    H5HL_free_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE_NOERR

    if (fl->prev)
        fl->prev->next = fl->next;
    if (fl->next)
        fl->next->prev = fl->prev;

    if (!fl->prev)
        heap->freelist = fl->next;

    
    ret_value = (H5HL_free_t *)H5FL_FREE(H5HL_free_t, fl);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HL__dirty(H5HL_t *heap)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(heap);
    assert(heap->prfx);

    
    if (!heap->single_cache_obj) {
        
        assert(heap->dblk);

        if (FAIL == H5AC_mark_entry_dirty(heap->dblk))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTMARKDIRTY, FAIL, "unable to mark heap data block as dirty");
    }

    
    if (FAIL == H5AC_mark_entry_dirty(heap->prfx))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTMARKDIRTY, FAIL, "unable to mark heap prefix as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL_insert(H5F_t *f, H5HL_t *heap, size_t buf_size, const void *buf, size_t *offset_out)
{
    H5HL_free_t *fl = NULL, *last_fl = NULL;
    size_t       need_size;
    size_t       offset = 0;
    bool         found;
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(heap);
    assert(buf_size > 0);
    assert(buf);
    assert(offset_out);

    
    
    if (FAIL == H5HL__dirty(heap))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTMARKDIRTY, FAIL, "unable to mark heap as dirty");

    
    need_size = H5HL_ALIGN(buf_size);

    
    for (fl = heap->freelist, found = false; fl; fl = fl->next) {
        if (fl->size > need_size && fl->size - need_size >= H5HL_SIZEOF_FREE(f)) {
            
            offset = fl->offset;
            fl->offset += need_size;
            fl->size -= need_size;
            assert(fl->offset == H5HL_ALIGN(fl->offset));
            assert(fl->size == H5HL_ALIGN(fl->size));
            found = true;
            break;
        }
        else if (fl->size == need_size) {
            
            offset = fl->offset;
            fl     = H5HL__remove_free(heap, fl);
            found  = true;
            break;
        }
        else if (!last_fl || last_fl->offset < fl->offset) {
            
            last_fl = fl;
        }
    }

    
    if (found == false) {
        size_t need_more;     
        size_t new_dblk_size; 
        size_t old_dblk_size; 
        htri_t was_extended;  

        
        need_more = MAX(need_size, heap->dblk_size);

        
        if (!(last_fl && last_fl->offset + last_fl->size == heap->dblk_size) &&
            (need_more < (need_size + H5HL_SIZEOF_FREE(f))))
            need_more = need_size;

        new_dblk_size = heap->dblk_size + need_more;
        assert(heap->dblk_size < new_dblk_size);
        old_dblk_size = heap->dblk_size;
        H5_CHECK_OVERFLOW(heap->dblk_size, size_t, hsize_t);
        H5_CHECK_OVERFLOW(new_dblk_size, size_t, hsize_t);

        
        was_extended = H5MF_try_extend(f, H5FD_MEM_LHEAP, heap->dblk_addr, (hsize_t)(heap->dblk_size),
                                       (hsize_t)need_more);
        if (FAIL == was_extended)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "error trying to extend heap");

        
        if (was_extended == true) {
            
            if (heap->single_cache_obj) {
                
                if (FAIL == H5AC_resize_entry(heap->prfx, (size_t)(heap->prfx_size + new_dblk_size)))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize heap prefix in cache");
            }
            else {
                
                if (FAIL == H5AC_resize_entry(heap->dblk, (size_t)new_dblk_size))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize heap data block in cache");
            }

            
            heap->dblk_size = new_dblk_size;
        }
        else { 
            
            if (FAIL == H5HL__dblk_realloc(f, heap, new_dblk_size))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "reallocating data block failed");
        }

        
        if (last_fl && last_fl->offset + last_fl->size == old_dblk_size) {
            
            offset = last_fl->offset;
            last_fl->offset += need_size;
            last_fl->size += need_more - need_size;
            assert(last_fl->offset == H5HL_ALIGN(last_fl->offset));
            assert(last_fl->size == H5HL_ALIGN(last_fl->size));

            if (last_fl->size < H5HL_SIZEOF_FREE(f))
                last_fl = H5HL__remove_free(heap, last_fl);
        }
        else {
            
            offset = old_dblk_size;
            if (need_more - need_size >= H5HL_SIZEOF_FREE(f)) {
                if (NULL == (fl = H5FL_MALLOC(H5HL_free_t)))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");
                fl->offset = old_dblk_size + need_size;
                fl->size   = need_more - need_size;
                assert(fl->offset == H5HL_ALIGN(fl->offset));
                assert(fl->size == H5HL_ALIGN(fl->size));
                fl->prev = NULL;
                fl->next = heap->freelist;
                if (heap->freelist)
                    heap->freelist->prev = fl;
                heap->freelist = fl;
            }
        }

        if (NULL == (heap->dblk_image = H5FL_BLK_REALLOC(lheap_chunk, heap->dblk_image, heap->dblk_size)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");

        
        
        memset(heap->dblk_image + offset + buf_size, 0, (new_dblk_size - (offset + buf_size)));
    }

    
    H5MM_memcpy(heap->dblk_image + offset, buf, buf_size);

    *offset_out = offset;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL_remove(H5F_t *f, H5HL_t *heap, size_t offset, size_t size)
{
    H5HL_free_t *fl        = NULL;
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(heap);
    assert(size > 0);
    assert(offset == H5HL_ALIGN(offset));

    size = H5HL_ALIGN(size);

    assert(offset < heap->dblk_size);
    assert(offset + size <= heap->dblk_size);

    
    if (FAIL == H5HL__dirty(heap))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTMARKDIRTY, FAIL, "unable to mark heap as dirty");

    
    fl = heap->freelist;
    while (fl) {
        H5HL_free_t *fl2 = NULL;

        if ((offset + size) == fl->offset) {
            fl->offset = offset;
            fl->size += size;
            assert(fl->offset == H5HL_ALIGN(fl->offset));
            assert(fl->size == H5HL_ALIGN(fl->size));
            fl2 = fl->next;
            while (fl2) {
                if ((fl2->offset + fl2->size) == fl->offset) {
                    fl->offset = fl2->offset;
                    fl->size += fl2->size;
                    assert(fl->offset == H5HL_ALIGN(fl->offset));
                    assert(fl->size == H5HL_ALIGN(fl->size));
                    fl2 = H5HL__remove_free(heap, fl2);
                    if (((fl->offset + fl->size) == heap->dblk_size) && ((2 * fl->size) > heap->dblk_size)) {
                        if (FAIL == H5HL__minimize_heap_space(f, heap))
                            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "heap size minimization failed");
                    }
                    HGOTO_DONE(SUCCEED);
                }
                fl2 = fl2->next;
            }
            if (((fl->offset + fl->size) == heap->dblk_size) && ((2 * fl->size) > heap->dblk_size)) {
                if (FAIL == H5HL__minimize_heap_space(f, heap))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "heap size minimization failed");
            }
            HGOTO_DONE(SUCCEED);
        }
        else if (fl->offset + fl->size == offset) {
            fl->size += size;
            fl2 = fl->next;
            assert(fl->size == H5HL_ALIGN(fl->size));
            while (fl2) {
                if (fl->offset + fl->size == fl2->offset) {
                    fl->size += fl2->size;
                    assert(fl->size == H5HL_ALIGN(fl->size));
                    fl2 = H5HL__remove_free(heap, fl2);
                    if (((fl->offset + fl->size) == heap->dblk_size) && ((2 * fl->size) > heap->dblk_size)) {
                        if (FAIL == H5HL__minimize_heap_space(f, heap))
                            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "heap size minimization failed");
                    }
                    HGOTO_DONE(SUCCEED);
                }
                fl2 = fl2->next;
            }
            if (((fl->offset + fl->size) == heap->dblk_size) && ((2 * fl->size) > heap->dblk_size)) {
                if (FAIL == H5HL__minimize_heap_space(f, heap))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "heap size minimization failed");
            }
            HGOTO_DONE(SUCCEED);
        }
        fl = fl->next;
    }

    
    if (size < H5HL_SIZEOF_FREE(f))
        HGOTO_DONE(SUCCEED);

    
    if (NULL == (fl = H5FL_MALLOC(H5HL_free_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "memory allocation failed");
    fl->offset = offset;
    fl->size   = size;
    assert(fl->offset == H5HL_ALIGN(fl->offset));
    assert(fl->size == H5HL_ALIGN(fl->size));
    fl->prev = NULL;
    fl->next = heap->freelist;
    if (heap->freelist)
        heap->freelist->prev = fl;
    heap->freelist = fl;

    if (((fl->offset + fl->size) == heap->dblk_size) && ((2 * fl->size) > heap->dblk_size))
        if (FAIL == H5HL__minimize_heap_space(f, heap))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "heap size minimization failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL_delete(H5F_t *f, haddr_t addr)
{
    H5HL_t              *heap = NULL;                      
    H5HL_cache_prfx_ud_t prfx_udata;                       
    H5HL_prfx_t         *prfx        = NULL;               
    H5HL_dblk_t         *dblk        = NULL;               
    unsigned             cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t               ret_value   = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    prfx_udata.sizeof_size = H5F_SIZEOF_SIZE(f);
    prfx_udata.sizeof_addr = H5F_SIZEOF_ADDR(f);
    prfx_udata.prfx_addr   = addr;
    prfx_udata.sizeof_prfx = H5HL_SIZEOF_HDR(f);

    
    if (NULL ==
        (prfx = (H5HL_prfx_t *)H5AC_protect(f, H5AC_LHEAP_PRFX, addr, &prfx_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load heap prefix");

    
    heap = prfx->heap;

    
    if (!heap->single_cache_obj)
        
        if (NULL == (dblk = (H5HL_dblk_t *)H5AC_protect(f, H5AC_LHEAP_DBLK, heap->dblk_addr, heap,
                                                        H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load heap data block");

    
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (dblk && heap && H5AC_unprotect(f, H5AC_LHEAP_DBLK, heap->dblk_addr, dblk, cache_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release local heap data block");

    
    if (prfx && heap && H5AC_unprotect(f, H5AC_LHEAP_PRFX, heap->prfx_addr, prfx, cache_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release local heap prefix");

    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5HL_heap_get_size(const H5HL_t *heap)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(heap);

    FUNC_LEAVE_NOAPI(heap->dblk_size)
} 

herr_t
H5HL_get_size(H5F_t *f, haddr_t addr, size_t *size)
{
    H5HL_cache_prfx_ud_t prfx_udata;       
    H5HL_prfx_t         *prfx      = NULL; 
    H5HL_t              *heap      = NULL; 
    herr_t               ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(size);

    
    prfx_udata.sizeof_size = H5F_SIZEOF_SIZE(f);
    prfx_udata.sizeof_addr = H5F_SIZEOF_ADDR(f);
    prfx_udata.prfx_addr   = addr;
    prfx_udata.sizeof_prfx = H5HL_SIZEOF_HDR(f);

    
    if (NULL ==
        (prfx = (H5HL_prfx_t *)H5AC_protect(f, H5AC_LHEAP_PRFX, addr, &prfx_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load heap prefix");

    
    heap = prfx->heap;

    
    *size = heap->dblk_size;

done:
    if (prfx && FAIL == H5AC_unprotect(f, H5AC_LHEAP_PRFX, heap->prfx_addr, prfx, H5AC__NO_FLAGS_SET))
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release local heap prefix");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL_heapsize(H5F_t *f, haddr_t addr, hsize_t *heap_size)
{
    H5HL_cache_prfx_ud_t prfx_udata;       
    H5HL_prfx_t         *prfx      = NULL; 
    H5HL_t              *heap      = NULL; 
    herr_t               ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(heap_size);

    
    prfx_udata.sizeof_size = H5F_SIZEOF_SIZE(f);
    prfx_udata.sizeof_addr = H5F_SIZEOF_ADDR(f);
    prfx_udata.prfx_addr   = addr;
    prfx_udata.sizeof_prfx = H5HL_SIZEOF_HDR(f);

    
    if (NULL ==
        (prfx = (H5HL_prfx_t *)H5AC_protect(f, H5AC_LHEAP_PRFX, addr, &prfx_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load heap prefix");

    
    heap = prfx->heap;

    
    *heap_size += (hsize_t)(heap->prfx_size + heap->dblk_size);

done:
    if (prfx && FAIL == H5AC_unprotect(f, H5AC_LHEAP_PRFX, heap->prfx_addr, prfx, H5AC__NO_FLAGS_SET))
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release local heap prefix");

    FUNC_LEAVE_NOAPI(ret_value)
} 
