/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Imodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Ipkg.h"      
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  

typedef struct {
    H5I_search_func_t app_cb;  
    void             *app_key; 
    void             *ret_obj; 
} H5I_search_ud_t;

typedef struct {
    H5I_iterate_func_t op;      
    void              *op_data; 
} H5I_iterate_pub_ud_t;

static int H5I__search_cb(void *obj, hid_t id, void *_udata);
static int H5I__iterate_pub_cb(void *obj, hid_t id, void *udata);

H5I_type_t
H5Iregister_type2(unsigned reserved, H5I_free_t free_func)
{
    H5I_type_t ret_value = H5I_BADID;

    FUNC_ENTER_API(H5I_BADID)

    if (H5I_BADID == (ret_value = H5I__register_type_common(reserved, free_func)))
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, H5I_BADID, "can't initialize ID class");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Itype_exists(H5I_type_t type)
{
    htri_t ret_value = true; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "cannot call public function on library type");
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");

    if (NULL == H5I_type_info_array_g[type])
        ret_value = false;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Inmembers(H5I_type_t type, hsize_t *num_members)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "cannot call public function on library type");

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");
    if (NULL == H5I_type_info_array_g[type])
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "supplied type does not exist");

    if (num_members) {
        int64_t members;

        if ((members = H5I_nmembers(type)) < 0)
            HGOTO_ERROR(H5E_ID, H5E_CANTCOUNT, FAIL, "can't compute number of members");

        H5_CHECKED_ASSIGN(*num_members, hsize_t, members, int64_t);
    }

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Iclear_type(H5I_type_t type, bool force)
{
    herr_t ret_value = FAIL; 

    FUNC_ENTER_API(FAIL)

    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "cannot call public function on library type");

    ret_value = H5I_clear_type(type, force, true);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Idestroy_type(H5I_type_t type)
{
    herr_t ret_value = FAIL; 

    FUNC_ENTER_API(FAIL)

    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "cannot call public function on library type");

    ret_value = H5I__destroy_type(type);

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Iregister(H5I_type_t type, const void *object)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, H5I_INVALID_HID, "cannot call public function on library type");

    
    if ((ret_value = H5I__register(type, object, true, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register object");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Iregister_future(H5I_type_t type, const void *object, H5I_future_realize_func_t realize_cb,
                   H5I_future_discard_func_t discard_cb)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == realize_cb)
        HGOTO_ERROR(H5E_ID, H5E_BADVALUE, H5I_INVALID_HID, "NULL pointer for realize_cb not allowed");
    if (NULL == discard_cb)
        HGOTO_ERROR(H5E_ID, H5E_BADVALUE, H5I_INVALID_HID, "NULL pointer for realize_cb not allowed");

    
    if ((ret_value = H5I__register(type, object, true, realize_cb, discard_cb)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register object");

done:
    FUNC_LEAVE_API(ret_value)
} 

void *
H5Iobject_verify(hid_t id, H5I_type_t type)
{
    void *ret_value = NULL; 

    FUNC_ENTER_API(NULL)

    
    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, NULL, "cannot call public function on library type");
    if (type < 1 || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, NULL, "identifier has invalid type");

    ret_value = H5I_object_verify(id, type);

done:
    FUNC_LEAVE_API(ret_value)
} 

H5I_type_t
H5Iget_type(hid_t id)
{
    H5I_type_t ret_value = H5I_BADID; 

    FUNC_ENTER_API(H5I_BADID)

    ret_value = H5I_get_type(id);

    if (ret_value <= H5I_BADID || (int)ret_value >= H5I_next_type_g || NULL == H5I_object(id))
        HGOTO_DONE(H5I_BADID);

done:
    FUNC_LEAVE_API(ret_value)
} 

void *
H5Iremove_verify(hid_t id, H5I_type_t type)
{
    void *ret_value = NULL; 

    FUNC_ENTER_API(NULL)

    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, NULL, "cannot call public function on library type");

    
    ret_value = H5I__remove_verify(id, type);

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Idec_ref(hid_t id)
{
    int ret_value = 0; 

    FUNC_ENTER_API((-1))

    
    if (id < 0)
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "invalid ID");

    
    if ((ret_value = H5I_dec_app_ref(id)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't decrement ID ref count");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Iinc_ref(hid_t id)
{
    int ret_value = -1; 

    FUNC_ENTER_API((-1))

    
    if (id < 0)
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "invalid ID");

    
    if ((ret_value = H5I_inc_ref(id, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINC, (-1), "can't increment ID ref count");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Iget_ref(hid_t id)
{
    int ret_value = -1; 

    FUNC_ENTER_API((-1))

    
    if (id < 0)
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "invalid ID");

    
    if ((ret_value = H5I_get_ref(id, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTGET, (-1), "can't get ID ref count");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Iinc_type_ref(H5I_type_t type)
{
    int ret_value = -1; 

    FUNC_ENTER_API((-1))

    
    if (type <= 0 || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "invalid ID type");
    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, (-1), "cannot call public function on library type");

    
    if ((ret_value = H5I__inc_type_ref(type)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINC, (-1), "can't increment ID type ref count");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Idec_type_ref(H5I_type_t type)
{
    herr_t ret_value = 0; 

    FUNC_ENTER_API((-1))

    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, (-1), "cannot call public function on library type");

    ret_value = H5I_dec_type_ref(type);

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Iget_type_ref(H5I_type_t type)
{
    int ret_value = -1; 

    FUNC_ENTER_API((-1))

    
    if (type <= 0 || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "invalid ID type");
    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, (-1), "cannot call public function on library type");

    
    if ((ret_value = H5I__get_type_ref(type)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTGET, (-1), "can't get ID type ref count");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Iis_valid(hid_t id)
{
    H5I_id_info_t *info      = NULL; 
    htri_t         ret_value = true; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (info = H5I__find_id(id)))
        ret_value = false;
    else if (!info->app_count) 
        ret_value = false;

done:
    FUNC_LEAVE_API(ret_value)
} 

static int
H5I__search_cb(void *obj, hid_t id, void *_udata)
{
    H5I_search_ud_t *udata      = (H5I_search_ud_t *)_udata; 
    herr_t           cb_ret_val = FAIL;                      
    int              ret_value  = H5_ITER_ERROR;             

    FUNC_ENTER_PACKAGE_NOERR

    
    H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
        {
            cb_ret_val = (*udata->app_cb)(obj, id, udata->app_key);
        }
    H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)

    
    if (cb_ret_val > 0) {
        ret_value      = H5_ITER_STOP; 
        udata->ret_obj = obj;          
    }
    else if (cb_ret_val < 0)
        ret_value = H5_ITER_ERROR; 

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5Isearch(H5I_type_t type, H5I_search_func_t func, void *key)
{
    H5I_search_ud_t udata;            
    void           *ret_value = NULL; 

    FUNC_ENTER_API(NULL)

    
    if (H5I_IS_LIB_TYPE(type))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, NULL, "cannot call public function on library type");

    
    udata.app_cb  = func;
    udata.app_key = key;
    udata.ret_obj = NULL;

    
    (void)H5I_iterate(type, H5I__search_cb, &udata, true);

    
    ret_value = udata.ret_obj;

done:
    FUNC_LEAVE_API(ret_value)
} 

static int
H5I__iterate_pub_cb(void H5_ATTR_UNUSED *obj, hid_t id, void *_udata)
{
    H5I_iterate_pub_ud_t *udata      = (H5I_iterate_pub_ud_t *)_udata; 
    herr_t                cb_ret_val = FAIL;                           
    int                   ret_value  = H5_ITER_ERROR;                  

    FUNC_ENTER_PACKAGE_NOERR

    
    H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
        {
            
            cb_ret_val = (*udata->op)(id, udata->op_data);
        }
    H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)

    
    if (cb_ret_val > 0)
        ret_value = H5_ITER_STOP; 
    else if (cb_ret_val < 0)
        ret_value = H5_ITER_ERROR; 
    else
        ret_value = H5_ITER_CONT; 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Iiterate(H5I_type_t type, H5I_iterate_func_t op, void *op_data)
{
    H5I_iterate_pub_ud_t int_udata;        
    herr_t               ret_value = FAIL; 

    FUNC_ENTER_API(FAIL)

    
    int_udata.op      = op;
    int_udata.op_data = op_data;

    
    if ((ret_value = H5I_iterate(type, H5I__iterate_pub_cb, &int_udata, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_BADITER, FAIL, "can't iterate over ids");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Iget_file_id(hid_t obj_id)
{
    H5I_type_t type;                        
    hid_t      ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    type = H5I_TYPE(obj_id);

    
    if (H5I_FILE == type || H5I_DATATYPE == type || H5I_GROUP == type || H5I_DATASET == type ||
        H5I_ATTR == type) {
        H5VL_object_t *vol_obj; 

        
        if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
            HGOTO_ERROR(H5E_ID, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

        
        if ((ret_value = H5F_get_file_id(vol_obj, type, true)) < 0)
            HGOTO_ERROR(H5E_ID, H5E_CANTGET, H5I_INVALID_HID, "can't retrieve file ID");
    }
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, H5I_INVALID_HID, "not an ID of a file object");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Iget_name(hid_t id, char *name , size_t size)
{
    H5VL_object_t         *vol_obj = NULL; 
    H5VL_object_get_args_t vol_cb_args;    
    H5VL_loc_params_t      loc_params;
    size_t                 obj_name_len = 0;  
    ssize_t                ret_value    = -1; 

    FUNC_ENTER_API((-1))

    
    if (NULL == (vol_obj = H5VL_vol_object(id)))
        HGOTO_ERROR(H5E_ID, H5E_BADTYPE, (-1), "invalid identifier");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(id);

    
    vol_cb_args.op_type                = H5VL_OBJECT_GET_NAME;
    vol_cb_args.args.get_name.buf_size = size;
    vol_cb_args.args.get_name.buf      = name;
    vol_cb_args.args.get_name.name_len = &obj_name_len;

    
    if (H5VL_object_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTGET, (-1), "can't retrieve object name");

    
    ret_value = (ssize_t)obj_name_len;

done:
    FUNC_LEAVE_API(ret_value)
} 
