/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5Lpublic_H
#define H5Lpublic_H

#include "H5public.h"  
#include "H5Ipublic.h" 
#include "H5Opublic.h" 
#include "H5Tpublic.h" 

#define H5L_MAX_LINK_NAME_LEN UINT32_MAX

#define H5L_SAME_LOC 0 

#ifdef __cplusplus
extern "C" {
#endif

typedef enum {
    H5L_TYPE_ERROR    = (-1), 
    H5L_TYPE_HARD     = 0,    
    H5L_TYPE_SOFT     = 1,    
    H5L_TYPE_EXTERNAL = 64,   
    H5L_TYPE_MAX      = 255   
} H5L_type_t;

#define H5L_TYPE_BUILTIN_MAX H5L_TYPE_SOFT

#define H5L_TYPE_UD_MIN H5L_TYPE_EXTERNAL

#define H5L_TYPE_UD_MAX H5L_TYPE_MAX

//! <!-- [H5L_info2_t_snip] -->
typedef struct {
    H5L_type_t type;         
    bool       corder_valid; 
    int64_t    corder;       
    H5T_cset_t cset;         
    union {
        H5O_token_t token;    
        size_t      val_size; 
    } u;
} H5L_info2_t;
//! <!-- [H5L_info2_t_snip] -->

//! <!-- [H5L_iterate2_t_snip] -->
typedef herr_t (*H5L_iterate2_t)(hid_t group, const char *name, const H5L_info2_t *info, void *op_data);
//! <!-- [H5L_iterate2_t_snip] -->

typedef herr_t (*H5L_elink_traverse_t)(const char *parent_file_name, const char *parent_group_name,
                                       const char *child_file_name, const char *child_object_name,
                                       unsigned *acc_flags, hid_t fapl_id, void *op_data);

H5_DLL herr_t H5Lmove(hid_t src_loc, const char *src_name, hid_t dst_loc, const char *dst_name, hid_t lcpl_id,
                      hid_t lapl_id);

H5_DLL herr_t H5Lcopy(hid_t src_loc, const char *src_name, hid_t dst_loc, const char *dst_name, hid_t lcpl_id,
                      hid_t lapl_id);

H5_DLL herr_t H5Lcreate_hard(hid_t cur_loc, const char *cur_name, hid_t dst_loc, const char *dst_name,
                             hid_t lcpl_id, hid_t lapl_id);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Lcreate_hard_async(const char *app_file, const char *app_func, unsigned app_line,
                                   hid_t cur_loc_id, const char *cur_name, hid_t new_loc_id,
                                   const char *new_name, hid_t lcpl_id, hid_t lapl_id, hid_t es_id);
#else
H5_DLL herr_t H5Lcreate_hard_async(hid_t cur_loc_id, const char *cur_name, hid_t new_loc_id,
                                   const char *new_name, hid_t lcpl_id, hid_t lapl_id, hid_t es_id);
#endif

H5_DLL herr_t H5Lcreate_soft(const char *link_target, hid_t link_loc_id, const char *link_name, hid_t lcpl_id,
                             hid_t lapl_id);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Lcreate_soft_async(const char *app_file, const char *app_func, unsigned app_line,
                                   const char *link_target, hid_t link_loc_id, const char *link_name,
                                   hid_t lcpl_id, hid_t lapl_id, hid_t es_id);
#else
H5_DLL herr_t H5Lcreate_soft_async(const char *link_target, hid_t link_loc_id, const char *link_name,
                                   hid_t lcpl_id, hid_t lapl_id, hid_t es_id);
#endif

H5_DLL herr_t H5Ldelete(hid_t loc_id, const char *name, hid_t lapl_id);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Ldelete_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                              const char *name, hid_t lapl_id, hid_t es_id);
#else
H5_DLL herr_t H5Ldelete_async(hid_t loc_id, const char *name, hid_t lapl_id, hid_t es_id);
#endif

H5_DLL herr_t H5Ldelete_by_idx(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                               H5_iter_order_t order, hsize_t n, hid_t lapl_id);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Ldelete_by_idx_async(const char *app_file, const char *app_func, unsigned app_line,
                                     hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                     H5_iter_order_t order, hsize_t n, hid_t lapl_id, hid_t es_id);
#else
H5_DLL herr_t H5Ldelete_by_idx_async(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                     H5_iter_order_t order, hsize_t n, hid_t lapl_id, hid_t es_id);
#endif

H5_DLL herr_t H5Lget_val(hid_t loc_id, const char *name, void *buf , size_t size, hid_t lapl_id);

H5_DLL herr_t H5Lget_val_by_idx(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                H5_iter_order_t order, hsize_t n, void *buf , size_t size,
                                hid_t lapl_id);

H5_DLL htri_t H5Lexists(hid_t loc_id, const char *name, hid_t lapl_id);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Lexists_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                              const char *name, bool *exists, hid_t lapl_id, hid_t es_id);
#else
H5_DLL herr_t H5Lexists_async(hid_t loc_id, const char *name, bool *exists, hid_t lapl_id, hid_t es_id);
#endif

H5_DLL herr_t H5Lget_info2(hid_t loc_id, const char *name, H5L_info2_t *linfo, hid_t lapl_id);

H5_DLL herr_t H5Lget_info_by_idx2(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                  H5_iter_order_t order, hsize_t n, H5L_info2_t *linfo, hid_t lapl_id);

H5_DLL ssize_t H5Lget_name_by_idx(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                  H5_iter_order_t order, hsize_t n, char *name , size_t size,
                                  hid_t lapl_id);

H5_DLL herr_t H5Literate2(hid_t grp_id, H5_index_t idx_type, H5_iter_order_t order, hsize_t *idx,
                          H5L_iterate2_t op, void *op_data);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Literate_async(const char *app_file, const char *app_func, unsigned app_line, hid_t group_id,
                               H5_index_t idx_type, H5_iter_order_t order, hsize_t *idx_p, H5L_iterate2_t op,
                               void *op_data, hid_t es_id);
#else
H5_DLL herr_t H5Literate_async(hid_t group_id, H5_index_t idx_type, H5_iter_order_t order, hsize_t *idx_p,
                               H5L_iterate2_t op, void *op_data, hid_t es_id);
#endif

H5_DLL herr_t H5Literate_by_name2(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                  H5_iter_order_t order, hsize_t *idx, H5L_iterate2_t op, void *op_data,
                                  hid_t lapl_id);

H5_DLL herr_t H5Lvisit2(hid_t grp_id, H5_index_t idx_type, H5_iter_order_t order, H5L_iterate2_t op,
                        void *op_data);

H5_DLL herr_t H5Lvisit_by_name2(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                H5_iter_order_t order, H5L_iterate2_t op, void *op_data, hid_t lapl_id);

H5_DLL herr_t H5Lcreate_ud(hid_t link_loc_id, const char *link_name, H5L_type_t link_type, const void *udata,
                           size_t udata_size, hid_t lcpl_id, hid_t lapl_id);

H5_DLL htri_t H5Lis_registered(H5L_type_t id);

H5_DLL herr_t H5Lunpack_elink_val(const void *ext_linkval , size_t link_size, unsigned *flags,
                                  const char **filename , const char **obj_path );

H5_DLL herr_t H5Lcreate_external(const char *file_name, const char *obj_name, hid_t link_loc_id,
                                 const char *link_name, hid_t lcpl_id, hid_t lapl_id);

/// \cond DEV

#ifndef H5L_MODULE
#define H5Lcreate_hard_async(...)   H5Lcreate_hard_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Lcreate_soft_async(...)   H5Lcreate_soft_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Ldelete_async(...)        H5Ldelete_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Ldelete_by_idx_async(...) H5Ldelete_by_idx_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Lexists_async(...)        H5Lexists_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Literate_async(...)       H5Literate_async(__FILE__, __func__, __LINE__, __VA_ARGS__)

#define H5Lcreate_hard_async_wrap   H5_NO_EXPAND(H5Lcreate_hard_async)
#define H5Lcreate_soft_async_wrap   H5_NO_EXPAND(H5Lcreate_soft_async)
#define H5Ldelete_async_wrap        H5_NO_EXPAND(H5Ldelete_async)
#define H5Ldelete_by_idx_async_wrap H5_NO_EXPAND(H5Ldelete_by_idx_async)
#define H5Lexists_async_wrap        H5_NO_EXPAND(H5Lexists_async)
#define H5Literate_async_wrap       H5_NO_EXPAND(H5Literate_async)
#endif 
/// \endcond

#ifndef H5_NO_DEPRECATED_SYMBOLS

//! <!-- [H5L_info1_t_snip] -->

typedef struct {
    H5L_type_t type;         
    bool       corder_valid; 
    int64_t    corder;       
    H5T_cset_t cset;         
    union {
        haddr_t address;  
        size_t  val_size; 
    } u;
} H5L_info1_t;
//! <!-- [H5L_info1_t_snip] -->

//! <!-- [H5L_iterate1_t_snip] -->
typedef herr_t (*H5L_iterate1_t)(hid_t group, const char *name, const H5L_info1_t *info, void *op_data);
//! <!-- [H5L_iterate1_t_snip] -->

H5_DLL herr_t H5Lget_info1(hid_t loc_id, const char *name, H5L_info1_t *linfo , hid_t lapl_id);

H5_DLL herr_t H5Lget_info_by_idx1(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                  H5_iter_order_t order, hsize_t n, H5L_info1_t *linfo ,
                                  hid_t lapl_id);

H5_DLL herr_t H5Literate1(hid_t grp_id, H5_index_t idx_type, H5_iter_order_t order, hsize_t *idx,
                          H5L_iterate1_t op, void *op_data);

H5_DLL herr_t H5Literate_by_name1(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                  H5_iter_order_t order, hsize_t *idx, H5L_iterate1_t op, void *op_data,
                                  hid_t lapl_id);

H5_DLL herr_t H5Lvisit1(hid_t grp_id, H5_index_t idx_type, H5_iter_order_t order, H5L_iterate1_t op,
                        void *op_data);

H5_DLL herr_t H5Lvisit_by_name1(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                H5_iter_order_t order, H5L_iterate1_t op, void *op_data, hid_t lapl_id);

#endif 

#ifdef __cplusplus
}
#endif
#endif 
