/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Iprivate.h"  
#include "H5Ppkg.h"      

static herr_t H5P__get_file_space(H5P_genplist_t *plist, H5F_file_space_type_t *strategy, hsize_t *threshold);

#ifndef H5_NO_DEPRECATED_SYMBOLS

herr_t
H5Pregister1(hid_t cls_id, const char *name, size_t size, void *def_value, H5P_prp_create_func_t prp_create,
             H5P_prp_set_func_t prp_set, H5P_prp_get_func_t prp_get, H5P_prp_delete_func_t prp_delete,
             H5P_prp_copy_func_t prp_copy, H5P_prp_close_func_t prp_close)
{
    H5P_genclass_t *pclass;      
    H5P_genclass_t *orig_pclass; 
    herr_t          ret_value;   

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(cls_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list class");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid class name");
    if (size > 0 && def_value == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "properties >0 size must have default");

    
    orig_pclass = pclass;
    if ((ret_value = H5P__register(&pclass, name, size, def_value, prp_create, prp_set, prp_get, NULL, NULL,
                                   prp_delete, prp_copy, NULL, prp_close)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to register property in class");

    
    if (pclass != orig_pclass) {
        H5P_genclass_t *old_pclass; 

        
        if (NULL == (old_pclass = (H5P_genclass_t *)H5I_subst(cls_id, pclass)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to substitute property class in ID");
        assert(old_pclass == orig_pclass);

        
        if (H5P__close_class(orig_pclass) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL,
                        "unable to close original property class after substitution");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pinsert1(hid_t plist_id, const char *name, size_t size, void *value, H5P_prp_set_func_t prp_set,
           H5P_prp_get_func_t prp_get, H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy,
           H5P_prp_close_func_t prp_close)
{
    H5P_genplist_t *plist;     
    herr_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid property name");
    if (size > 0 && value == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "properties >0 size must have default");

    
    if ((ret_value = H5P_insert(plist, name, size, value, prp_set, prp_get, NULL, NULL, prp_delete, prp_copy,
                                NULL, prp_close)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to register property in plist");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_version(hid_t plist_id, unsigned *super , unsigned *freelist , unsigned *stab ,
               unsigned *shhdr )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_FILE_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (super)
        if (H5P_get(plist, H5F_CRT_SUPER_VERS_NAME, super) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get superblock version");
    if (freelist)
        *freelist = HDF5_FREESPACE_VERSION; 
    if (stab)
        *stab = HDF5_OBJECTDIR_VERSION; 
    if (shhdr)
        *shhdr = HDF5_SHAREDHEADER_VERSION; 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pencode1(hid_t plist_id, void *buf, size_t *nalloc)
{
    H5P_genplist_t *plist; 
    hid_t           temp_fapl_id = H5P_DEFAULT;
    herr_t          ret_value    = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

    
    if (H5CX_set_apl(&temp_fapl_id, H5P_CLS_FACC, H5I_INVALID_HID, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, H5I_INVALID_HID, "can't set access property list info");

    
    if ((ret_value = H5P__encode(plist, true, buf, nalloc)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "unable to encode property list");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_file_space(hid_t plist_id, H5F_file_space_type_t strategy, hsize_t threshold)
{

    H5P_genplist_t       *plist;                                        
    H5F_fspace_strategy_t new_strategy;                                 
    bool                  new_persist   = H5F_FREE_SPACE_PERSIST_DEF;   
    hsize_t               new_threshold = H5F_FREE_SPACE_THRESHOLD_DEF; 
    H5F_file_space_type_t in_strategy   = strategy;                     
    hsize_t               in_threshold  = threshold;                    
    herr_t                ret_value     = SUCCEED;                      

    FUNC_ENTER_API(FAIL)

    
    if ((unsigned)in_strategy >= H5F_FILE_SPACE_NTYPES)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid strategy");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_FILE_CREATE, false)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADID, FAIL, "can't find object for ID");

    
    if (!in_strategy)
        if (H5P__get_file_space(plist, &in_strategy, NULL) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get file space strategy");
    if (!in_threshold)
        if (H5P__get_file_space(plist, NULL, &in_threshold) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get free-space threshold");

    switch (in_strategy) {
        case H5F_FILE_SPACE_ALL_PERSIST:
            new_strategy  = H5F_FSPACE_STRATEGY_FSM_AGGR;
            new_persist   = true;
            new_threshold = in_threshold;
            break;

        case H5F_FILE_SPACE_ALL:
            new_strategy  = H5F_FSPACE_STRATEGY_FSM_AGGR;
            new_threshold = in_threshold;
            break;

        case H5F_FILE_SPACE_AGGR_VFD:
            new_strategy = H5F_FSPACE_STRATEGY_AGGR;
            break;

        case H5F_FILE_SPACE_VFD:
            new_strategy = H5F_FSPACE_STRATEGY_NONE;
            break;

        case H5F_FILE_SPACE_NTYPES:
        case H5F_FILE_SPACE_DEFAULT:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid file space strategy");
    }

    if (H5P__set_file_space_strategy(plist, new_strategy, new_persist, new_threshold) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set file space strategy");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__get_file_space(H5P_genplist_t *plist, H5F_file_space_type_t *strategy, hsize_t *threshold)
{
    H5F_fspace_strategy_t new_strategy;        
    bool                  new_persist;         
    hsize_t               new_threshold;       
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__get_file_space_strategy(plist, &new_strategy, &new_persist, &new_threshold) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get file space strategy values");

    
    if (strategy) {
        switch (new_strategy) {

            case H5F_FSPACE_STRATEGY_FSM_AGGR:
                if (new_persist)
                    *strategy = H5F_FILE_SPACE_ALL_PERSIST;
                else
                    *strategy = H5F_FILE_SPACE_ALL;
                break;

            case H5F_FSPACE_STRATEGY_AGGR:
                *strategy = H5F_FILE_SPACE_AGGR_VFD;
                break;

            case H5F_FSPACE_STRATEGY_NONE:
                *strategy = H5F_FILE_SPACE_VFD;
                break;

            case H5F_FSPACE_STRATEGY_PAGE:
            case H5F_FSPACE_STRATEGY_NTYPES:
            default:
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid file space strategy");
        }
    }

    if (threshold)
        *threshold = new_threshold;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pget_file_space(hid_t plist_id, H5F_file_space_type_t *strategy , hsize_t *threshold )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_FILE_CREATE, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P__get_file_space(plist, strategy, threshold) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get file space strategy");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 
