/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Fprivate.h" 
#include "H5Ppkg.h"     

#define H5P_STRCRT_CHAR_ENCODING_SIZE sizeof(H5T_cset_t)
#define H5P_STRCRT_CHAR_ENCODING_DEF  H5F_DEFAULT_CSET
#define H5P_STRCRT_CHAR_ENCODING_ENC  H5P__strcrt_char_encoding_enc
#define H5P_STRCRT_CHAR_ENCODING_DEC  H5P__strcrt_char_encoding_dec

static herr_t H5P__strcrt_reg_prop(H5P_genclass_t *pclass);

static herr_t H5P__strcrt_char_encoding_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__strcrt_char_encoding_dec(const void **_pp, void *value);

const H5P_libclass_t H5P_CLS_STRCRT[1] = {{
    "string create",        
    H5P_TYPE_STRING_CREATE, 

    &H5P_CLS_ROOT_g,             
    &H5P_CLS_STRING_CREATE_g,    
    &H5P_CLS_STRING_CREATE_ID_g, 
    NULL,                        
    H5P__strcrt_reg_prop,        

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static const H5T_cset_t H5P_def_char_encoding_g =
    H5P_STRCRT_CHAR_ENCODING_DEF; 

static herr_t
H5P__strcrt_reg_prop(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5P_STRCRT_CHAR_ENCODING_NAME, H5P_STRCRT_CHAR_ENCODING_SIZE,
                           &H5P_def_char_encoding_g, NULL, NULL, NULL, H5P_STRCRT_CHAR_ENCODING_ENC,
                           H5P_STRCRT_CHAR_ENCODING_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pset_char_encoding(hid_t plist_id, H5T_cset_t encoding)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (encoding <= H5T_CSET_ERROR || encoding >= H5T_NCSET)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "character encoding is not valid");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_STRING_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5P_STRCRT_CHAR_ENCODING_NAME, &encoding) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set character encoding");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_char_encoding(hid_t plist_id, H5T_cset_t *encoding )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_STRING_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (encoding)
        if (H5P_get(plist, H5P_STRCRT_CHAR_ENCODING_NAME, encoding) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get character encoding flag");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__strcrt_char_encoding_enc(const void *value, void **_pp, size_t *size)
{
    const H5T_cset_t *encoding = (const H5T_cset_t *)value; 
    uint8_t         **pp       = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(encoding);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*encoding;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__strcrt_char_encoding_dec(const void **_pp, void *_value)
{
    H5T_cset_t     *encoding = (H5T_cset_t *)_value; 
    const uint8_t **pp       = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(encoding);

    
    *encoding = (H5T_cset_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
