/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5SMmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5SMpkg.h"     

static herr_t H5SM__cache_table_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5SM__cache_table_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5SM__cache_table_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5SM__cache_table_image_len(const void *thing, size_t *image_len);
static herr_t H5SM__cache_table_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5SM__cache_table_free_icr(void *thing);

static herr_t H5SM__cache_list_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5SM__cache_list_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5SM__cache_list_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5SM__cache_list_image_len(const void *thing, size_t *image_len);
static herr_t H5SM__cache_list_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5SM__cache_list_free_icr(void *thing);

const H5AC_class_t H5AC_SOHM_TABLE[1] = {{
    H5AC_SOHM_TABLE_ID,                      
    "shared message table",                  
    H5FD_MEM_SOHM_TABLE,                     
    H5AC__CLASS_NO_FLAGS_SET,                
    H5SM__cache_table_get_initial_load_size, 
    NULL,                                    
    H5SM__cache_table_verify_chksum,         
    H5SM__cache_table_deserialize,           
    H5SM__cache_table_image_len,             
    NULL,                                    
    H5SM__cache_table_serialize,             
    NULL,                                    
    H5SM__cache_table_free_icr,              
    NULL,                                    
}};

const H5AC_class_t H5AC_SOHM_LIST[1] = {{
    H5AC_SOHM_LIST_ID,                      
    "shared message list",                  
    H5FD_MEM_SOHM_TABLE,                    
    H5AC__CLASS_NO_FLAGS_SET,               
    H5SM__cache_list_get_initial_load_size, 
    NULL,                                   
    H5SM__cache_list_verify_chksum,         
    H5SM__cache_list_deserialize,           
    H5SM__cache_list_image_len,             
    NULL,                                   
    H5SM__cache_list_serialize,             
    NULL,                                   
    H5SM__cache_list_free_icr,              
    NULL,                                   
}};

static herr_t
H5SM__cache_table_get_initial_load_size(void *_udata, size_t *image_len)
{
    const H5SM_table_cache_ud_t *udata = (const H5SM_table_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->f);
    assert(image_len);

    
    *image_len = H5SM_TABLE_SIZE(udata->f);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

htri_t
H5SM__cache_table_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;                

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_SOHM, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5SM__cache_table_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                              bool H5_ATTR_UNUSED *dirty)
{
    H5F_t                 *f;            
    H5SM_master_table_t   *table = NULL; 
    H5SM_table_cache_ud_t *udata = (H5SM_table_cache_ud_t *)_udata; 
    const uint8_t         *image = (const uint8_t *)_image;         
    uint32_t               stored_chksum;                           
    size_t                 u;                                       
    void                  *ret_value = NULL;                        

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(len > 0);
    assert(udata);
    assert(udata->f);
    f = udata->f;
    assert(dirty);

    
    assert(H5F_SOHM_VERS(f) == HDF5_SHAREDHEADER_VERSION);

    
    if (NULL == (table = H5FL_CALLOC(H5SM_master_table_t)))
        HGOTO_ERROR(H5E_SOHM, H5E_NOSPACE, NULL, "memory allocation failed");

    
    table->num_indexes = H5F_SOHM_NINDEXES(f);
    assert(table->num_indexes > 0);

    
    table->table_size = H5SM_TABLE_SIZE(f);
    assert(table->table_size == len);

    
    if (memcmp(image, H5SM_TABLE_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_SOHM, H5E_CANTLOAD, NULL, "bad SOHM table signature");
    image += H5_SIZEOF_MAGIC;

    
    if (NULL == (table->indexes =
                     (H5SM_index_header_t *)H5FL_ARR_MALLOC(H5SM_index_header_t, (size_t)table->num_indexes)))
        HGOTO_ERROR(H5E_SOHM, H5E_NOSPACE, NULL, "memory allocation failed for SOHM indexes");

    
    for (u = 0; u < table->num_indexes; ++u) {
        
        if (H5SM_LIST_VERSION != *image++)
            HGOTO_ERROR(H5E_SOHM, H5E_VERSION, NULL, "bad shared message list version number");

        
        table->indexes[u].index_type = (H5SM_index_type_t)*image++;

        
        UINT16DECODE(image, table->indexes[u].mesg_types);

        
        UINT32DECODE(image, table->indexes[u].min_mesg_size);

        
        UINT16DECODE(image, table->indexes[u].list_max);

        
        UINT16DECODE(image, table->indexes[u].btree_min);

        
        UINT16DECODE(image, table->indexes[u].num_messages);

        
        H5F_addr_decode(f, &image, &(table->indexes[u].index_addr));

        
        H5F_addr_decode(f, &image, &(table->indexes[u].heap_addr));

        
        table->indexes[u].list_size = H5SM_LIST_SIZE(f, table->indexes[u].list_max);
    } 

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == table->table_size);

    
    ret_value = table;

done:
    if (!ret_value && table)
        if (H5SM__table_free(table) < 0)
            HDONE_ERROR(H5E_SOHM, H5E_CANTFREE, NULL, "unable to destroy sohm table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5SM__cache_table_image_len(const void *_thing, size_t *image_len)
{
    const H5SM_master_table_t *table =
        (const H5SM_master_table_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(table);
    assert(table->cache_info.type == H5AC_SOHM_TABLE);
    assert(image_len);

    *image_len = table->table_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5SM__cache_table_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_thing)
{
    H5SM_master_table_t *table = (H5SM_master_table_t *)_thing; 
    uint8_t             *image = (uint8_t *)_image;             
    uint32_t             computed_chksum;                       
    size_t               u;                                     

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(table);
    assert(table->cache_info.type == H5AC_SOHM_TABLE);
    assert(table->table_size == len);

    
    assert(H5F_SOHM_VERS(f) == HDF5_SHAREDHEADER_VERSION);

    
    H5MM_memcpy(image, H5SM_TABLE_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    for (u = 0; u < table->num_indexes; ++u) {
        
        *image++ = H5SM_LIST_VERSION;

        
        *image++ = (uint8_t)table->indexes[u].index_type;

        
        UINT16ENCODE(image, table->indexes[u].mesg_types);

        
        UINT32ENCODE(image, table->indexes[u].min_mesg_size);

        
        UINT16ENCODE(image, table->indexes[u].list_max);

        
        UINT16ENCODE(image, table->indexes[u].btree_min);

        
        UINT16ENCODE(image, table->indexes[u].num_messages);

        
        H5F_addr_encode(f, &image, table->indexes[u].index_addr);

        
        H5F_addr_encode(f, &image, table->indexes[u].heap_addr);
    } 

    
    computed_chksum = H5_checksum_metadata(_image, (table->table_size - H5SM_SIZEOF_CHECKSUM), 0);
    UINT32ENCODE(image, computed_chksum);

    
    assert((size_t)(image - ((uint8_t *)_image)) == table->table_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5SM__cache_table_free_icr(void *_thing)
{
    H5SM_master_table_t *table     = (H5SM_master_table_t *)_thing; 
    herr_t               ret_value = SUCCEED;                       

    FUNC_ENTER_PACKAGE

    
    assert(table);
    assert(table->cache_info.type == H5AC_SOHM_TABLE);

    
    if (H5SM__table_free(table) < 0)
        HGOTO_ERROR(H5E_SOHM, H5E_CANTRELEASE, FAIL, "unable to free shared message table");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5SM__cache_list_get_initial_load_size(void *_udata, size_t *image_len)
{
    const H5SM_list_cache_ud_t *udata = (const H5SM_list_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->header);
    assert(udata->header->list_size > 0);
    assert(image_len);

    
    *image_len = udata->header->list_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

htri_t
H5SM__cache_list_verify_chksum(const void *_image, size_t H5_ATTR_UNUSED len, void *_udata)
{
    const uint8_t        *image = (const uint8_t *)_image;        
    H5SM_list_cache_ud_t *udata = (H5SM_list_cache_ud_t *)_udata; 
    size_t                chk_size;         
    uint32_t              stored_chksum;    
    uint32_t              computed_chksum;  
    htri_t                ret_value = true; 

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    chk_size = H5SM_LIST_SIZE(udata->f, udata->header->num_messages);

    
    if (H5F_get_checksums(image, chk_size, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_SOHM, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5SM__cache_list_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                             bool H5_ATTR_UNUSED *dirty)
{
    H5SM_list_t          *list  = NULL;                           
    H5SM_list_cache_ud_t *udata = (H5SM_list_cache_ud_t *)_udata; 
    H5SM_bt2_ctx_t        ctx;                                    
    const uint8_t        *image = (const uint8_t *)_image;        
    uint32_t              stored_chksum;                          
    size_t                u;                                      
    void                 *ret_value = NULL;                       

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(len > 0);
    assert(udata);
    assert(udata->header);
    assert(udata->header->list_size == len);
    assert(dirty);

    
    if (NULL == (list = H5FL_MALLOC(H5SM_list_t)))
        HGOTO_ERROR(H5E_SOHM, H5E_NOSPACE, NULL, "memory allocation failed");
    memset(&list->cache_info, 0, sizeof(H5AC_info_t));

    
    if (NULL == (list->messages = (H5SM_sohm_t *)H5FL_ARR_MALLOC(H5SM_sohm_t, udata->header->list_max)))
        HGOTO_ERROR(H5E_SOHM, H5E_NOSPACE, NULL, "file allocation failed for SOHM list");
    list->header = udata->header;

    
    if (memcmp(image, H5SM_LIST_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_SOHM, H5E_CANTLOAD, NULL, "bad SOHM list signature");
    image += H5_SIZEOF_MAGIC;

    
    ctx.sizeof_addr = H5F_SIZEOF_ADDR(udata->f);
    for (u = 0; u < udata->header->num_messages; u++) {
        if (H5SM__message_decode(image, &(list->messages[u]), &ctx) < 0)
            HGOTO_ERROR(H5E_SOHM, H5E_CANTLOAD, NULL, "can't decode shared message");

        image += H5SM_SOHM_ENTRY_SIZE(udata->f);
    } 

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) <= udata->header->list_size);

    
    for (u = udata->header->num_messages; u < udata->header->list_max; u++)
        list->messages[u].location = H5SM_NO_LOC;

    
    ret_value = list;

done:
    if (!ret_value && list) {
        if (list->messages)
            list->messages = H5FL_ARR_FREE(H5SM_sohm_t, list->messages);
        list = H5FL_FREE(H5SM_list_t, list);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5SM__cache_list_image_len(const void *_thing, size_t *image_len)
{
    const H5SM_list_t *list = (const H5SM_list_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(list);
    assert(list->cache_info.type == H5AC_SOHM_LIST);
    assert(list->header);
    assert(image_len);

    *image_len = list->header->list_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5SM__cache_list_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_thing)
{
    H5SM_list_t   *list = (H5SM_list_t *)_thing; 
    H5SM_bt2_ctx_t ctx;                          
    uint8_t       *image = (uint8_t *)_image;    
    uint32_t       computed_chksum;              
    size_t         mesgs_serialized;             
    size_t         u;                            
    herr_t         ret_value = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(list);
    assert(list->cache_info.type == H5AC_SOHM_LIST);
    assert(list->header);
    assert(list->header->list_size == len);

    
    H5MM_memcpy(image, H5SM_LIST_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    mesgs_serialized = 0;
    ctx.sizeof_addr  = H5F_SIZEOF_ADDR(f);
    for (u = 0; ((u < list->header->list_max) && (mesgs_serialized < list->header->num_messages)); u++) {
        if (list->messages[u].location != H5SM_NO_LOC) {
            if (H5SM__message_encode(image, &(list->messages[u]), &ctx) < 0)
                HGOTO_ERROR(H5E_SOHM, H5E_CANTFLUSH, FAIL, "unable to serialize shared message");

            image += H5SM_SOHM_ENTRY_SIZE(f);
            ++mesgs_serialized;
        } 
    }     

    assert(mesgs_serialized == list->header->num_messages);

    
    computed_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);
    UINT32ENCODE(image, computed_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) <= list->header->list_size);

    
    memset(image, 0, (list->header->list_size - (size_t)(image - (uint8_t *)_image)));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5SM__cache_list_free_icr(void *_thing)
{
    H5SM_list_t *list      = (H5SM_list_t *)_thing; 
    herr_t       ret_value = SUCCEED;               

    FUNC_ENTER_PACKAGE

    
    assert(list);
    assert(list->cache_info.type == H5AC_SOHM_LIST);

    
    if (H5SM__list_free(list) < 0)
        HGOTO_ERROR(H5E_SOHM, H5E_CANTRELEASE, FAIL, "unable to free shared message list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
