/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5module.h" 

#include "H5private.h" 

herr_t
H5_buffer_dump(FILE *stream, int indent, const uint8_t *buf, const uint8_t *marker, size_t buf_offset,
               size_t buf_size)
{
    size_t u, v; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(stream);
    assert(indent >= 0);
    assert(buf);
    assert(marker);
    assert(buf_size > 0);

    
    Rfprintf(stream, "%*sData follows (`__' indicates free region)...\n", indent, "");
    for (u = 0; u < buf_size; u += 16) {
        uint8_t c;

        Rfprintf(stream, "%*s %8zu: ", indent, "", u + buf_offset);

        
        for (v = 0; v < 16; v++) {
            if (u + v < buf_size) {
                if (marker[u + v])
                    Rfprintf(stream, "__ ");
                else {
                    c = buf[buf_offset + u + v];
                    Rfprintf(stream, "%02x ", c);
                } 
            }     
            else
                Rfprintf(stream, "   ");

            if (7 == v)
                fputc(' ', stream);
        } 
        fputc(' ', stream);

        
        for (v = 0; v < 16; v++) {
            if (u + v < buf_size) {
                if (marker[u + v])
                    fputc(' ', stream);
                else {
                    c = buf[buf_offset + u + v];

                    if (isprint(c))
                        fputc(c, stream);
                    else
                        fputc('.', stream);
                } 
            }     

            if (7 == v)
                fputc(' ', stream);
        } 

        fputc('\n', stream);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
