% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_ieegio_transform.R
\name{io_read_ants_transform}
\alias{io_read_ants_transform}
\title{Read \code{ANTs} transform file}
\usage{
io_read_ants_transform(
  file,
  space_from,
  space_to,
  interpretation = c("passive", "active")
)
}
\arguments{
\item{file}{character string specifying the path to the transform file.
Supported formats include:
\itemize{
\item \code{.mat}: \code{ITK}/\code{ANTs} affine transform (4x4 matrix)
\item \code{.h5}: HDF5 composite transform (may contain affine and/or
deformation components)
\item \code{.nii}, \code{.nii.gz}: Deformation field images
}}

\item{space_from}{character string or \code{ieegio_space} object identifying
the source space. If missing, will be inferred from the filename using
BIDS-style \code{from-<space>} entity (e.g.,
\code{"sub-01_from-T1w_to-MNI_xfm.h5"} yields \code{"T1w"}).}

\item{space_to}{character string or \code{ieegio_space} object identifying
the target space. If missing, will be inferred from the filename using
BIDS-style \code{to-<space>} entity.}

\item{interpretation}{character string specifying how to interpret the
transform:
\itemize{
\item \code{"passive"} (default): Axis/coordinate frame transform.
Represents how coordinate systems relate to each other. This is the
typical interpretation for brain imaging registration transforms.
\item \code{"active"}: Point transform. Directly transforms point
coordinates from source to target space.
}}
}
\value{
An \code{ieegio_transforms} object with:
\describe{
\item{data}{List containing the transform data (matrix for affine,
\code{ANTsTransform} object for deformation)}
\item{type}{\code{"affine"} or \code{"deformation"}}
\item{interpretation}{\code{"active"} or \code{"passive"}}
\item{space_from}{Source space (with \code{"LPS"} orientation for \code{ANTs})}
\item{space_to}{Target space (with \code{"LPS"} orientation for \code{ANTs})}
\item{dimension}{Spatial dimension (typically 3)}
}
}
\description{
Reads spatial transformation files in \code{ANTs} (Advanced Normalization Tools)
format, including affine matrices (\code{.mat}) and deformation fields
(\code{.h5}, \code{.nii.gz}).
}
\details{
\code{ANTs} transforms operate in \code{LPS} (Left-Posterior-Superior) coordinate
convention. The returned transform object automatically sets orientation
to \code{"LPS"} for both source and target spaces.

For composite transforms (e.g., \code{.h5} files containing both affine
and deformation components), the function returns a single transform object.
Use \code{\link{as_ieegio_transform}} with a list to combine multiple transforms.

This function requires the \code{rpyANTs} package and a configured Python
environment.
}
\section{BIDS Support}{

The function can automatically infer space names from BIDS-compliant
file names:
\itemize{
\item \code{from-<source>}: Source space identifier
\item \code{to-<target>}: Target space identifier
}
Example: \code{"sub-01_from-T1w_to-MNI152NLin2009cAsym_mode-image_xfm.h5"}
}

\examples{
\dontrun{
# Read an affine transform
xfm <- io_read_ants_transform("sub-01_from-T1w_to-MNI_xfm.mat")

# Explicitly specify spaces
xfm <- io_read_ants_transform(
  "transform.h5",
  space_from = "native",
  space_to = "MNI152"
)

# Read as active (point) transform
xfm <- io_read_ants_transform(
  "transform.mat",
  interpretation = "active"
)

# Chain multiple transforms
xfm1 <- io_read_ants_transform("from-T1w_to-T2w_xfm.mat")
xfm2 <- io_read_ants_transform("from-T2w_to-MNI_xfm.h5")
combined <- as_ieegio_transform(list(xfm1, xfm2))
}

}
\seealso{
\code{\link{as_ieegio_transform}} for converting objects to transforms and chaining
\code{\link{transform_orientation}} for orientation conversion transforms
}
