% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter_nfi.R
\name{filter_nfi}
\alias{filter_nfi}
\title{Filter National Forest Inventory Data}
\usage{
filter_nfi(data, expr_texts, hier = TRUE)
}
\arguments{
\item{data}{: A \code{list} generated by \code{\link{read_nfi}}. Each dataframe should have a 'SUB_PLOT' column.}

\item{expr_texts}{: @param expr_texts : A character vector; expressions specifying filtering conditions. Each expression should combine the dataframe name, dollar sign, and condition, with separate expressions for each data frame. (e.g., c("plot$OWN_CD == '5'", "tree$FAMILY == 'Pinaceae'"). Conditions must be valid R expressions.}

\item{hier}{: A logical flag (default TRUE); indicates whether to apply hierarchical filtering (\code{TRUE}) or non-hierarchical filtering (\code{FALSE}). Hierarchical filtering ensures that connected dataframes are filtered based on the results of filters applied to the parent frame.}
}
\value{
A \code{list} of dataframes.
}
\description{
The \code{filter_nfi()} function provides hierarchical and non-hierarchical filtering approaches for the complex structure of National Forest Inventory data based on user-provided condition expressions (\code{expr_texts}).
This function enables effective filtering while maintaining the relationship between plot data (parent data) and other data (child data).
}
\details{
This function parses expressions targeting specific columns in the dataframes within the provided list.

Expression requirements:
\itemize{
\item Each expression in \code{expr_texts} must start with a valid dataframe name in the list (e.g., "plot", "tree", "cwd") and combine the dataframe name, dollar sign, and condition (e.g. c("plot$OWN_CD == '5'").
\item Separate expressions must be provided for each dataframe being filtered (e.g. c("plot$OWN_CD == '5'", "tree$FAMILY == 'Pinaceae' | tree$WDY_PLNTS_TYP_CD == '1'").
}

Hierarchical filtering (\code{hier = TRUE}):
\itemize{
\item Filters applied to plot table affect all connected child data (tree, CWD, stump, etc.).
\item Filters applied to child data only operate within that dataframe and do not affect other dataframes.
\item Example: If only coniferous forest subplots are selected in the plot table, the child data will retain only the tree, CWD, stump, etc., associated with those subplots.
}

Non-hierarchical filtering (\code{hier = FALSE}):
\itemize{
\item Filters applied to the parent dataframe (plot table) do not affect the child data.
\item Filtering results from child data affect all other parent and child data.
\item Example: If only certain species are selected in the tree table, the plot table, CWD table, stump table, etc., will be filtered based on the remaining subplots from this selection.
}
}
\examples{

data("nfi_donghae")

# Applying hierarchical filtering to select only privately owned forest subplots.
# Ensures all child tables' subplots match the filtered plot table's subplots.
# Expected results after filtering:
# all(nfi_donghae$tree$SUB_PLOT \%in\% nfi_donghae$plot$SUB_PLOT)  result: TRUE
nfi_donghae <- filter_nfi(nfi_donghae, c("plot$OWN_CD == '5'"), hier = TRUE)

\donttest{ 
# Non-hierarchical filtering to select only privately owned forest subplots.
# Child tables remain unfiltered and may not correspond to the plot table subplots.
# Expected results after filtering:
# all(nfi_donghae$tree$SUB_PLOT \%in\% nfi_donghae$plot$SUB_PLOT)  result: FALSE
nfi_donghae <- filter_nfi(nfi_donghae, c("plot$OWN_CD == '5'"), hier = FALSE)

# Non-Hierarchical Filtering with only woody plants.
# Other tables remain filtered and correspond to the tree table.
# Expected results after filtering:
# all(nfi_donghae$plot$SUB_PLOT \%in\% nfi_donghae$tree$SUB_PLOT)  result: TRUE
nfi_donghae <- filter_nfi(nfi_donghae, c("tree$WDY_PLNTS_TYP_CD == '1'"), hier = FALSE)

# Combining multiple filters across different dataframes
nfi_donghae <- filter_nfi(nfi_donghae, 
                    c("plot$OWN_CD == '5'", 
                    "tree$FAMILY == 'Pinaceae' | tree$WDY_PLNTS_TYP_CD == '1'"))
}

}
