% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/moonboot.R
\name{estimate.tau}
\alias{estimate.tau}
\title{Estimating the convergence rate}
\usage{
estimate.tau(
  data,
  statistic,
  R = 1000,
  replace = FALSE,
  min.m = 3,
  gamma = seq(0.2, 0.7, length.out = 5),
  method = "variance",
  ...
)
}
\arguments{
\item{data}{The data to be bootstrapped.}

\item{statistic}{A function returning the statistic of interest. It must take two arguments. The first argument passed will be the original data, the second
will be a vector of indices. Any further arguments can be passed through the \code{...} argument.}

\item{R}{Amount of bootstrap replicates used to estimate tau.}

\item{replace}{If sampling should be done with replacement.}

\item{min.m}{Minimal subsampling size used to estimate tau. Should be set to the minimum size for which the statistic makes sense.}

\item{gamma}{The tested subsample sizes m are \code{n^gamma}.}

\item{method}{Method to estimate tau, can be one of \code{c("variance", "quantile")}.}

\item{...}{Additional parameters to be passed to the \code{mboot} function.}
}
\value{
A function for the square root of the convergence rate of the variance, i.e., \code{f(n) = tau_n}. This function can directly be passed to \code{mboot.ci}.
}
\description{
This function estimates the convergence rate of the bootstrap estimator
and returns it as a function of the form \code{tau_n = n^a}, where \code{n} is the input parameter.
}
\details{
There are two methods to choose from, \code{variance} and \code{quantile}.
The provided \code{gamma} values are used to select subsample sizes \code{m} by using \code{ms = n^gamma}.
Note that the choice of the \code{gamma} values can impact the accuracy of the estimated \code{tau} (Dalitz & Lögler, 2024).
For each selected subsample size \code{m} a bootstrap with \code{R} replications is performed.
The method 'variance' then fits a linear function to log(variance) of the bootstrap statistics as function of log(m).
The method 'quantile' averages over multiple quantile ranges Q and fits a linear function to log(Q) as a function of log(m).
}
\examples{
data <- runif(1000)
estimate.max <- function(data, indices) {return(max(data[indices]))}
estimated.tau <- estimate.tau(data, estimate.max)
boot.out <- mboot(data, estimate.max, R = 1000, m = 2*sqrt(NROW(data)), replace = FALSE)
cis <- mboot.ci(boot.out, conf=0.95, tau=estimated.tau, types=c("all"))
ci.basic <- cis$basic
print(ci.basic)

}
\references{
Bertail P. et al. (1999) On subsampling estimators with unknown rate of convergence.
\emph{Journal of the American Statistical Association}, 94(446):568-579.

Politis D.N. et al. (1999)
\emph{Subsampling}, Springer, New York.

Dalitz, C, and Lögler, F. (2025)
\emph{moonboot: An R Package Implementing m-out-of-n Bootstrap Methods}.
\doi{10.32614/RJ-2025-031}
}
\seealso{
mboot.ci
}
