#' @title Toxic Dose 50 (TD50) Pharmacodynamic Model
#' @name td50_model
#' @description
#' Fits quantal toxicity response data to a logistic dose-response model
#' to estimate the Toxic Dose 50 (TD50), defined as the dose producing
#' toxicity in 50% of the population.
#'
#' The model uses binomial logistic regression and supports optional
#' grouping (e.g., sex, species, formulation) and stratification by
#' experimental conditions (e.g., exposure route).
#'
#' In addition to TD50 estimation, the model provides the following
#' interpretable parameters:
#' \itemize{
#'   \item \strong{Slope}: Represents the steepness of the dose-response curve.
#'   A larger slope indicates a rapid increase in toxicity with small
#'   increases in dose (narrow tolerance or high population sensitivity),
#'   whereas a smaller slope reflects a more gradual response, suggesting
#'   greater inter-individual variability in susceptibility.
#'
#'   \item \strong{Intercept}: Represents the baseline log-odds of observing
#'   toxicity at zero dose. A strongly negative intercept indicates minimal
#'   background toxicity, while a positive intercept suggests appreciable
#'   toxicity in the absence of administered dose, which may indicate
#'   experimental bias or background risk.
#'
#'   \item \strong{TD50 95\% Confidence Interval}: An approximate 95% confidence
#'   interval for the TD50, computed using the delta method. This provides
#'   an uncertainty range around the estimated dose causing 50% toxicity.
#'
#'   \item \strong{McFadden Pseudo-R\eqn{^2}}: A likelihood-based measure of
#'   model goodness-of-fit that quantifies how much better the fitted model
#'   explains the data compared to a null (intercept-only) model. Values
#'   between 0.1 and 0.2 indicate acceptable biological fit, while values
#'   above 0.3 suggest a strong and reliable dose-response relationship.
#' }
#'
#' The function can generate dose-response plots with fitted curves and
#' annotate TD50, slope, intercept, TD50 confidence intervals, and
#' McFadden pseudo-R\eqn{^2}.
#'
#'
#' @param data A data frame containing toxicity response data.
#' @param dose_col Character string specifying the dose column.
#' @param response_col Character string specifying the binary toxicity response
#'   (0 = no toxicity, 1 = toxic response).
#' @param group_col Optional character string specifying a grouping variable.
#' @param condition_col Optional character string specifying an experimental condition.
#' @param plot Logical; if TRUE, generates dose-response plots.
#' @param annotate Logical; if TRUE, annotates the plot with TD50, confidence
#'   intervals, and model parameters (only if <=2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats na.omit glm vcov logLik AIC BIC
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text labs theme
#' theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with TD50, 95% confidence intervals,
#'         slope, intercept, and pseudo-R2 values for each group.}
#'   \item{\code{data}}{The processed data used for model fitting and plotting.}
#' }
#' @examples
#' # Example I: Single population toxicity study
#' df1 <- data.frame(
#'   dose = c(5, 10, 20, 40, 80, 160),
#'   toxic = c(0, 0, 0, 1, 1, 1)
#' )
#' td50_model(
#'   data = df1,
#'   dose_col = "dose",
#'   response_col = "toxic"
#' )
#'
#' # Example II: Grouped analysis (Male vs Female)
#' df2 <- data.frame(
#'   dose = rep(c(5, 10, 20, 40, 80), 2),
#'   toxic = c(0,0,1,1,1, 0,0,0,1,1),
#'   sex = rep(c("Male","Female"), each = 5)
#' )
#' td50_model(
#'   data = df2,
#'   dose_col = "dose",
#'   response_col = "toxic",
#'   group_col = "sex"
#' )
#'
#' # Example III: Grouped by formulation and exposure route
#' df3 <- data.frame(
#'   dose = rep(c(10, 25, 50, 100), 4),
#'   toxic = c(0,0,1,1, 0,1,1,1, 0,0,0,1, 0,0,1,1),
#'   formulation = rep(c("A","B"), each = 8),
#'   route = rep(c("Oral","IV"), each = 4, times = 2)
#' )
#' td50_model(
#'   data = df3,
#'   dose_col = "dose",
#'   response_col = "toxic",
#'   group_col = "formulation",
#'   condition_col = "route"
#' )
#' @references Bliss, C. I. (1935) <doi:10.1111/j.1744-7348.1935.tb07713.x> The
#' calculation of the dosage-mortality curve. Annals of Applied Biology, 22(1),
#' 134–167.
#' @references Finney, D. J. (1971) <isbn:9780521080415> Probit Analysis, 3rd
#' Edition. Cambridge University Press, Cambridge.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("dose", "response", "group", "intercept", "slope", "TD50",
                         "TD50_lower", "TD50_upper", "pseudo_R2", "AIC", "BIC",
                         "label", "x_pos", "y_pos"))

td50_model <- function(data,
                       dose_col = "dose",
                       response_col = "response",
                       group_col = NULL,
                       condition_col = NULL,
                       plot = TRUE,
                       annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop("Package 'ggplot2' is required.")
  }

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(dose_col, response_col, group_col, condition_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("dose", "response")

  if (!all(df$response %in% c(0, 1))) {
    stop("response_col must contain binary values (0 or 1).")
  }

  if (!is.null(group_col) && !is.null(condition_col)) {
    df$group <- paste0(df[[group_col]], " | ", df[[condition_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else if (!is.null(condition_col)) {
    df$group <- as.factor(df[[condition_col]])
  } else {
    df$group <- "Experimental"
  }

  df$group <- as.factor(df$group)

  # -------------------------
  # Logistic TD50 fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fit <- stats::glm(response ~ dose, family = binomial, data = d)

    intercept <- coef(fit)[1]
    slope <- coef(fit)[2]

    TD50 <- -intercept / slope

    # --- Delta method for approximate 95% CI ---
    vcov_mat <- stats::vcov(fit)
    var_TD50 <- (1 / slope)^2 * vcov_mat[1,1] +
      (intercept / slope^2)^2 * vcov_mat[2,2] -
      2 * (intercept / slope^3) * vcov_mat[1,2]
    TD50_lower <- TD50 - 1.96 * sqrt(var_TD50)
    TD50_upper <- TD50 + 1.96 * sqrt(var_TD50)

    ll_null <- stats::logLik(stats::glm(response ~ 1, family = binomial, data = d))
    ll_full <- stats::logLik(fit)
    pseudo_R2 <- 1 - as.numeric(ll_full / ll_null)

    AIC_val <- stats::AIC(fit)
    BIC_val <- stats::BIC(fit)

    data.frame(
      group = unique(d$group),
      intercept = intercept,
      slope = slope,
      TD50 = TD50,
      TD50_lower = TD50_lower,
      TD50_upper = TD50_upper,
      pseudo_R2 = pseudo_R2,
      AIC = AIC_val,
      BIC = BIC_val
    )
  }))

  # -------------------------
  # Fitted curves for plotting
  # -------------------------
  df_fit <- do.call(rbind, lapply(split(df, df$group), function(d) {
    fr <- fit_results[fit_results$group == unique(d$group), ]
    dose_seq <- seq(min(d$dose), max(d$dose), length.out = 100)
    prob <- 1 / (1 + exp(-(fr$intercept + fr$slope * dose_seq)))
    data.frame(dose = dose_seq, response = prob, group = unique(d$group))
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(df, ggplot2::aes(x = dose, y = response, color = group)) +
      ggplot2::geom_point(size = 3, alpha = 0.8) +
      ggplot2::geom_line(
        data = df_fit,
        ggplot2::aes(x = dose, y = response, color = group),
        linewidth = 1.2
      ) +
      ggplot2::labs(
        title = "Toxic Dose 50 (TD50) Dose-Response Model",
        subtitle = "Logistic regression of quantal toxicity data",
        x = "Dose",
        y = "Probability of Toxic Response",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Conditional annotations
    # -------------------------
    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {

      ann <- fit_results
      ann$label <- paste0(
        "TD50 = ", round(ann$TD50, 2), " [", round(ann$TD50_lower, 2), ", ", round(ann$TD50_upper, 2), "]\n",
        "Slope = ", round(ann$slope, 3), "\n",
        "Intercept = ", round(ann$intercept, 2), "\n",
        "McFadden Pseudo R2 = ", round(ann$pseudo_R2, 3), "\n",
        "AIC = ", round(ann$AIC, 1), "\n",
        "BIC = ", round(ann$BIC, 1)
      )

      ann$x_pos <- max(df$dose) * 0.95
      ann$y_pos <- c(0.85, 0.15)[seq_len(nrow(ann))]

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(
            x = x_pos,
            y = y_pos,
            label = label,
            color = group
          ),
          hjust = 1,
          size = 4,
          show.legend = FALSE
        )

    } else if (num_groups > 2 && annotate) {
      message("More than 2 groups detected - annotations disabled to prevent overlap.")
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
