% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tfr_to_asfr.R
\name{tfr_to_asfr_scale}
\alias{tfr_to_asfr_scale}
\title{Derive Age-Specific Fertility Rates that
Match Total Fertility Rates by Scaling}
\usage{
tfr_to_asfr_scale(target, standard, suffix = NULL)
}
\arguments{
\item{target}{A data frame containing a variable called
\code{"tfr"}, and possibly others. See Details.}

\item{standard}{A data frame containing variables
called \code{age} and \code{asfr}, and possibly others.
See Details.}

\item{suffix}{Optional suffix added to
\code{asfr} column in results.}
}
\value{
A \link[tibble:tibble]{tibble}.
}
\description{
Turn total fertility rates (TFRs) into sets
of age-specific fertility rates, by scaling
a set of standard rates upwards or downwards.
}
\section{Method}{


Let \eqn{{}_nf_x} be the age-specific fertility rate for people
aged between \eqn{x} and \eqn{x+n}.
Values for \eqn{{}_nf_x} are obtained by
scaling the standard rates \eqn{{}_nf_x^{\mathrm{std}}}
so that they agree with the target
total fertility rate \eqn{F}. That is,
\code{tfr_to_asfr_scale()} sets

\deqn{{}_nf_x = \alpha \times {}_nf_x^{\mathrm{std}}}

where

\deqn{\alpha = \frac{F}{\sum_x n \times {}_nf_x^{\mathrm{std}}}}
}

\section{The \code{target} argument}{


\code{target} is a data frame specifying
total fertility rates for each population being modelled.

\code{target} contains the following variables:
\itemize{
\item A variable called \code{"tfr"}. An ordinary
numeric vector or an \link[rvec:rvec]{rvec()}.
\item Optionally, 'by' variables. Typical examples
are time, region, and model variant.
}
}

\section{The \code{standard} argument}{


\code{standard} is a data frame specifying
standard fertility schedules to be used
with each life expectancy
in \code{target}. Values in \code{standard} are age-specific.

\code{standard} contains the following variables:
\itemize{
\item A variable called \code{"age"}, with labels that
can be parsed by \code{\link[=reformat_age]{reformat_age()}}.
\item A variable called \code{"value"}, containing
non-negative values.  Cannot be an rvec.
\item Additional variables used to match rows in \code{standard}
to rows in \code{target}.
}
}

\examples{
## create age-specific fertility rates
## based on the [Booth standard][booth_standard]
library(dplyr, warn.conflicts = FALSE)
target <- data.frame(region = c("A", "B"), 
                     tfr = c(5.5, 4.7))
asfr <- tfr_to_asfr_scale(target = target,
                          standard = booth_standard)
asfr

## check consistency with original TFRs
asfr |>
  tfr(asfr = asfr, by = region)

## target is an rvec
library(rvec, warn.conflicts = FALSE)
target_rvec <- data.frame(region = c("A", "B"), 
                          tfr = rnorm_rvec(n = 2,
                                           mean = c(5.5, 4.7),
                                           n_draw = 1000))
tfr_to_asfr_scale(target = target_rvec,
                  standard = booth_standard)
}
\seealso{
\itemize{
\item \code{\link[=ex_to_lifetab_brass]{ex_to_lifetab_brass()}} Life table equivalent
of \code{tfr_to_asfr_scale()}.
\item \link{booth_standard} The 'Booth standard' fertility schedule
\item \link{tfr} Calculate total fertility rate from age-specific
fertility rates
}
}
