#' @title Test All Moderation Effects
#'
#' @description Test all moderation
#' effects by testing all product
#' terms for a `power4test` object.
#'
#' @details
#' This function is to be used in
#' [power4test()] for testing all
#' product terms, by
#' setting it to the `test_fun`
#' argument.
#'
#' It is just a wrapper to
#' [test_parameters()]. It will
#' first identifies all product
#' terms (terms with `:` in the names),
#' and then call [test_parameters()],
#' with `pars` set to select the
#' regression coefficients of these
#' terms.
#'
#' @return
#' In its normal usage, it returns
#' the output returned by
#' [lavaan::parameterEstimates()]
#' or [lmhelprs::lm_list_to_partable()],
#' with the following modifications:
#'
#' - `est`: The parameter estimates,
#'  even if standardized estimates
#'  are requested (not `est.std`).
#'
#' - `cilo` and `cihi`: The
#'  lower and upper limits of the
#'  confidence interval (95% by
#'  default), respectively (not
#'  `ci.lower` and `ci.upper`).
#'
#' - `sig`: Whether a test by confidence
#'  interval is significant (`1`) or
#'  not significant (`0`).
#'
#' - `test_label`: A column of labels
#'  generated by
#'  [lavaan::lav_partable_labels()],
#'  which are usually the labels used by
#'  `coef()` to label the parameters.
#'
#' @inheritParams test_parameters
#'
#' @seealso [power4test()],
#' [test_parameters()]
#'
#' @examples
#'
#' # Specify the model
#'
#' mod <-
#' "
#' m ~ x + w1 + x:w1
#' y ~ m + x
#' "
#'
#' # Specify the population values
#'
#' mod_es <-
#' "
#' m ~ x: n
#' y ~ x: m
#' m ~ w1: n
#' m ~ x:w1: l
#' "
#'
#' # Simulate the data
#'
#' sim_only <- power4test(nrep = 4,
#'                        model = mod,
#'                        pop_es = mod_es,
#'                        n = 100,
#'                        do_the_test = FALSE,
#'                        iseed = 1234)
#'
#' # Do the test in each replication
#'
#' test_out <- power4test(object = sim_only,
#'                        test_fun = test_moderation)
#'
#' print(test_out,
#'       test_long = TRUE)
#'
#' @export
test_moderation <- function(fit = fit,
                            standardized = FALSE,
                            check_post_check = TRUE,
                            ...,
                            fit_name = "fit",
                            get_map_names = FALSE,
                            get_test_name = FALSE) {
  map_names <- c(fit = fit_name)
  if (get_map_names) {
    return(map_names)
  }
  if (get_test_name) {
    tmp <- character(0)
    if (standardized) {
      return(paste("test_moderation: CIs (standardized)", tmp))
    } else {
      return(paste("test_moderation: CIs", tmp))
    }
  }

  if (inherits(fit, "lm_list")) {
    fit_type <- "lm_list"
  } else if (inherits(fit, "lavaan")) {
    fit_type <- "lavaan"
  } else {
    stop("fit is not a supported object.")
  }

  if (fit_type == "lm_list") {
    est0 <- lmhelprs::lm_list_to_partable(fit)
  } else {
    est0 <- lavaan::parameterEstimates(fit,
                                      se = FALSE,
                                      pvalue = FALSE,
                                      ci = FALSE)
  }
  est0$test_label <- lavaan::lav_partable_labels(est0)
  i <- est0$op %in% "~"
  j <- grepl(":", est0$test_label)
  k <- i & j
  if (length(k) == 0) {
    stop("No product terms found in the model.")
  }
  pars <- est0$test_label[k]
  out <- test_parameters(fit = fit,
                         standardized = standardized,
                         pars = pars,
                         ...)
  return(out)
}
