% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prepost_mi_list.R
\name{prepost_mi}
\alias{prepost_mi}
\title{Prepost Analysis with Pre-Imputed Datasets}
\usage{
prepost_mi(
  data_list,
  formula,
  k = 1,
  nboot = 500,
  pool_method = "detailed",
  coef_labels = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data_list}{List of imputed datasets (data frames) or a `mids` object from `mice`}

\item{formula}{Formula for the prepost model (e.g., post_score ~ pre_score + age + bmi)}

\item{k}{Resilience parameter (default = 1.2)}

\item{nboot}{Number of bootstrap samples (default = 200)}

\item{pool_method}{Pooling method: "detailed" (Barnard & Rubin) or "simple" (default = "detailed")}

\item{coef_labels}{Optional named vector for renaming coefficients (e.g., c("age" = "Age (years)"))}

\item{verbose}{Whether to print progress messages (default = TRUE)}

\item{...}{Additional arguments passed to prepost()}
}
\value{
A list containing:
\describe{
  \item{\code{pooled_results}}{The pooled estimates}
  \item{\code{results_table}}{Formatted results table}
  \item{\code{individual_results}}{List of results from each imputation}
  \item{\code{model_info}}{Information about the model specification}
  \item{\code{imputation_info}}{Summary of imputation characteristics}
}
}
\description{
This function takes a list of already imputed datasets and runs prepost (resilience) regression analysis
on each, then pools the results using Rubin's rules. Users are responsible for
the imputation process, allowing maximum flexibility in imputation methods.
}
\examples{
\dontrun{
# Example 1: List of data frames from any imputation method
library(mice)
data <- data.frame(
  post_score = rnorm(100),
  pre_score = rnorm(100),
  age = rnorm(100, 50, 10),
  bmi = rnorm(100, 25, 5)
)

# Create some missing data
data$age[1:10] <- NA
data$bmi[5:15] <- NA

# User does their own imputation (could be from any package/method)
# Method 1: Using mice
imp_mice <- mice(data, m = 5, printFlag = FALSE)
imp_list <- list()
for (i in 1:5) {
  imp_list[[i]] <- complete(imp_mice, i)
}

# Method 2: Using Amelia
if (requireNamespace("Amelia", quietly = TRUE)) {
  imp_amelia <- Amelia::amelia(data, m = 5)
  imp_list <- imp_amelia$imputations
}

# Method 3: Using aregImpute from Hmisc
if (requireNamespace("Hmisc", quietly = TRUE)) {
  set.seed(123)
  imp_areg <- Hmisc::aregImpute(~ post_score + pre_score + age + bmi, data = data, n.impute = 5)
  imp_list <- list()
  for (i in 1:5) {
    imp_data <- data
    imp_data$age[is.na(imp_data$age)] <- imp_areg$imputed$age[, i]
    imp_data$bmi[is.na(imp_data$bmi)] <- imp_areg$imputed$bmi[, i]
    imp_list[[i]] <- imp_data
  }
}

# Run resilience analysis on the pre-imputed list
result <- prepost_mi(
  data_list = imp_list,
  formula = post_score ~ pre_score + age + bmi,
  k = 1.2,
  nboot = 200
)

# Example 2: Directly using a mice mids object
result2 <- prepost_mi(
  data_list = imp_mice,  # mids object
  formula = post_score ~ pre_score + age + bmi
)
}

}
\seealso{
\code{\link{prepost}} for complete-case analysis without missing data
\code{\link[mice]{mice}} for multiple imputation
}
