#' Get genomic regions for coding and non-coding sequences of a transcript
#'
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' @param tx.id The Ensmble ID of the transcript
#' @param keep.stop.codon A logical value indicating whether the stop codon to be included in genomic regions
#' 
#' @return A list of two GRanges objects for CDS regions and non-CDS regions
#' 
#' @importFrom ensembldb cdsBy
#' @importFrom GenomicRanges gaps
#' @importFrom IRanges ranges
#' @importFrom BiocGenerics strand
#' @importFrom BiocGenerics end
#' @importFrom BiocGenerics start
#' 
#' @examples
#' \dontrun{
#'     getCdsAndNoncds()
#' }
#'
#' @noRd
getCdsAndNoncds <- function(ensdb, tx.id, keep.stop.codon=TRUE){
    
    cds.gr <- ensembldb::cdsBy(ensdb, filter = ~ (tx_id==tx.id))[[1]]
    cds.gr <- sort(cds.gr)
    
    strand <- as.character(BiocGenerics::strand(range(cds.gr)))
    
    if( !keep.stop.codon ){
        if(strand!="-"){
            BiocGenerics::end(IRanges::ranges(cds.gr[length(cds.gr)])) <- BiocGenerics::end(IRanges::ranges(cds.gr[length(cds.gr)])) - 3
        }else{
            BiocGenerics::start(IRanges::ranges(cds.gr[1])) <- BiocGenerics::start(IRanges::ranges(cds.gr[1])) + 3
        }
    }
    
    noncds.gr <- GenomicRanges::gaps( cds.gr, start=BiocGenerics::start(IRanges::ranges(range(cds.gr))), end=BiocGenerics::end(IRanges::ranges(range(cds.gr))) )
    noncds.gr <- noncds.gr[ BiocGenerics::strand(noncds.gr)==strand ]
    
    gr.list <- list(cds=cds.gr, noncds=noncds.gr)
    
    return(gr.list)
}

