#' Left-align an indel
#'
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' @param chr.name The chromosome name 
#' @param mut.type The type of the mutation: DEL or INS
#' @param mut.start The start position of the mutation
#' @param mut.end The end position of the mutation
#' @param mut.ref The reference allele of the mutation
#' @param mut.alt The alternative allele of the mutation
#' @param flank.bp A non-negative integer specifying the length of flanking regions to be added to both ends of the mutation locus for left-alignment
#' 
#' @return A list of a left-aligned indel including mutation type, mutation start position, mutation end position, mutation reference allele, mutation alternative allele  
#' 
#' @importFrom BSgenome getSeq
#' @importFrom GenomicRanges GRanges
#' @importFrom IRanges IRanges
#'
#' @examples
#' \dontrun{
#'     leftAlignIndel()
#' }
#'
#' @noRd
leftAlignIndel <- function(
        reference.genome,
        chr.name,
        mut.type,
        mut.start, 
        mut.end, 
        mut.ref,
        mut.alt,
        flank.bp = 100
    ){
    
    chr.length <- length( unlist( BSgenome::getSeq(reference.genome, chr.name) ) )
    
    ref.start <- mut.start - flank.bp
    ref.end <- mut.end + flank.bp
    if(ref.start < 1){
        ref.start <- 1
    }
    if(ref.end > chr.length){
        ref.end <- chr.length
    }
    
    ref <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(ref.start, ref.end))) )
    
    mutDNA <- data.frame(
        ref_pos = ref.start:ref.end,
        ref_seq = unlist( strsplit(ref,"") ),
        stringsAsFactors = FALSE
    )
    mutDNA$mut_seq <- mutDNA$ref_seq
    mutDNA$cigar <- rep("M", nrow(mutDNA))
    
    if(mut.type=="DEL"){
        mutDNA$mut_seq[ mutDNA$ref_pos %in% mut.start:mut.end ] <- "-"
        mutDNA$cigar[ mutDNA$ref_pos %in% mut.start:mut.end ] <- "D"
    }else if(mut.type=="INS"){
        temp.df <- data.frame(
            ref_pos = NA,
            ref_seq = "-",
            mut_seq = unlist( strsplit(mut.alt,"") ),
            cigar = "I",
            stringsAsFactors = FALSE
        )
        mutDNA <- rbind(
            mutDNA[ 1:which(mutDNA$ref_pos==mut.start), ],
            temp.df,
            mutDNA[ which(mutDNA$ref_pos==mut.end):nrow(mutDNA), ] 
        )
    }
    
    mutant.ref <- paste(mutDNA$mut_seq[mutDNA$mut_seq!="-"], collapse="")
    
    left.aln <- gapShiftingPairwiseAlignment(ref, mutant.ref, "left", "global", 1, 4, 6, 0)
    left.aln.cigar <- encodeCIGAR(left.aln[1], left.aln[2])
    left.aln.df <- decodeCIGAR(left.aln.cigar, ref, mutant.ref)
    left.aln.df$ref_pos <- ref.start + left.aln.df$ref_index - 1
    
    if(mut.type=="DEL"){
        
        left.aln.del <- left.aln.df[ !is.na(left.aln.df$del) | !is.na(left.aln.df$ins) | !is.na(left.aln.df$snv), ]
        if( nrow(left.aln.del)==1 ){
            left.aln.start <- left.aln.del$ref_pos
            left.aln.ref <- left.aln.del$del
        }else{
            left.aln.start <- mut.start
            left.aln.ref <- mut.ref
        }
        left.aln.end <- left.aln.start + left.aln.ref - 1 
        left.aln.alt <- "-"
        
    }else if(mut.type=="INS"){
        
        left.aln.ins <- left.aln.df[ !is.na(left.aln.df$del) | !is.na(left.aln.df$ins) | !is.na(left.aln.df$snv), ]
        if( nrow(left.aln.ins)==1 ){
            left.aln.start <- left.aln.ins$ref_pos
            left.aln.alt <- left.aln.ins$ins
        }else{
            left.aln.start <- mut.start
            left.aln.alt <- mut.alt
        }
        left.aln.end <- left.aln.start + 1
        left.aln.ref <- "-"
    }
    
    left.aln.mut <- list(
        mut.type = mut.type,
        mut.start = left.aln.start, 
        mut.end = left.aln.end, 
        mut.ref = left.aln.ref, 
        mut.alt = left.aln.alt
    )
    
    return(left.aln.mut)
}



