#' Assess an R Package riskmetric with package name and version
#'
#' This function use `risk.assessr::assess_pkg` assessment function with only the package name and version
#'
#' @param package_name A character string specifying the name of the package to assess.
#' @param version A character string specifying the version of the package to assess. Default is `NA`, which assesses the latest version.
#' @param repos A character string specifying the repo directly. Default is NULL, which uses the mirrors
#'
#' @return The function returns a list of assessment results generated by the `risk.assessr::assess_pkg` function. 
#' If the package cannot be downloaded or installed, an error message is returned.
#'
#' @details This function follows these steps:
#' \enumerate{
#'   \item Downloads the specified R package
#'   \item Installs the downloaded package in a temporary location.
#'   \item Runs the `risk.assessr::assess_pkg` function to assess the package for risks and issues.
#'   \item Returns the results of the assessment.
#' }
#'
#' @examples
#' \donttest{
#' r <- getOption("repos")
#' # save current repo options  
#' old <- options(repos = r)
#' r["CRAN"] = "http://cran.us.r-project.org"
#' options(repos = r) 
#' 
#' results <- assess_pkg_r_package("here", version = "1.0.1")
#' 
#' # restore user's repo options
#' options(old)
#' 
#' print(results)
#' }
#' 
#' @importFrom remotes download_version
#' @export
assess_pkg_r_package <- function(package_name, version=NA, repos = NULL) {
  
  # Save current repo options
  old_repos <- getOption("repos")
  
  
  if (!is.null(repos)) {
    repo_to_use <- repos
  } else {
    repo_to_use <- getOption("repos")
  }
  
  options(repos = repo_to_use)
  
  download_successful <- FALSE
  message(paste("Checking", package_name, "on CRAN..."))
  
  temp_file <- NULL
  
  # Try CRAN first
  if (check_cran_package(package_name)) {
    tryCatch({
      temp_file <- remotes::download_version(package = package_name, version = version)
      download_successful <- TRUE
    }, error = function(e) {
      message(paste("CRAN download failed:", e$message))
    })
  }
  
  # If not successful on CRAN, try Bioconductor
  if (!download_successful) {
    message(paste("Checking", package_name, "on Bioconductor..."))
    
    html_content <- fetch_bioconductor_releases()
    release_data <- parse_bioconductor_releases(html_content)
    result_bio <- get_bioconductor_package_url(package_name, version, release_data)
    
    if (!is.null(result_bio$url)) {
      temp_file <- tempfile()
      tryCatch({
        download.file(url = result_bio$url, destfile = temp_file, mode = "wb")
        download_successful <- TRUE
      }, error = function(e) {
        message(paste("Bioconductor download failed:", e$message))
      })
    } else {
      message(paste("No", package_name, "package found on Bioconductor"))
    }
  }
  
  # Final fallback: attempt internal mirror
  if (!download_successful) {
    message(paste("Attempting internal fallback download for", package_name))
    
    tryCatch({
      package_info <- get_internal_package_url(package_name = package_name, version = version)
      
      if (!is.null(package_info$url)) {
        temp_file <- tempfile(fileext = ".tar.gz")
        download.file(package_info$url, temp_file, mode = "wb", quiet = TRUE)
      } else {
        stop("Failed to find the package URL from the internal repository.")
      }
    }, error = function(e) {
      message(paste("Failed to download the package from any source. Error:", e$message), call. = FALSE)
    })
  }
  
  if (is.null(temp_file)) {
    return(NULL)
  }
  
  # If no error, proceed with the package tarball
  modified_tar_file <- modify_description_file(temp_file)
  
  # Set up the package using the temporary file
  install_list <- set_up_pkg(modified_tar_file)
  
  # Extract information from the installation list
  build_vignettes <- install_list$build_vignettes
  package_installed <- install_list$package_installed
  pkg_source_path <- install_list$pkg_source_path
  rcmdcheck_args <- install_list$rcmdcheck_args
  
  # Check if the package needs to be installed locally
  package_installed <- install_package_local(pkg_source_path)
  
  # Check if the package was installed successfully
  if (package_installed == TRUE) {
    rcmdcheck_args$path <- pkg_source_path
    
    # Pass custom risk config if provided
    assess_package <- assess_pkg(
      pkg_source_path = pkg_source_path,
      rcmdcheck_args = rcmdcheck_args
    )
  } else {
    message("Package installation failed.")
    assess_package <- NULL
  }
  
  # Clean up: remove the temporary file
  unlink(temp_file)
  
  
  # Restore original repo options if they were changed
  if (!identical(getOption("repos"), old_repos)) {
    options(repos = old_repos)
  }
  
  return(assess_package)
}