% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/snreg.R
\name{snreg}
\alias{snreg}
\title{Linear Regression with Skew-Normal Errors}
\usage{
snreg(
  formula,
  data,
  subset,
  init.sk = NULL,
  ln.var.v = NULL,
  skew.v = NULL,
  start.val = NULL,
  technique = c("nr"),
  vcetype = c("aim"),
  lmtol = 1e-05,
  reltol = 1e-12,
  maxit = 199,
  optim.report = 1,
  optim.trace = 1,
  print.level = 0,
  digits = 4,
  only.data = FALSE,
  ...
)
}
\arguments{
\item{formula}{an object of class \code{formula} specifying the regression:
typically \code{y ~ x1 + ...}, where \code{y} is the dependent variable
and \code{x}'s are regressors.}

\item{data}{an optional \code{data.frame} containing the variables in \code{formula}.
If not found in \code{data}, variables are taken from \code{environment(formula)}.}

\item{subset}{an optional logical or numeric vector specifying the subset of observations
to be used in estimation.}

\item{init.sk}{numeric. Initial value for the (global) skewness parameter of the noise;
can be \code{NULL} if \code{skew.v} is supplied with its own coefficients to initialize.}

\item{ln.var.v}{optional one-sided formula; e.g. \code{ln.var.v ~ z1 + z2}. Specifies
exogenous variables entering the (log) variance of the random noise component.
If \code{NULL}, the noise variance is homoskedastic.}

\item{skew.v}{optional one-sided formula; e.g. \code{skew.v ~ z3 + z4}. Specifies exogenous
variables determining the skewness of the noise via a linear index; if
\code{NULL}, the skewness is constant (scalar).}

\item{start.val}{optional numeric vector of starting values for all free parameters
(regression coefficients, variance/heteroskedasticity parameters, skewness parameters).}

\item{technique}{character vector giving the preferred maximization routine(s) in order of
preference. Currently recognized keywords include \code{"nr"} (Newton–Raphson),
\code{"bhhh"}, \code{"nm"} (Nelder–Mead), \code{"bfgs"}, \code{"cg"}.
This scaffold does not implement them yet, but records the choice.}

\item{vcetype}{character specifying the variance-covariance estimator type:
\code{"aim"} for the approximated information matrix or \code{"opg"}
for the outer product of gradients. Default is \code{"aim"}.}

\item{lmtol}{numeric. Convergence tolerance based on the scaled gradient (if applicable).
Default is \code{1e-5}.}

\item{reltol}{numeric. Relative convergence tolerance for likelihood maximization.
Default is \code{1e-12}.}

\item{maxit}{integer. Maximum number of iterations for the optimizer. Default is \code{199}.}

\item{optim.report}{integer. Verbosity for reporting progress (if implemented). Default is \code{1}.}

\item{optim.trace}{integer. If positive, tracing information is printed (if implemented).
Default is \code{1}.}

\item{print.level}{integer. Printing level for summaries: \code{1}—print estimation results;
\code{2}—print optimization details; \code{3}—print compact summary. Default \code{3}.}

\item{digits}{integer. Number of digits for printing. Default \code{4}.}

\item{only.data}{logical. If \code{TRUE}, the function returns only the constructed model
matrices and design sets (no estimation). Default \code{FALSE}.}

\item{...}{additional arguments reserved for future methods (e.g., box constraints).}
}
\value{
An object of class \code{"snreg"} containing the maximum-likelihood results and,
depending on the optimization routine, additional diagnostics:

\describe{
  \item{\code{par}}{Numeric vector of parameter estimates at the optimum.}
  \item{\code{coef}}{Named numeric vector equal to \code{par}.}

  \item{\code{vcov}}{Variance–covariance matrix of the estimates.}
  \item{\code{sds}}{Standard errors, computed as \code{sqrt(diag(vcov))}.}
  \item{\code{ctab}}{Coefficient table with columns:
        \code{Estimate}, \code{Std.Err}, \code{Z value}, \code{Pr(>z)}.}

  \item{\code{RSS}}{Residual sum of squares.}
  \item{\code{esample}}{Logical vector indicating which observations were used in estimation.}
  \item{\code{n}}{Number of observations used in the estimation sample.}
  \item{\code{skewness}}{Vector of the fitted skewness index.}

  \item{\code{hessian}}{(BFGS only) Observed Hessian at the optimum. If \code{vcetype == "opg"},
        this is set to the negative outer product of the individual gradients;
        otherwise a numerical Hessian is computed.}
  \item{\code{value}}{(BFGS only) Objective value returned by \code{optim}. With
        \code{control$fnscale = -1}, this equals the maximized log-likelihood.}
  \item{\code{counts}}{(BFGS only) Number of iterations / function evaluations returned by \code{optim}.}
  \item{\code{convergence}}{(BFGS only) Convergence code from \code{optim}.}
  \item{\code{message}}{(BFGS only) Additional \code{optim} message, if any.}

  \item{\code{ll}}{Maximized log-likelihood value.}
  \item{\code{gradient}}{(NR only) Gradient at the solution.}
  \item{\code{gg}}{(NR only) Optional gradient-related diagnostic.}
  \item{\code{gHg}}{(NR only) Optional Newton-step diagnostic.}
  \item{\code{theta_rel_ch}}{(NR only) Relative parameter change metric across iterations.}
}

The returned object has class \code{"snreg"}.
}
\description{
\code{snreg} fits a linear regression model where the disturbance term follows
a skew-normal distribution. The function supports multiplicative
heteroskedasticity of the noise variance via a log-linear specification
(\code{ln.var.v}) and allows the skewness parameter to vary linearly with
exogenous variables (\code{skew.v}).
}
\details{
Linear Regression with Skew-Normal Errors


The model is
\deqn{y_i = x_i^\top \beta + \varepsilon_i,\quad \varepsilon_i \sim SN(0, \sigma_i^2, \alpha_i),}
where \eqn{SN} denotes the skew-normal distribution (Azzalini).

Heteroskedasticity in the noise variance (if specified via \code{ln.var.v}) is modeled as
\deqn{\log(\sigma_i^2) = w_i^\top \gamma_v,}
and the (optional) covariate-driven skewness (if specified via \code{skew.v}) as
\deqn{\alpha_i = s_i^\top \delta.}

This function constructs the model frame and design matrices for
\eqn{\beta}, \eqn{\gamma_v}, and \eqn{\delta}, and is designed to be paired with
a maximum likelihood routine to estimate parameters and (optionally) their
asymptotic covariance via either AIM or OPG.
}
\examples{
library(snreg)

data("banks07")
head(banks07)

# Translog cost function specification

spe.tl <- log(TC) ~ (log(Y1) + log(Y2) + log(W1) + log(W2))^2 +
  I(0.5 * log(Y1)^2) + I(0.5 * log(Y2)^2) +
  I(0.5 * log(W1)^2) + I(0.5 * log(W2)^2)

# Specification 1: homoskedastic noise and skewness

formSV <- NULL   # variance equation; constant variance
formSK <- NULL   # skewness equation; constant skewness

m1 <- snreg(
  formula  = spe.tl,
  data     = banks07,
  ln.var.v = formSV,
  skew.v   = formSK
)

summary(m1)

# Specification 2: heteroskedastic

formSV <- ~ log(TA)      #' variance equation; heteroskedastic noise (variance depends on TA)
formSK <- ~ ER           #' skewness equation; with determinants (skewness is determined by ER)

m2 <- snreg(
  formula  = spe.tl,
  data     = banks07,
  prod     = myprod,
  ln.var.v = formSV,
  skew.v   = formSK
)

summary(m2)

}
\references{
Azzalini, A. (1985).
\emph{A Class of Distributions Which Includes the Normal Ones}.
Scandinavian Journal of Statistics, 12(2), 171–178.

Azzalini, A., & Capitanio, A. (2014).
\emph{The Skew-Normal and Related Families}.
Cambridge University Press.
}
\seealso{
\code{\link{snsf}}, \code{\link{lm.mle}}
}
\keyword{heteroskedasticity}
\keyword{maximum-likelihood}
\keyword{regression}
\keyword{skew-normal}
