% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/labels.R
\name{add_auto_labels}
\alias{add_auto_labels}
\title{Add automatic labels from dictionary to a gtsummary table}
\usage{
add_auto_labels(tbl, dictionary)
}
\arguments{
\item{tbl}{A gtsummary table object created by
\code{tbl_summary()}, \code{tbl_svysummary()}, or
\code{tbl_regression()}.}

\item{dictionary}{A data frame or tibble with columns
named \code{variable} and \code{description} (column name
matching is case-insensitive). If not provided
(missing), the function will search for a \code{dictionary}
object in the environment. If no dictionary is
found, the function will attempt to read label
attributes from the data. Set to \code{NULL} explicitly to
skip dictionary search and only use attributes.}
}
\value{
A gtsummary table object with labels applied. Manual labels set via
\code{label = list(...)} in the original table call are always preserved.
}
\description{
Applies variable labels from a dictionary or label
attributes to \code{tbl_summary}, \code{tbl_svysummary}, or \code{tbl_regression} objects.
Preserves manual label overrides set in the original table
call. The dictionary can be passed explicitly or will be
searched for in the calling environment. If no dictionary
is found, the function reads label attributes from the
underlying data.
}
\details{
\subsection{Label Priority Hierarchy}{

The function applies labels according to this priority (highest to lowest):
\enumerate{
\item \strong{Manual labels} -- Labels set via \code{label = list(...)}
in \code{tbl_summary()} etc. are always preserved
\item \strong{Attribute labels} -- Labels from \code{attr(data$var, "label")}
\item \strong{Dictionary labels} -- Labels from the dictionary data frame
\item \strong{Default} -- If no label source is available, uses variable name
}

Set \code{options(sumExtras.prefer_dictionary = TRUE)} to swap priorities 2
and 3 so that dictionary labels take precedence over attribute labels.
See \code{vignette("options")} for details.
}

\subsection{Dictionary Format}{

The dictionary must be a data frame with columns
(column names are case-insensitive):
\itemize{
\item \code{variable}: Character column with exact variable names from datasets
\item \code{description}: Character column with human-readable labels
}
}

\subsection{Label Attributes}{

The function reads label attributes from data using
\code{attr(data$var, "label")}, following the same convention
used by haven, Hmisc, and ggplot2 4.0+.
If your data already has labels (from imported files,
other packages, or manual assignment), this function
picks them up automatically.
}

\subsection{Implementation Note}{

\strong{This function relies on internal gtsummary structures} (\code{tbl$call_list},
\code{tbl$inputs}, \code{tbl$table_body}) to detect manually set labels. Major updates
to gtsummary may require corresponding updates to sumExtras.
Requires gtsummary >= 1.7.0.
}
}
\examples{
\donttest{
# Create a dictionary
my_dict <- tibble::tribble(
  ~variable, ~description,
  "age", "Age at Enrollment",
  "trt", "Treatment Group",
  "grade", "Tumor Grade"
)

# Strip built-in labels so dictionary labels are visible
trial_data <- gtsummary::trial
for (col in names(trial_data)) attr(trial_data[[col]], "label") <- NULL

# Pass dictionary explicitly
trial_data |>
  gtsummary::tbl_summary(by = trt, include = c(age, grade)) |>
  add_auto_labels(dictionary = my_dict)

# Automatic dictionary search (dictionary in environment)
dictionary <- my_dict
trial_data |>
  gtsummary::tbl_summary(by = trt, include = c(age, grade)) |>
  add_auto_labels() # Finds dictionary automatically

# Working with pre-labeled data (no dictionary needed)
labeled_data <- gtsummary::trial
attr(labeled_data$age, "label") <- "Patient Age (years)"
attr(labeled_data$marker, "label") <- "Marker Level (ng/mL)"

labeled_data |>
  gtsummary::tbl_summary(include = c(age, marker)) |>
  add_auto_labels() # Reads from label attributes

# Manual overrides always win
trial_data |>
  gtsummary::tbl_summary(
    by = trt,
    include = c(age, grade),
    label = list(age ~ "Custom Age Label") # Manual override
  ) |>
  add_auto_labels(dictionary = my_dict) # grade: dict, age: manual

}

}
\seealso{
\itemize{
\item \code{gtsummary::modify_table_body()} for advanced table
customization
}
}
