\name{sumer-package}
\alias{sumer-package}
\alias{sumer}
\docType{package}
\title{Tools for Working with Sumerian Cuneiform Texts}
\description{
A toolkit for analyzing and translating Sumerian cuneiform texts. The package
provides functions for converting sign names to cuneiform characters, creating
and querying dictionaries, and analyzing the structure of Sumerian words.
}
\section{Getting Started}{
Load the package and explore the built-in dictionary:

\preformatted{
library(sumer)

# Load the built-in dictionary
dic <- read_dictionary()

# Look up a Sumerian word
look_up("d-suen", dic)

# Search for a term in translations
look_up("water", dic, "en")
}
}
\section{Cuneiform Conversion and Analysis}{
Functions for converting sign names to cuneiform and analyzing sign structure:

\describe{
\item{\code{\link{as.cuneiform}}}{
Converts Sumerian text to cuneiform characters.
\preformatted{
as.cuneiform("lugal")
as.cuneiform("AN.EN.ZU")
}
}
\item{\code{\link{as.sign_name}}}{
Converts Sumerian text to sign names.
\preformatted{
as.sign_name("lugal")
}
}
\item{\code{\link{split_sumerian}}}{
Splits compound sign names into individual components.
\preformatted{
split_sumerian("AN.EN.ZU")
}
}
\item{\code{\link{info}}}{
Displays detailed information about a Sumerian text.
\preformatted{
info("jic-tukul")
}
}
}
}
\section{Dictionary Creation}{
Functions for creating dictionaries from annotated translation files:

\describe{
\item{\code{\link{read_translated_text}}}{
Reads annotated translation files (.docx or .txt) and extracts sign names,
grammatical types, and meanings.
\preformatted{
filename     <- system.file("extdata", "text_with_translations.txt", package = "sumer")
translations <- read_translated_text(filename)
}
}
\item{\code{\link{convert_to_dictionary}}}{
Converts translation data into dictionary format, adding cuneiform
representations and phonetic readings.
\preformatted{
dictionary <- convert_to_dictionary(translations)
}
}
\item{\code{\link{make_dictionary}}}{
Convenience function that combines reading and conversion in one step.
\preformatted{
dictionary <- make_dictionary(filename)
}
}
}
}
\section{Dictionary Input/Output}{
Functions for saving and loading dictionaries:

\describe{
\item{\code{\link{save_dictionary}}}{
Saves a dictionary to a text file with metadata header.
\preformatted{
save_dictionary(dic, "my_dictionary.txt",
                author = "John Doe",
                version = "1.0")
}
}
\item{\code{\link{read_dictionary}}}{
Loads a dictionary from file. Without arguments, loads the built-in dictionary.
\preformatted{
dic <- read_dictionary()
dic <- read_dictionary("my_dictionary.txt")
}
}
}
}
\section{Dictionary Lookup}{
\describe{
\item{\code{\link{look_up}}}{
Interactive dictionary search. Supports both forward lookup (Sumerian to
English) and reverse lookup (English to Sumerian).
\preformatted{
# Forward lookup: Sumerian sign name
look_up("AN", dic)
look_up("AN.EN.ZU", dic)

# Reverse lookup: search in translations
look_up("king", dic, "en")
look_up("Gilgamesh", dic, "en")
}
}
\item{\code{\link{skeleton}}}{
Creates a structured template (skeleton) for translating Sumerian text.
\preformatted{
skeleton("e-ta-na an-ce3 ba-ed3-de3")
}
}
}
}
\section{Typical Workflows}{
\subsection{Workflow A: Analyze and translate a text}{
\preformatted{
library(sumer)

# Convert sign name to cuneiform
as.cuneiform("d-en-líl")

# Load dictionary
dic <- read_dictionary()

# Look up the meaning
look_up("d-en-líl", dic)

# Get information about individual signs
info("d-en-líl")
info("lil")
info("lil2")
}
}
\subsection{Workflow B: Create your own dictionary}{
\preformatted{
library(sumer)

# Read annotated translations from file
filename     <- system.file("extdata", "text_with_translations.txt", package = "sumer")
translations <- read_translated_text(filename)

# Convert to dictionary format
dictionary <- convert_to_dictionary(translations)

# Save for later use
save_dictionary(dictionary, "my_dictionary.txt",
                author  = "My Name",
                year    = "2025",
                version = "1.0")

# Load and use
my_dic <- read_dictionary("my_dictionary.txt")
look_up("ki", my_dic)
}
}
}
\author{
\strong{Maintainer}: [Robin Wellmann] \email{ro.wellmann@gmail.com}
}
\seealso{
Core functions:
\code{\link{info}},
\code{\link{read_dictionary}},
\code{\link{look_up}}

Dictionary creation:
\code{\link{read_translated_text}},
\code{\link{convert_to_dictionary}},
\code{\link{make_dictionary}},
\code{\link{save_dictionary}}
\code{\link{skeleton}}

Analysis tools:
\code{\link{split_sumerian}},
\code{\link{as.cuneiform}},
\code{\link{as.sign_name}}
}
