% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tmt_rm.R
\name{tmt_rm}
\alias{tmt_rm}
\title{Estimation (CML) ot the Rasch model with or without multistage designs.}
\usage{
tmt_rm(
  dat,
  mstdesign = NULL,
  weights = NULL,
  start = NULL,
  sum0 = TRUE,
  se = TRUE,
  optimization = "nlminb",
  ...
)
}
\arguments{
\item{dat}{a matrix of dichotomous (0/1) data or a list of the function \code{tmt_designsim}}

\item{mstdesign}{Model for the multistage design, if CML estimation without multistage designs is required, than leave the default value}

\item{weights}{is optional for the weights of cases}

\item{start}{Vector of start values. If no vector is provided, the start values will be automatic generated}

\item{sum0}{logical: If the item parameters should be normed to 'sum = 0' as recommended by Glas (2016, p. 208). Otherwise sum0=FALSE}

\item{se}{logical: should the standard error should be estimated?}

\item{optimization}{character: Per default 'nlminb' is used but 'optim' is also supported.}

\item{...}{optional further arguments for optim and nlminb use control = list() with arguments.}
}
\value{
List with following entries

	\item{betapar}{Estimated item difficulty parameters (if sum0=FALSE, than the first item is set to 0)}
	\item{se.beta}{Standard errors of the estimated item parameters}
	\item{loglik}{Conditional log-likelihood of the model}
	\item{df}{Number of estimated parameters}
	\item{N}{Number of Persons}
	\item{I}{Number of items}
	\item{data_orig}{Submitted data frame with item responses}
	\item{data}{Used data frame with item responses}
	\item{desmat}{Design matrix}
	\item{convergence}{Convergence criterion}
	\item{iterations}{Number of iterations}
	\item{hessian}{Hessian-Matrix}
	\item{model}{Used model ((mst) for Rasch model with multistage design)}
	\item{call}{Submitted arguments for the function (matched call)}
	\item{designelements}{If the multistage version is requested, the preprocessed design is returned, otherwise NULL}
	\item{mstdesign}{If the multistage version is requested, the submitted design is returned, otherwise NULL}
}
\description{
The \code{tmt_rm} function estimates the Rasch model. If the data are collected based on a multistage design (see Zwitser and Maris, 2015) the specific multistage design \code{mstdesign} has to be submitted.
}
\details{
According to Glas (1988) <doi:10.3102/10769986013001045> CML estimation of item parameters is biased if the data is collected in multistage designs and this design is not considered. Zwitser and Maris (2015) <doi:10.1007/s11336-013-9369-6> propose to use an additional design matrix to fragment the elementary symmetric function. Their approach is implemented in this package. MST designs with a probabilistic instead of a deterministic routing rule (see, e.g. Chen, Yamamoto, & von Davier, 2014 <doi:10.1201/b16858>) are not estimated with this method, therefore the proposed solouting is again modified by Steinfeld and Robitzsch (2021) <doi:10.31234/osf.io/ew27f> which is also integrated into this package.
}
\examples{
# example for tmt_rm
#############################################################################
# Example-1 simple Rasch model 
#############################################################################
dat <- tmt:::sim.rm(theta = 100, b = 10, seed = 1111)
dat.rm <- tmt_rm(dat = dat)
summary(dat.rm)

#############################################################################
# Example-1 for multistage-design
#############################################################################
mstdesign <- "
  M1 =~ c(i1, i2, i3, i4, i5)
  M2 =~ c(i6, i7, i8, i9, i10)
  M3 =~ c(i11, i12, i13, i14, i15)

  # define path
  p1 := M2(0,2) + M1
  p2 := M2(3,5) + M3
"

items <- seq(-1,1,length.out = 15)
names(items) <- paste0("i",1:15)
persons = 1000

dat <- tmt_sim(mstdesign = mstdesign, 
  items = items, persons = persons)
dat.rm <- tmt_rm(dat = dat, mstdesign = mstdesign)
summary(dat.rm)

\dontrun{
  ############################################################################
  # Example-2 simple Rasch model 
  ############################################################################
  dat <- tmt:::sim.rm(theta = 100, b = 10, seed = 1111)
  dat.rm <- tmt_rm(dat = dat)
  summary(dat.rm)

  ############################################################################
  # Example-2 for multistage-design
  ############################################################################
  # also using 'paste' is possible
  mstdesign <- "
    M1 =~ paste0('i',1:5)
    M2 =~ paste0('i',6:10)
    M3 =~ paste0('i',11:15)
    M4 =~ paste0('i',16:20)
    M5 =~ paste0('i',21:25)
    M6 =~ paste0('i',26:30)

    # define path
    p1 := M4(0,2) + M2(0,2) + M1
    p2 := M4(0,2) + M2(3,5) + M3
    p3 := M4(3,5) + M5(0,2) + M3
    p4 := M4(3,5) + M5(3,5) + M6
  "
  items <- seq(-1,1,length.out = 30)
  names(items) <- paste0("i",1:30)
  persons = 1000
  dat <- tmt_sim(mstdesign = mstdesign, 
    items = items, persons = persons)
  dat.rm <- tmt_rm(dat = dat, mstdesign = mstdesign)
  summary(dat.rm)

    ############################################################################
  # Example-3 for cumulative multistage-design
  ############################################################################
  # also using 'paste' is possible
  mstdesign <- "
    M1  =~ paste0('i',21:30)
    M2  =~ paste0('i',11:20)
    M3  =~ paste0('i', 1:10)
    M4  =~ paste0('i',31:40)
    M5  =~ paste0('i',41:50)
    M6  =~ paste0('i',51:60)
    
    # define path
    p1 := M1(0, 5) += M2( 0,10) += M3
    p2 := M1(0, 5) += M2(11,15) += M4
    p3 := M1(6,10) += M5( 6,15) += M4
    p4 := M1(6,10) += M5(16,20) += M6
    "
  items <- seq(-1,1,length.out = 60)
  names(items) <- paste0("i",1:60)
  persons = 1000
  dat <- tmt_sim(mstdesign = mstdesign, 
    items = items, persons = persons)
  dat.rm <- tmt_rm(dat = dat, mstdesign = mstdesign)
  summary(dat.rm)

}
}
\references{
\itemize{
\item Baker, F. B., & Harwell, M. R. (1996). Computing elementary symmetric functions and their derivatives:
   A didactic. \emph{Applied Psychological Measurement, 20}(2), 169-192.
\item Baker, F. B., & Kim, S. H. (2004). \emph{Item response theory: Parameter estimation techniques}. CRC Press.
\item Chen, H., Yamamoto, K., & von Davier, M. (2014). Controlling multistage testing exposure rates in international large-scale assessments. 
   In A. Yan, A. A. von Davier, & C. Lewis (Eds.), \emph{Computerized Multistage Testing: Theory and Applications} (pp. 391-409). 
   New York: CRC Press. https://doi.org/10.1201/b16858
\item Fischer, G. H., & Molenaar, I. W. (Eds.). (2012). \emph{Rasch models: Foundations, recent developments, and applications}. 
	  Springer Science & Business Media.
\item Formann, A. K. (1986). A note on the computation of the second-order derivatives of the elementary symmetric 
	  functions in the Rasch model. \emph{Psychometrika, 51}(2), 335-339.
\item Glas, C.A.W. (1988). The Rasch model and multistage testing. \emph{Journal of Educational Statistics, 13}(1), 45-52.
\item Glas, C.A.W. (2016). Maximum-Likelihood Estimation. In van der Linden, W.J. (Ed.), \emph{Handbook of Item Response Theory: Volume two: Statistical tools.} (pp. 197 - 236). New York: CRC Press.
\item Rasch, G. (1960). \emph{Probabalistic models for some intelligence and attainment tests.} Danmarks
   paedagogiske institut.
\item Steinfeld, J., & Robitzsch, A. (2024). Conditional maximum likelihood estimation in probability-based multistage designs. 
   \emph{Behaviormetrika, 51}(2), 617-634. 
\item Steinfeld, J., Robitzsch, A. (2023). Estimating item parameters in multistage designs with the tmt package in R. 
   \emph{Quantitative and Computational Methods in Behavioral Science, 3}, e10087. https://doi.org/10.5964/qcmb.10087
\item Steinfeld, J., & Robitzsch, A. (2021). Item parameter estimation in multistage designs: A comparison of different estimation approaches for the Rasch model. 
   \emph{Psych, 3}(3), 279-307. https://doi.org/10.3390/psych3030022
\item Verhelst, N.D., Glas, C.A.W., & van der Sluis, A. (1984). Estimation Problems in the Rasch-Model:
   The Basic Symmetric Functions. \emph{Computational Statistics Quarterly, 1}(3), 245-262.
\item Zwitser, R. J., & Maris, G. (2015). Conditional statistical inference with multistage testing designs.
   \emph{Psychometrika, 80}(1), 65-84.
}
}
\seealso{
\link{tmt_lrtest}
}
\author{
Jan Steinfeld
}
