# Tests for scramble_values function

test_that("scramble_values returns correct length and elements", {
  # Test with numeric vector
  set.seed(123)
  x <- 1:5
  result <- scramble_values(x)

  expect_equal(length(result), length(x))
  expect_setequal(result, x)  # Same elements, possibly different order
  expect_type(result, "integer")
})

test_that("scramble_values works with different vector types", {
  # Test with character vector
  set.seed(123)
  chars <- letters[1:5]
  result_chars <- scramble_values(chars)
  expect_equal(length(result_chars), 5)
  expect_setequal(result_chars, chars)
  expect_type(result_chars, "character")

  # Test with logical vector
  set.seed(123)
  logicals <- c(TRUE, FALSE, TRUE, FALSE)
  result_logicals <- scramble_values(logicals)
  expect_equal(length(result_logicals), 4)
  expect_setequal(result_logicals, logicals)
  expect_type(result_logicals, "logical")
})

test_that("scramble_values works with factors", {
    factors <- factor(c("low", "medium", "high"))
    result <- scramble_values(factors)
    expect_s3_class(result, "factor")
    expect_setequal(as.character(result), as.character(factors))
})

test_that("scramble_values produces consistent results with set.seed", {
  # Test deterministic behavior
  set.seed(42)
  x <- 1:10
  result1 <- scramble_values(x)

  set.seed(42)
  result2 <- scramble_values(x)

  expect_equal(result1, result2)
})

test_that("scramble_values actually scrambles (with high probability)", {
  # Test that scrambling actually changes order (probabilistic test)
  set.seed(123)
  x <- 1:20  # Larger vector makes it very unlikely to stay in order
  result <- scramble_values(x)

  # It's extremely unlikely that a 20-element vector stays in the same order
  expect_false(identical(result, x))
})

test_that("scramble_values handles single element vector", {
  # single numeric values are treated by sample() as the upper bound for sampling from 1:x
  # This is a quirk of R's sample() function that affects scramble_values

  set.seed(123)
  x <- c(5)  # Explicitly make it a vector
  result <- scramble_values(x)
  expect_equal(length(result), 1)

  # Test with another single element
  set.seed(42)
  x_single <- c(42)
  result_single <- scramble_values(x_single)
  # Documents actual behavior rather than expected behavior
  expect_equal(length(result_single), 1)
  expect_true(result_single >= 1 && result_single <= 42)
})

test_that("scramble_values validates input correctly", {
  # Test NULL input
  expect_error(
    scramble_values(NULL),
    "Input 'x' cannot be NULL. Please provide a vector.",
    fixed = TRUE
  )

  # Test non-vector input
    expect_error(
        scramble_values(data.frame(a = 1:3)),
        "Input 'x' must be a 1-dimensional vector",
        fixed = TRUE
    )

  # Test empty vector
  expect_error(
    scramble_values(integer(0)),
    "Input 'x' cannot be an empty vector. Please provide a vector with at least one element.",
    fixed = TRUE
  )

  # Test with matrix (should fail as it's not a vector)
  expect_error(
    scramble_values(matrix(1:6, nrow = 2)),
    "Input 'x' must be a 1-dimensional vector",
    fixed = TRUE
  )
})

test_that("scramble_values preserves special values", {
  # Test with NA values
  set.seed(123)
  x_with_na <- c(1, 2, NA, 4, 5)
  result <- scramble_values(x_with_na)
  expect_equal(length(result), 5)
  expect_equal(sum(is.na(result)), 1)
  expect_setequal(result[!is.na(result)], c(1, 2, 4, 5))

  # Test with Inf values
  set.seed(123)
  x_with_inf <- c(1, 2, Inf, 4, 5)
  result <- scramble_values(x_with_inf)
  expect_equal(length(result), 5)
  expect_setequal(result, x_with_inf)
})

test_that("scramble_values works with duplicated values", {
  # Test with duplicated elements
  set.seed(123)
  x_dup <- c(1, 2, 2, 3, 3, 3)
  result <- scramble_values(x_dup)
  expect_equal(length(result), 6)
  # Compare just the values of the tables, ignoring names/attributes
  expect_equal(as.vector(sort(table(result))), as.vector(sort(table(x_dup))))
})


