% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vi.R
\name{vi}
\alias{vi}
\alias{vi.default}
\title{Variable importance}
\usage{
vi(object, ...)

\method{vi}{default}(
  object,
  method = c("model", "firm", "permute", "shap"),
  feature_names = NULL,
  abbreviate_feature_names = NULL,
  sort = TRUE,
  decreasing = TRUE,
  scale = FALSE,
  rank = FALSE,
  ...
)
}
\arguments{
\item{object}{A fitted model object (e.g., a
\link[randomForest:randomForest]{randomForest} object) or an object that inherits
from class \code{"vi"}.}

\item{...}{Additional optional arguments to be passed on to
\link[=vi_model]{vi_model}, \link[=vi_firm]{vi_firm},
\link[=vi_permute]{vi_permute}, or \link[=vi_shap]{vi_shap}; see their
respective help pages for details.}

\item{method}{Character string specifying the type of variable importance
(VI) to compute. Current options are:
\itemize{
\item \code{"model"} (the default), for model-specific VI scores (see
\link[=vi_model]{vi_model} for details).
\item \code{"firm"}, for variance-based VI scores (see \link[=vi_firm]{vi_firm} for
details).
\item \code{"permute"}, for permutation-based VI scores (see
\link[=vi_permute]{vi_permute} for details).
\item \code{"shap"}, for Shapley-based VI scores (see \link[=vi_shap]{vi_shap} for
details).
}}

\item{feature_names}{Character string giving the names of the predictor
variables (i.e., features) of interest.}

\item{abbreviate_feature_names}{Integer specifying the length at which to
abbreviate feature names. Default is \code{NULL} which results in no
abbreviation (i.e., the full name of each feature will be printed).}

\item{sort}{Logical indicating whether or not to order the sort the variable
importance scores. Default is \code{TRUE}.}

\item{decreasing}{Logical indicating whether or not the variable importance
scores should be sorted in descending (\code{TRUE}) or ascending
(\code{FALSE}) order of importance. Default is \code{TRUE}.}

\item{scale}{Logical indicating whether or not to scale the variable
importance scores so that the largest is 100. Default is \code{FALSE}.}

\item{rank}{Logical indicating whether or not to rank the variable
importance scores (i.e., convert to integer ranks). Default is \code{FALSE}.
Potentially useful when comparing variable importance scores across different
models using different methods.}
}
\value{
A tidy data frame (i.e., a \link[tibble:tibble]{tibble} object) with two
columns:
\itemize{
\item \code{Variable} - the corresponding feature name;
\item \code{Importance} - the associated importance, computed as the average change in
performance after a random permutation (or permutations, if \code{nsim > 1}) of
the feature in question.
}

For \link[stats:lm]{lm}/\link[stats:glm]{glm}-like objects, whenever
\code{method = "model"}, the sign (i.e., POS/NEG) of the original coefficient is
also included in a column called \code{Sign}.

If \code{method = "permute"} and \code{nsim > 1}, then an additional column (\code{StDev})
containing the standard deviation of the individual permutation scores for
each feature is also returned; this helps assess the stability/variation of
the individual permutation importance for each feature.
}
\description{
Compute variable importance scores for the predictors in a model.
}
\examples{
#
# A projection pursuit regression example
#

# Load the sample data
data(mtcars)

# Fit a projection pursuit regression model
mtcars.ppr <- ppr(mpg ~ ., data = mtcars, nterms = 1)

# Prediction wrapper that tells vi() how to obtain new predictions from your
# fitted model
pfun <- function(object, newdata) predict(object, newdata = newdata)

# Compute permutation-based variable importance scores
set.seed(1434)  # for reproducibility
(vis <- vi(mtcars.ppr, method = "permute", target = "mpg", nsim = 10,
           metric = "rmse", pred_wrapper = pfun, train = mtcars))

# Plot variable importance scores
vip(vis, include_type = TRUE, all_permutations = TRUE,
    geom = "point", aesthetics = list(color = "forestgreen", size = 3))

#
# A binary classification example
#
\dontrun{
library(rpart)  # for classification and regression trees

# Load Wisconsin breast cancer data; see ?mlbench::BreastCancer for details
data(BreastCancer, package = "mlbench")
bc <- subset(BreastCancer, select = -Id)  # for brevity

# Fit a standard classification tree
set.seed(1032)  # for reproducibility
tree <- rpart(Class ~ ., data = bc, cp = 0)

# Prune using 1-SE rule (e.g., use `plotcp(tree)` for guidance)
cp <- tree$cptable
cp <- cp[cp[, "nsplit"] == 2L, "CP"]
tree2 <- prune(tree, cp = cp)  # tree with three splits

# Default tree-based VIP
vip(tree2)

# Computing permutation importance requires a prediction wrapper. For
# classification, the return value depends on the chosen metric; see
# `?vip::vi_permute` for details.
pfun <- function(object, newdata) {
  # Need vector of predicted class probabilities when using  log-loss metric
  predict(object, newdata = newdata, type = "prob")[, "malignant"]
}

# Permutation-based importance (note that only the predictors that show up
# in the final tree have non-zero importance)
set.seed(1046)  # for reproducibility
vi(tree2, method = "permute", nsim = 10, target = "Class", train = bc,
   metric = "logloss", pred_wrapper = pfun, reference_class = "malignant")

# Equivalent (but not sorted)
set.seed(1046)  # for reproducibility
vi_permute(tree2, nsim = 10, target = "Class", metric = "logloss",
           pred_wrapper = pfun, reference_class = "malignant")
}
}
