// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/wm/overview/birch/birch_bar_context_menu_model.h"

#include <utility>

#include "ash/birch/birch_coral_provider.h"
#include "ash/birch/coral_util.h"
#include "ash/constants/ash_features.h"
#include "ash/constants/ash_pref_names.h"
#include "ash/resources/vector_icons/vector_icons.h"
#include "ash/session/session_controller_impl.h"
#include "ash/shell.h"
#include "ash/strings/grit/ash_strings.h"
#include "ash/wm/overview/overview_utils.h"
#include "chromeos/ash/components/geolocation/system_location_provider.h"
#include "components/prefs/pref_service.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/models/menu_separator_types.h"
#include "ui/views/controls/menu/menu_types.h"

namespace ash {
namespace {

// Returns whether the weather item should be enabled based on the geolocation
// permission. See BirchWeatherProvider.
bool IsWeatherAllowedByGeolocation() {
  return SystemLocationProvider::GetInstance()
      ->IsGeolocationUsageAllowedForSystem();
}

}  // namespace

BirchBarContextMenuModel::BirchBarContextMenuModel(
    ui::SimpleMenuModel::Delegate* delegate,
    Type type)
    : ui::SimpleMenuModel(delegate) {
  // Show suggestions option is in both expanded and collapsed menu.
  AddItem(std::to_underlying(CommandId::kShowSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_SHOW_SUGGESTIONS));

  // Expanded menu also has customizing suggestions options.
  if (type == Type::kExpandedBarMenu) {
    AddSeparator(ui::MenuSeparatorType::NORMAL_SEPARATOR);

    if (features::IsCoralFeatureEnabled()) {
      AddItem(std::to_underlying(CommandId::kCoralSuggestions),
              l10n_util::GetStringUTF16(IDS_ASH_BIRCH_CORAL_BAR_MENU_ITEM));
      auto coral_index =
          GetIndexOfCommandId(std::to_underlying(CommandId::kCoralSuggestions));
      SetEnabledAt(coral_index.value(),
                   BirchCoralProvider::Get()->IsCoralServiceAvailable());
    }

    bool enabled = IsWeatherAllowedByGeolocation();
    std::u16string weather_label =
        enabled ? l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_WEATHER)
                : l10n_util::GetStringUTF16(
                      IDS_ASH_BIRCH_MENU_WEATHER_NOT_AVAILABLE);
    AddItem(std::to_underlying(CommandId::kWeatherSuggestions), weather_label);
    auto weather_index =
        GetIndexOfCommandId(std::to_underlying(CommandId::kWeatherSuggestions));
    SetEnabledAt(weather_index.value(), enabled);
    if (!enabled) {
      SetMinorText(weather_index.value(),
                   l10n_util::GetStringUTF16(
                       IDS_ASH_BIRCH_MENU_WEATHER_NOT_AVAILABLE_TOOLTIP));
    }

    AddItem(std::to_underlying(CommandId::kCalendarSuggestions),
            l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_CALENDAR));
    AddItem(std::to_underlying(CommandId::kDriveSuggestions),
            l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_DRIVE));
    AddItem(std::to_underlying(CommandId::kChromeTabSuggestions),
            l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_CHROME_BROWSER));
    AddItem(std::to_underlying(CommandId::kMediaSuggestions),
            l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_MEDIA));
    AddSeparator(ui::MenuSeparatorType::NORMAL_SEPARATOR);
    AddItemWithIcon(std::to_underlying(CommandId::kReset),
                    l10n_util::GetStringUTF16(IDS_ASH_BIRCH_MENU_RESET),
                    CreateIconForMenuItem(kResetIcon));
  }
}

BirchBarContextMenuModel::~BirchBarContextMenuModel() = default;

}  // namespace ash
