/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; fill-column: 100 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#pragma once

#include <sal/config.h>

#include <config_oauth2.h>

#if defined OAUTH2REQUEST_SUPPORTED

#include <rtl/ustring.hxx>

#include <memory>

/// The class that incapsulates OAuth2 interaction.
/// It is instantiated with an auth URL, usually generated by libcmis.
/// 1. It extracts redirect URL from auth URL using OAUTH2_REDIRECT_URI_PREFIX from config_oauth2.h.
/// 2. It creates a minimal localhost webserver listening to the redirect URL.
/// 3. It opens a web browser with the auth URL (the user will authenticate there, and authorize
///    the application for the access).
/// 4. It creates a message box with a Cancel button, allowing to stop the procedure.
/// 5. The auth provider (opened at #3) will eventually redirect to the localhost address listened
///    by the webserver created at #2. It may be a success, or a failure.
/// 6. The task of webserver is to handle the query of the redirect URL (from #5). If it's success,
///    it must store the auth code, close message box (#4) with success code, and exit. In case of
///    failure, it must close the message box with Cancel (which will eventually terminate the
///    webserver at #7). In both success and failure cases, it must send a web page with a text to
///    close the page, because it's impossible to auto-close it from the page script.
///    Here it may be required to distinguish the OAuth2 providers. The OAUTH2_REDIRECT_URI_PREFIX
///    allows to do that: the rest of the redirect URL is supposed to define the provider, as
///    defined in config_oauth2.h.
/// 7. If the message box (#4) is dismissed using Cancel, the webserver is terminated.
class OAuth2Request
{
public:
    OAuth2Request(const OUString& auth_url);
    ~OAuth2Request();

    void run();
    const OUString& getRetCode() const;

private:
    struct Impl;
    std::unique_ptr<Impl> impl;
};

#endif // OAUTH2REQUEST_SUPPORTED

/* vim:set shiftwidth=4 softtabstop=4 expandtab cinoptions=b1,g0,N-s cinkeys+=0=break: */
