//
// Created by uos on 2022/3/18.
//

#include "ItemFrame.h"
#include "utils/Utils.h"
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QMouseEvent>
#include <DFontSizeManager>
#include <QDebug>

ItemFrame::ItemFrame(const QString &title, const QString &description, const QString &icon, QWidget *parent)
    : DFrame(parent)
{
    setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
    setAttribute(Qt::WA_Hover,true);
    setObjectName("ItemFrame");
    setBackgroundRole(DPalette::ItemBackground);
    setLineWidth(0);

    m_title = new DLabel;
    m_title->setText(title);
    m_description = new DTipLabel;
    m_description->setText(description);
    m_descriptionText = description;

    m_icon = new DLabel;
    m_icon->setAlignment(Qt::AlignCenter);
//    m_icon->setPixmap(Utils::hidpiPixmap(icon, QSize(96, 96)));
    m_radioPixmap = Utils::hidpiPixmap(":/resources/icons/radio_button.svg", QSize(20, 20));
    m_checkIcon = new DLabel;
    m_checkIcon->setVisible(true);
    m_checkIcon->setFixedSize(QSize(20, 20));
    DFontSizeManager::instance()->bind(m_title, DFontSizeManager::T4);
}

void ItemFrame::setItemFrameAccessibleName(const QString &name)
{
    this->setAccessibleName(name);
}

void ItemFrame::setTitleAccessibleName(const QString &name)
{
    if (nullptr != m_title) {
        m_title->setAccessibleName(name);
    }
}

void ItemFrame::setDescriptionAccessibleName(const QString &name)
{
    if (nullptr != m_description) {
        m_description->setAccessibleName(name);
    }
}

void ItemFrame::setIconAccessibleName(const QString &name)
{
    if (nullptr != m_icon) {
        m_icon->setAccessibleName(name);
    }
}

void ItemFrame::setCheckIconAccessibleName(const QString &name)
{
    if (nullptr != m_checkIcon) {
        m_checkIcon->setAccessibleName(name);
    }
}

void ItemFrame::setIconPixmap(const QString &iconRes, const QSize &iconSize)
{
    if (nullptr != m_icon) {
        m_icon->setPixmap(Utils::hidpiPixmap(iconRes, iconSize));
    }
}

void ItemFrame::setSelected(bool select)
{
    m_selected = select;
}

bool ItemFrame::isSelected()
{
    return m_selected;
}

QString ItemFrame::getDescriptionOriginText()
{
    return m_descriptionText;
}

void ItemFrame::setLayoutStype(int layout)
{
    m_layout = layout;
}

void ItemFrame::initLayout()
{
    QList<int> vLayout = {
            UI::UI_LAYOUT_BACKUP_VERTICAL,
            UI::UI_LAYOUT_RESTORE_VERTICAL,
            UI::UI_LAYOUT_ADVANCE_VERTICAL
    };

    QList<int> hLayout = {
            UI::UI_LAYOUT_BACKUP_HORIZONTAL,
            UI::UI_LAYOUT_RESTORE_HORIZONTAL,
            UI::UI_LAYOUT_ADVANCE_HORIZONTAL
    };

    if (vLayout.contains(m_layout)) {
        this->initVerticalLayout();
    } else if (hLayout.contains(m_layout)) {
        this->initHorizontalLayout();
    } else {
        qWarning()<<"ItemFrame::initLayout, invalid layout, m_layout = "<<m_layout;
    }
}

void ItemFrame::initHorizontalLayout()
{
    QVBoxLayout *centralLayout = new QVBoxLayout(this);
    auto hBoxLayout = new QHBoxLayout;

    hBoxLayout->addStretch();
    hBoxLayout->addWidget(m_checkIcon);

    centralLayout->addLayout(hBoxLayout);
    //centralLayout->addSpacing(10);
    centralLayout->addWidget(m_icon, 0, Qt::AlignHCenter);
    centralLayout->addSpacing(10);

    QVBoxLayout *vLayout = new QVBoxLayout();

    m_title->setAlignment(Qt::AlignTop | Qt::AlignHCenter);
    m_title->setWordWrap(true);
    //m_title->setFixedHeight(60);

    QFont font = m_title->font();
    font.setWeight(QFont::DemiBold);
    font.setPixelSize(18);
    m_title->setFont(font);
    DFontSizeManager::instance()->bind(m_title, DFontSizeManager::T5);

    vLayout->addWidget(m_title);
    if (!m_showDescription) {
        m_description->hide();
    }
    m_description->setAlignment(Qt::AlignTop | Qt::AlignHCenter);
    m_description->setWordWrap(true);
    //m_description->setElideMode(Qt::TextElideMode::ElideRight);
    QFont descriptionFont = m_description->font();
    descriptionFont.setWeight(QFont::Normal);
    descriptionFont.setPixelSize(12);
    m_description->setFont(descriptionFont);
    DFontSizeManager::instance()->bind(m_description, DFontSizeManager::T6);
    vLayout->addWidget(m_description);
    centralLayout->addLayout(vLayout);
    centralLayout->setSpacing(0);
}

void ItemFrame::initVerticalLayout()
{
    m_title->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    m_title->setWordWrap(true);
//    m_title->setFixedHeight(60);

    QFont font = m_title->font();
    font.setWeight(QFont::DemiBold);
    font.setPixelSize(17);
    m_title->setFont(font);
    DFontSizeManager::instance()->bind(m_title, DFontSizeManager::T5);
    QPalette titlePalette;
    titlePalette.setColor(QPalette::BrightText, QColor(qRgba(0,0,0,0.85)));
    m_title->setPalette(titlePalette);

    m_description->setAlignment(Qt::AlignLeft);
    QPalette desPalette;
    desPalette.setColor(QPalette::BrightText, QColor(qRgba(0,0,0,0.7)));
    m_description->setPalette(desPalette);
    m_description->setWordWrap(true);
    m_description->setFixedWidth(490); // 占位置，不然icon左右两个有很多空格
    m_description->setElideMode(Qt::TextElideMode::ElideRight);
    QFont descriptionFont = m_description->font();
    descriptionFont.setWeight(QFont::ExtraLight);
    descriptionFont.setPixelSize(12);
    m_description->setFont(descriptionFont);
    DFontSizeManager::instance()->bind(m_description, DFontSizeManager::T8);

    QVBoxLayout *centralLayout = new QVBoxLayout(this);
    centralLayout->setContentsMargins(0, 0, 0, 0);

    auto iconHBoxLayout = new QHBoxLayout;
    iconHBoxLayout->setSpacing(0);
    iconHBoxLayout->setContentsMargins(10, 0, 0, 0);
    iconHBoxLayout->addWidget(m_icon, 0, Qt::AlignLeft);
//    iconHBoxLayout->addStretch();

    QVBoxLayout *textVLayout = new QVBoxLayout();
    textVLayout->setSpacing(0);
  //  textVLayout->setContentsMargins(0, 10, 0, 0);
    textVLayout->addStretch();
    textVLayout->addWidget(m_title);
    textVLayout->addSpacing(3);
    textVLayout->addWidget(m_description);
    textVLayout->addStretch();
   // textVLayout->setContentsMargins(0, 30, 0, 10);

    QVBoxLayout *checkBoxVLayout = new QVBoxLayout;
    if (Utils::isDirectionRTL()) {
        checkBoxVLayout->setContentsMargins(10, 10, 0, 0);
    } else {
        checkBoxVLayout->setContentsMargins(0, 10, 10, 0);
    }
    checkBoxVLayout->addWidget(m_checkIcon, 0, Qt::AlignTop);
    checkBoxVLayout->addStretch();

    iconHBoxLayout->addLayout(textVLayout);
    auto hBoxLayout = new QHBoxLayout;
    hBoxLayout->addLayout(iconHBoxLayout);
    hBoxLayout->addLayout(checkBoxVLayout);

    centralLayout->addLayout(hBoxLayout);
//    centralLayout->setSpacing(0);
    this->setLayout(centralLayout);
}

void ItemFrame::setCheckState(bool checked)
{
    m_bChecked = checked;
    if (m_bChecked) {
        m_checkIcon->setVisible(true);
        m_checkIcon->setPixmap(m_radioPixmap);
        m_bChecked = true;
    } else {
        m_checkIcon->setPixmap(QPixmap());
        m_bChecked = false;
    }
}

bool ItemFrame::checkState()
{
    return m_bChecked;
}

void ItemFrame::showDescription(bool isShow)
{
    m_showDescription = isShow;
    m_description->setVisible(isShow);
}

bool ItemFrame::event(QEvent *e)
{
    if (e->type() == QEvent::HoverEnter) {
        setBackgroundRole(DPalette::ObviousBackground);

        QString m_desText = m_descriptionText;
        QFontMetrics fontWidth(m_description->font());
        QString elideNote = fontWidth.elidedText(m_desText, Qt::ElideRight, m_description->width(), 
            Qt::AlignLeft | Qt::AlignBottom | Qt::TextSingleLine);
        if (elideNote.indexOf("…") != -1) {
            m_description->setToolTip(m_desText);
        } else {
            m_description->setToolTip("");
        }
    } else if (e->type() == QEvent::HoverLeave) {
        setBackgroundRole(DPalette::ItemBackground);
    } else if (e->type() == QEvent::MouseButtonRelease) {
        if (this->isEnabled()) {
            if (!m_bChecked) {
               m_checkIcon->setVisible(true);
               m_checkIcon->setPixmap(m_radioPixmap);
               m_bChecked = true;
           }
           Q_EMIT clicked(m_bChecked);
        }
    }
    return QFrame::event(e);
}
