//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.checkers.checks;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.checkers.CifCheck;
import org.eclipse.escet.cif.checkers.CifCheckViolations;
import org.eclipse.escet.cif.common.CifControllerPropertiesAnnotationUtils;
import org.eclipse.escet.cif.common.CifDocAnnotationUtils;
import org.eclipse.escet.cif.common.CifPartialSpecAnnotationUtils;
import org.eclipse.escet.cif.common.CifReachabilityRequirementAnnotationUtils;
import org.eclipse.escet.cif.common.CifStateAnnotationUtils;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.annotations.Annotation;

/** CIF check that does not allow certain annotations. */
public class AnnoNoSpecificAnnosCheck extends CifCheck {
    /** The names of annotations to disallow. */
    private final Set<String> disalloweds;

    /**
     * Constructor for the {@link AnnoNoSpecificAnnosCheck} class.
     *
     * @param disalloweds The annotations to disallow.
     */
    public AnnoNoSpecificAnnosCheck(NoSpecificAnno... disalloweds) {
        this(EnumSet.copyOf(Arrays.asList(disalloweds)));
    }

    /**
     * Constructor for the {@link AnnoNoSpecificAnnosCheck} class.
     *
     * @param disalloweds The annotations to disallow.
     */
    public AnnoNoSpecificAnnosCheck(Set<NoSpecificAnno> disalloweds) {
        this.disalloweds = Collections.unmodifiableSet(
                disalloweds.stream().map(disallow -> disallow.name).collect(Collectors.toSet()));
    }

    @Override
    protected void preprocessAnnotatedObject(AnnotatedObject obj, CifCheckViolations violations) {
        for (Annotation anno: obj.getAnnotations()) {
            if (disalloweds.contains(anno.getName())) {
                violations.add(anno, "An annotation with name \"%s\" is used", anno.getName());
            }
        }
    }

    /** The annotation to disallow. */
    public static enum NoSpecificAnno {
        /** Disallow controller properties annotations. */
        CONTROLLER_PROPERTIES(CifControllerPropertiesAnnotationUtils.NAME),

        /** Disallow documentation annotations. */
        DOC(CifDocAnnotationUtils.NAME),

        /** Disallow partial specification annotations. */
        PARTIAL_SPEC(CifPartialSpecAnnotationUtils.NAME),

        /** Disallow reachability requirement annotations. */
        REQUIREMENT_REACHABLE(CifReachabilityRequirementAnnotationUtils.NAME),

        /** Disallow state annotations. */
        STATE(CifStateAnnotationUtils.NAME);

        /** The name of the annotation. */
        private final String name;

        /**
         * Constructor for the {@link NoSpecificAnno} enum.
         *
         * @param name The name of the annotation.
         */
        private NoSpecificAnno(String name) {
            this.name = name;
        }
    }
}
