//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion.bitvectors;

import static org.eclipse.escet.common.java.Pair.pair;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.math.BigInteger;
import java.util.Arrays;

import org.eclipse.escet.cif.bdd.spec.CifBddDomain;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Pair;

import com.github.javabdd.BDD;
import com.github.javabdd.BDDFactory;

/**
 * BDD bit vector.
 *
 * @param <T> The type of BDD bit vector.
 * @param <TC> The type of BDD bit vector and carry.
 */
public abstract class BddBitVector<T extends BddBitVector<T, TC>, TC extends BddBitVectorAndCarry<T, TC>> {
    /** The BDD factory to use. */
    protected BDDFactory factory;

    /**
     * The BDDs for each of the bits of the bit vector.
     */
    protected BDD[] bits;

    /**
     * Constructor for the {@link BddBitVector} class.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @throws IllegalArgumentException If the length is negative, or not supported by the bit vector representation.
     */
    protected BddBitVector(BDDFactory factory, int length) {
        // Precondition check.
        if (length < getMinimumLength()) {
            throw new IllegalArgumentException(fmt("Length is less than %d.", getMinimumLength()));
        }

        // Create.
        this.factory = factory;
        bits = new BDD[length];
    }

    /**
     * Returns the minimum length of bit vectors of this bit vector representation.
     *
     * @return The minimum length.
     */
    protected abstract int getMinimumLength();

    /**
     * Creates an empty BDD bit vector (bits are all {@code null}) of the same representation as this bit vector, and
     * with the same {@link #factory}.
     *
     * @param length The number of bits of the bit vector.
     * @return The new bit vector.
     */
    protected abstract T createEmpty(int length);

    /**
     * Creates a copy of this bit vector. A new instance of the bit vector is created, that has the same length. Each
     * bit is {@link BDD#id copied} to the new bit vector.
     *
     * @return The copy.
     */
    public T copy() {
        T vector = createEmpty(bits.length);
        for (int i = 0; i < bits.length; i++) {
            vector.bits[i] = bits[i].id();
        }
        return vector;
    }

    /**
     * Modifies this bit vector to represent the given other bit vector. This bit vector and the given other bit vector
     * don't need to have the same length. This bit vector is first {@link #free freed}, then the bits of the other bit
     * vector are moved to this bit vector. The other bit vector is essentially {@link #free freed}, and can no longer
     * be used.
     *
     * @param other The other bit vector.
     */
    public void replaceBy(T other) {
        free();
        this.factory = other.factory;
        this.bits = other.bits;
        other.factory = null;
        other.bits = null;
    }

    /**
     * Make the two given bit vectors compatible, in terms of having the same representation, converting one of them
     * from unsigned representation to signed representation if needed.
     *
     * <p>
     * If both bit vectors already have the same representation, either being both in unsigned representation, or both
     * in signed representation, they are returned as is. If one of the bit vectors is in unsigned representation and
     * the other is in signed representation, then the unsigned bit vector is converted to signed representation, the
     * unsigned bit vector is {@link #free freed}, the signed bit vector is kept as is, and both bit vectors are
     * returned in signed representation.
     * </p>
     *
     * @param vector1 The first bit vector. Is freed if it is converted from unsigned to signed representation.
     * @param vector2 The second bit vector. Is freed if it is converted from unsigned to signed representation.
     * @return The first and second bit vectors, in compatible representations.
     */
    public static Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> ensureCompatible(BddBitVector<?, ?> vector1,
            BddBitVector<?, ?> vector2)
    {
        if (vector1 instanceof UnsignedBddBitVector uVector1) {
            if (vector2 instanceof UnsignedBddBitVector) {
                // Already compatible.
                return pair(vector1, vector2);
            } else {
                // Convert first vector, keep second vector.
                Assert.check(vector2 instanceof SignedBddBitVector);
                SignedBddBitVector signedVector1 = SignedBddBitVector.createFromUnsignedBitVector(uVector1);
                uVector1.free();
                return pair(signedVector1, vector2);
            }
        } else {
            Assert.check(vector1 instanceof SignedBddBitVector);
            if (vector2 instanceof UnsignedBddBitVector uVector2) {
                // Keep first vector, convert second vector.
                SignedBddBitVector signedVector2 = SignedBddBitVector.createFromUnsignedBitVector(uVector2);
                uVector2.free();
                return pair(vector1, signedVector2);
            } else {
                // Already compatible.
                Assert.check(vector2 instanceof SignedBddBitVector);
                return pair(vector1, vector2);
            }
        }
    }

    /**
     * Ensure that two given bit vectors have the same length as the longest of the two. One of the two bit vectors may
     * be resized to a larger length if needed.
     *
     * @param vector1 The first bit vector. May be modified in-place.
     * @param vector2 The second bit vector. May be modified in-place.
     */
    public static void ensureSameLength(BddBitVector<?, ?> vector1, BddBitVector<?, ?> vector2) {
        ensureSameLength(vector1, vector2, 0);
    }

    /**
     * Ensure that two given bit vectors have the same length as the longest of the two, optionally incremented with
     * some extra bits. One of the two bit vectors, or both, may be resized to a larger length if needed.
     *
     * @param vector1 The first bit vector. May be modified in-place.
     * @param vector2 The second bit vector. May be modified in-place.
     * @param extraBits The number of extra bits. Must be at least zero.
     * @throws IllegalArgumentException If the number of extra bits is negative.
     */
    public static void ensureSameLength(BddBitVector<?, ?> vector1, BddBitVector<?, ?> vector2, int extraBits) {
        // Precondition check.
        if (extraBits < 0) {
            throw new IllegalArgumentException("The number of extra bits is negative.");
        }

        // Get the new length, accounting for the requested extra bits.
        int length = Math.max(vector1.length(), vector2.length());
        length += extraBits;

        // Resize to the new length.
        vector1.resize(length);
        vector2.resize(length);
    }

    /**
     * Returns the length of the bit vector, in number of bits.
     *
     * @return The length of the bit vector, in number of bits.
     */
    public int length() {
        return bits.length;
    }

    /**
     * Returns the value count, the number of values that can be represented by the bit vector (if it can be represented
     * as a Java 'int').
     *
     * @return The value count.
     * @throws IllegalStateException If the bit vector has more than 30 bits.
     */
    int countInt() {
        // For 31 bits or more, the count is too high to be represented as a Java 'int'.
        if (bits.length > 30) {
            throw new IllegalStateException("More than 30 bits in vector.");
        }

        // Return the count.
        return 1 << bits.length;
    }

    /**
     * Returns the value count, the number of values that can be represented by the bit vector (if it can be represented
     * as a Java 'long').
     *
     * @return The value count.
     * @throws IllegalStateException If the bit vector has more than 62 bits.
     */
    long countLong() {
        // For 63 bits or more, the count is too high to be represented as a Java 'long'.
        if (bits.length > 62) {
            throw new IllegalStateException("More than 62 bits in vector.");
        }

        // Return the count.
        return 1L << bits.length;
    }

    /**
     * Returns the BDD for the bit with the given index. The lowest bit is at index zero.
     *
     * @param index The 0-based index of the bit.
     * @return The BDD for the bit with the given index.
     * @throws IndexOutOfBoundsException If the index is negative, or greater than or equal to {@link #length}.
     */
    public BDD getBit(int index) {
        return bits[index];
    }

    /**
     * Returns the lower bound of this bit vector, that is, its lowest representable value.
     *
     * @return The lower bound.
     */
    public abstract BigInteger getLower();

    /**
     * Returns the lower bound of this bit vector, that is, its lowest representable value.
     *
     * @return The lower bound.
     * @throws ArithmeticException If the lower bound doesn't fit in a Java 'int'.
     */
    public abstract int getLowerInt();

    /**
     * Returns the upper bound of this bit vector, that is, its highest representable value.
     *
     * @return The upper bound.
     */
    public abstract BigInteger getUpper();

    /**
     * Returns the upper bound of this bit vector, that is, its highest representable value.
     *
     * @return The upper bound.
     * @throws ArithmeticException If the upper bound doesn't fit in a Java 'int'.
     */
    public abstract int getUpperInt();

    /**
     * Returns the value represented by the bit vector, if it is a constant bit vector, or {@code null} otherwise.
     *
     * @return The value represented by the bit vector, or {@code null}.
     * @throws IllegalStateException If the bit vector doesn't fit in a Java 'int'.
     */
    public abstract Integer getInt();

    /**
     * Returns the value represented by the bit vector, if it is a constant bit vector, or {@code null} otherwise.
     *
     * @return The value represented by the bit vector, or {@code null}.
     * @throws IllegalStateException If the bit vector doesn't fit in a Java 'long'.
     */
    public abstract Long getLong();

    /**
     * Updates the bit vector, setting the bit with the given index to a given BDD. The previous BDD stored at the bit
     * is first {@link BDD#free freed}.
     *
     * @param idx The 0-based index of the bit to set.
     * @param bdd The BDD to use as new value for the given bit.
     * @throws IndexOutOfBoundsException If the index is negative, or greater than or equal to {@link #length}.
     */
    public void setBit(int idx, BDD bdd) {
        bits[idx].free();
        bits[idx] = bdd;
    }

    /**
     * Updates the bit vector, setting the bit with the given index to a given value. The previous BDD stored at the bit
     * is first {@link BDD#free freed}.
     *
     * @param idx The 0-based index of the bit to set.
     * @param value The boolean value to use as new value for the given bit.
     * @throws IndexOutOfBoundsException If the index is negative, or greater than or equal to {@link #length}.
     */
    public void setBit(int idx, boolean value) {
        setBit(idx, value ? factory.one() : factory.zero());
    }

    /**
     * Updates the bit vector, setting each bit to the given value. The BDDs that were previously stored, are first
     * {@link BDD#free freed}.
     *
     * @param value The value to set for each bit.
     */
    public void setBitsToValue(boolean value) {
        for (int i = 0; i < bits.length; i++) {
            bits[i].free();
            bits[i] = value ? factory.one() : factory.zero();
        }
    }

    /**
     * Updates the bit vector to represent the given value. The BDDs that were previously stored, are first
     * {@link BDD#free freed}.
     *
     * @param value The value to which to set the bit vector.
     * @throws IllegalArgumentException If the value isn't supported by the bit vector representation.
     * @throws IllegalArgumentException If the value doesn't fit within the bit vector.
     */
    public abstract void setInt(int value);

    /**
     * Updates the bit vector to represent the given BDD domain. The BDDs that were previously stored, are first
     * {@link BDD#free freed}.
     *
     * @param domain The domain to which to set the bit vector.
     * @throws IllegalArgumentException If the length of the domain differs from the length of the bit vector.
     */
    public void setDomain(CifBddDomain domain) {
        // Precondition check.
        int varCnt = domain.getVarCount();
        if (varCnt != bits.length) {
            throw new IllegalArgumentException("Domain length is not equal to the bit vector length.");
        }

        // Set domain.
        int[] vars = domain.getVarIndices();
        for (int i = 0; i < varCnt; i++) {
            bits[i].free();
            bits[i] = factory.ithVar(vars[i]);
        }
    }

    /**
     * Resizes the bit vector to have the given length. If the new length is larger than the current length, the way the
     * additional (most significant) bits are set depends on the bit vector representation. If the new length is smaller
     * than the current length, the most significant bits are dropped. The BDDs for dropped bits, are {@link BDD#free
     * freed}.
     *
     * @param length The new length of the bit vector.
     * @throws IllegalArgumentException If the new length is negative, or not supported by the bit vector
     *     representation.
     */
    public abstract void resize(int length);

    /**
     * Resizes this bit vector to have the minimum-required length to still represent the same value(s). That is, it
     * removes all unnecessary most-significant bits from this bit vector. If this bit vector already has the
     * minimum-required length, it is not modified in any way. If this bit vector is longer than its minimum-required
     * length, some of its unnecessary most-significant bits are {@link BDD#free freed} and dropped.
     *
     * @return This bit vector, resized to its minimum-rquired length.
     */
    public abstract T shrink();

    /**
     * Negates this bit vector. This operation returns a new bit vector. The vector that is negated is neither modified
     * nor {@link #free freed}.
     *
     * @return The result.
     * @throws UnsupportedOperationException If the bit vector representation doesn't support the operation.
     */
    public abstract TC negate();

    /**
     * Computes the absolute value of this bit vector. This operation returns a new bit vector. The vector for which the
     * absolute value is computed is neither modified nor {@link #free freed}.
     *
     * @return The result.
     */
    public abstract TC abs();

    /**
     * Returns the sign of this bit vector: {@code -1} if the bit vector is negative, {@code 0} if it is zero, and
     * {@code 1} if its positive. This operation returns a new bit vector. The vector for which the sign is computed is
     * neither modified nor {@link #free freed}.
     *
     * @return The result.
     */
    public abstract T sign();

    /**
     * Adds the given bit vector to this bit vector. This bit vector and the given bit vector must have the same
     * representation. This operation returns a new bit vector and carry. The bit vectors on which the operation is
     * performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to add to this bit vector.
     * @return The result.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract TC add(T other);

    /**
     * Adds the given bit vector to this bit vector. This bit vector and the given bit vector must have the same
     * representation. This operation returns a new bit vector and carry. The bit vectors on which the operation is
     * performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to add to this bit vector.
     * @return The result.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BddBitVectorAndCarry<?, ?> addAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.add((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.add((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Subtracts the given bit vector from this bit vector. This bit vector and the given bit vector must have the same
     * representation. This operation returns a new bit vector and carry. The bit vectors on which the operation is
     * performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to subtract from this bit vector.
     * @return The result.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract TC subtract(T other);

    /**
     * Subtracts the given bit vector from this bit vector. This bit vector and the given bit vector must have the same
     * representation. This operation returns a new bit vector and carry. The bit vectors on which the operation is
     * performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to subtract from this bit vector.
     * @return The result.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BddBitVectorAndCarry<?, ?> subtractAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.subtract((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.subtract((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Multiplies this bit vector by the given bit vector. This bit vector and the given bit vector must have the same
     * representation. This operation returns a new bit vector and carry. The bit vectors on which the operation is
     * performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector by which to multiply this bit vector.
     * @return The result.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract TC multiply(T other);

    /**
     * Multiplies this bit vector by the given bit vector. This bit vector and the given bit vector must have the same
     * representation. This operation returns a new bit vector and carry. The bit vectors on which the operation is
     * performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector by which to multiply this bit vector.
     * @return The result.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BddBitVectorAndCarry<?, ?> multiplyAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.multiply((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.multiply((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Computes the quotient ('div' result) of dividing this vector (the dividend) by the given value (the divisor).
     * This operation returns a new bit vector. The bit vector on which the operation is performed is neither modified
     * nor {@link #free freed}.
     *
     * @param divisor The value by which to divide this bit vector.
     * @return The quotient ('div' result).
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the operation can't be performed due to additional bit vector
     *     representation-specific constraints.
     */
    public abstract T div(int divisor);

    /**
     * Computes the remainder ('mod' result) of dividing this vector (the dividend) by the given value (the divisor).
     * This operation returns a new bit vector. The bit vector on which the operation is performed is neither modified
     * nor {@link #free freed}.
     *
     * @param divisor The value by which to divide this bit vector.
     * @return The remainder ('mod' result).
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalStateException If the operation can't be performed due to additional bit vector
     *     representation-specific constraints.
     */
    public abstract T mod(int divisor);

    /**
     * Computes the bit vector resulting from shifting this bit vector {@code amount} bits to the left. The given
     * {@code carry} is shifted in {@code amount} times. This operation returns a new bit vector. The bit vector on
     * which the operation is performed is neither modified nor {@link #free freed}.
     *
     * @param amount The amount of bits to shift.
     * @param carry The carry to shift in. Only copies are used.
     * @return The shifted bit vector.
     * @throws IllegalArgumentException If the shift amount is negative.
     */
    public T shiftLeft(int amount, BDD carry) {
        // Precondition check.
        if (amount < 0) {
            throw new IllegalArgumentException("Amount is negative.");
        }

        // Compute result.
        T result = createEmpty(bits.length);

        int numberOfCarryBits = Math.min(bits.length, amount);
        int i = 0;
        for (; i < numberOfCarryBits; i++) {
            result.bits[i] = carry.id();
        }
        for (; i < bits.length; i++) {
            result.bits[i] = bits[i - amount].id();
        }

        return result;
    }

    /**
     * Computes the bit vector resulting from shifting this bit vector {@code amount} bits to the right. The given
     * {@code carry} is shifted in {@code amount} times. This operation returns a new bit vector. The bit vector on
     * which the operation is performed is neither modified nor {@link #free freed}.
     *
     * @param amount The amount of bits to shift.
     * @param carry The carry to shift in. Only copies are used.
     * @return The shifted bit vector.
     * @throws IllegalArgumentException If the shift amount is negative.
     */
    public T shiftRight(int amount, BDD carry) {
        // Precondition check.
        if (amount < 0) {
            throw new IllegalArgumentException("Amount is negative.");
        }

        // Compute result.
        T result = createEmpty(bits.length);

        int numberOfPreservedBits = Math.max(0, bits.length - amount);
        int i = 0;
        for (; i < numberOfPreservedBits; i++) {
            result.bits[i] = bits[i + amount].id();
        }
        for (; i < bits.length; i++) {
            result.bits[i] = carry.id();
        }

        return result;
    }

    /**
     * Computes an if-then-else with the given 'if' condition, 'then' bit vector and 'else' bit vector. The 'then' and
     * 'else' bit vectors must have the same representation. This operation returns a new bit vector of the same
     * representation as the 'then' and 'else' bit vectors. The condition and bit vectors on which the operation are
     * performed are neither modified nor {@link #free freed}.
     *
     * @param <T> The type of the bit vector representations.
     * @param <TC> The type of the bit vector and carry representations.
     * @param condition The 'if' condition.
     * @param thenVector The 'then' bit vector.
     * @param elseVector The 'else' bit vector.
     * @return The result.
     * @throws IllegalArgumentException If the 'then' bit vector and 'else' bit vector have a different length.
     */
    @SuppressWarnings("unchecked")
    public static <T extends BddBitVector<T, TC>, TC extends BddBitVectorAndCarry<T, TC>> T ifThenElse(BDD condition,
            T thenVector, T elseVector)
    {
        // Precondition check.
        if (thenVector.bits.length != elseVector.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Initialize result.
        T rslt;
        if (thenVector instanceof UnsignedBddBitVector) {
            rslt = (T)UnsignedBddBitVector.create(thenVector.factory, thenVector.bits.length);
        } else if (thenVector instanceof SignedBddBitVector) {
            rslt = (T)SignedBddBitVector.create(thenVector.factory, thenVector.bits.length);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + thenVector.getClass());
        }

        // Compute result.
        for (int i = 0; i < thenVector.bits.length; i++) {
            rslt.bits[i] = condition.ite(thenVector.getBit(i), elseVector.getBit(i));
        }
        return rslt;
    }

    /**
     * Computes an if-then-else with the given 'if' condition, 'then' bit vector and 'else' bit vector. The 'then' and
     * 'else' bit vectors must have the same representation. This operation returns a new bit vector of the same
     * representation as the 'then' and 'else' bit vectors. The condition and bit vectors on which the operation are
     * performed are neither modified nor {@link #free freed}.
     *
     * @param condition The 'if' condition.
     * @param thenVector The 'then' bit vector.
     * @param elseVector The 'else' bit vector.
     * @return The result.
     * @throws ClassCastException If the 'then' bit vector and 'else' bit vector have different representations.
     * @throws IllegalArgumentException If the 'then' bit vector and 'else' bit vector have a different length.
     */
    public static BddBitVector<?, ?> ifThenElseAny(BDD condition, BddBitVector<?, ?> thenVector,
            BddBitVector<?, ?> elseVector)
    {
        if (thenVector instanceof UnsignedBddBitVector uThenVector) {
            return ifThenElse(condition, uThenVector, (UnsignedBddBitVector)elseVector);
        } else if (thenVector instanceof SignedBddBitVector sThenVector) {
            return ifThenElse(condition, sThenVector, (SignedBddBitVector)elseVector);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + thenVector.getClass());
        }
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be strictly less than the given bit
     * vector. This bit vector and the given bit vector must have the same representation. This operation returns a new
     * BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be strictly less than the given bit
     *     vector.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract BDD lessThan(T other);

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be strictly less than the given bit
     * vector. This bit vector and the given bit vector must have the same representation. This operation returns a new
     * BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be strictly less than the given bit
     *     vector.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD lessThanAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.lessThan((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.lessThan((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be less than or equal to the given
     * bit vector. This bit vector and the given bit vector must have the same representation. This operation returns a
     * new BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be less than or equal to the given
     *     bit vector.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract BDD lessOrEqual(T other);

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be less than or equal to the given
     * bit vector. This bit vector and the given bit vector must have the same representation. This operation returns a
     * new BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be less than or equal to the given
     *     bit vector.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD lessOrEqualAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.lessOrEqual((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.lessOrEqual((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be strictly greater than the given
     * bit vector. This bit vector and the given bit vector must have the same representation. This operation returns a
     * new BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be strictly greater than the given
     *     bit vector.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD greaterThan(T other) {
        BDD le = lessOrEqual(other);
        BDD gt = le.not();
        le.free();
        return gt;
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be strictly greater than the given
     * bit vector. This bit vector and the given bit vector must have the same representation. This operation returns a
     * new BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be strictly greater than the given
     *     bit vector.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD greaterThanAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.greaterThan((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.greaterThan((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be greater than or equal to the
     * given bit vector. This bit vector and the given bit vector must have the same representation. This operation
     * returns a new BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free
     * freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be greater than or equal to the
     *     given bit vector.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD greaterOrEqual(T other) {
        BDD lt = lessThan(other);
        BDD ge = lt.not();
        lt.free();
        return ge;
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be greater than or equal to the
     * given bit vector. This bit vector and the given bit vector must have the same representation. This operation
     * returns a new BDD. The bit vectors on which the operation is performed are neither modified nor {@link #free
     * freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be greater than or equal to the
     *     given bit vector.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD greaterOrEqualAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.greaterOrEqual((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.greaterOrEqual((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be equal to the given bit vector.
     * This bit vector and the given bit vector must have the same representation. This operation returns a new BDD. The
     * bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be equal to the given bit vector.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD equalTo(T other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute result.
        BDD eq = factory.one();
        for (int i = 0; i < bits.length; i++) {
            BDD bit = this.bits[i].biimp(other.bits[i]);
            eq = eq.andWith(bit);
        }
        return eq;
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be equal to the given bit vector.
     * This bit vector and the given bit vector must have the same representation. This operation returns a new BDD. The
     * bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be equal to the given bit vector.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD equalToAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.equalTo((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.equalTo((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be unequal to the given bit vector.
     * This bit vector and the given bit vector must have the same representation. This operation returns a new BDD. The
     * bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be unequal to the given bit vector.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD unequalTo(T other) {
        BDD eq = this.equalTo(other);
        BDD uneq = eq.not();
        eq.free();
        return uneq;
    }

    /**
     * Returns a BDD indicating the conditions that must hold for this bit vector to be unequal to the given bit vector.
     * This bit vector and the given bit vector must have the same representation. This operation returns a new BDD. The
     * bit vectors on which the operation is performed are neither modified nor {@link #free freed}.
     *
     * @param other The bit vector to compare against.
     * @return A BDD indicating the conditions that must hold for this bit vector to be unequal to the given bit vector.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BDD unequalToAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.unequalTo((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.unequalTo((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Computes the minimum of this and the given bit vector. This bit vector and the given bit vector must have the
     * same representation. This operation returns a new bit vector. The bit vectors on which the operation is performed
     * are neither modified nor {@link #free freed}.
     *
     * @param other The other bit vector.
     * @return The result.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract T min(T other);

    /**
     * Computes the minimum of this and the given bit vector. This bit vector and the given bit vector must have the
     * same representation. This operation returns a new bit vector. The bit vectors on which the operation is performed
     * are neither modified nor {@link #free freed}.
     *
     * @param other The other bit vector.
     * @return The result.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BddBitVector<?, ?> minAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.min((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.min((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Computes the maximum of this and the given bit vector. This bit vector and the given bit vector must have the
     * same representation. This operation returns a new bit vector. The bit vectors on which the operation is performed
     * are neither modified nor {@link #free freed}.
     *
     * @param other The other bit vector.
     * @return The result.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public abstract T max(T other);

    /**
     * Computes the maximum of this and the given bit vector. This bit vector and the given bit vector must have the
     * same representation. This operation returns a new bit vector. The bit vectors on which the operation is performed
     * are neither modified nor {@link #free freed}.
     *
     * @param other The other bit vector.
     * @return The result.
     * @throws ClassCastException If this bit vector and the given bit vector have different representations.
     * @throws IllegalArgumentException If this bit vector and the given bit vector have a different length.
     */
    public BddBitVector<?, ?> maxAny(BddBitVector<?, ?> other) {
        if (this instanceof UnsignedBddBitVector uThis) {
            return uThis.max((UnsignedBddBitVector)other);
        } else if (this instanceof SignedBddBitVector sThis) {
            return sThis.max((SignedBddBitVector)other);
        } else {
            throw new AssertionError("Unknown bit vector representation: " + this.getClass());
        }
    }

    /**
     * Frees the bit vector, {@link BDD#free freeing} the BDDs representing the bits. The bit vector should not be used
     * after calling this method.
     */
    public void free() {
        for (int i = 0; i < bits.length; i++) {
            bits[i].free();
        }
        factory = null;
        bits = null;
    }

    @Override
    public String toString() {
        if (bits == null) {
            return "freed";
        }
        return Arrays.toString(bits);
    }
}
