//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.typechecker.annotations.builtin;

import static org.eclipse.escet.common.java.Lists.first;
import static org.eclipse.escet.common.java.Strings.fmt;

import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.metamodel.cif.ComplexComponent;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.annotations.Annotation;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotationArgument;
import org.eclipse.escet.cif.metamodel.cif.types.BoolType;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.typechecker.annotations.AnnotationProblemReporter;
import org.eclipse.escet.cif.typechecker.annotations.AnnotationProvider;
import org.eclipse.escet.common.typechecker.SemanticProblemSeverity;

/**
 * Annotation provider for "requirement:reachable" annotations.
 *
 * <p>
 * A reachability requirement annotation can be added to components. It has exactly one unnamed argument, and this
 * argument must have a boolean-typed expression as its value. The argument specifies a predicate over states. For each
 * state in the controlled system, it must either satisfy that predicate or it must be possible to reach a state where
 * the predicate holds.
 * </p>
 */
public class RequirementReachableAnnotationProvider extends AnnotationProvider {
    @Override
    public final void checkAnnotation(AnnotatedObject annotatedObject, Annotation annotation,
            AnnotationProblemReporter reporter)
    {
        // Annotation must be on a component. Note that component definition/instantiation has already been eliminated.
        if (!(annotatedObject instanceof ComplexComponent)) {
            reporter.reportProblem(annotation, "annotation must be specified on a specification, component, component "
                    + "definition or component instantiation.", annotation.getPosition(),
                    SemanticProblemSeverity.ERROR);
        }

        // Check for exactly one argument.
        if (annotation.getArguments().size() != 1) {
            reporter.reportProblem(annotation, "annotation must have exactly one argument.", annotation.getPosition(),
                    SemanticProblemSeverity.ERROR);
            return;
        }

        // Check that the argument is unnamed.
        AnnotationArgument arg = first(annotation.getArguments());
        if (arg.getName() != null) {
            reporter.reportProblem(annotation, "unsupported named argument.", arg.getPosition(),
                    SemanticProblemSeverity.ERROR);
            // Non-fatal problem.
        }

        // Check that argument is a predicate.
        CifType valueType = CifTypeUtils.normalizeType(arg.getValue().getType());
        if (!(valueType instanceof BoolType)) {
            reporter.reportProblem(annotation,
                    fmt("argument must have a value of type \"bool\", but has a value of type \"%s\".",
                            CifTextUtils.typeToStr(valueType)),
                    arg.getValue().getPosition(), SemanticProblemSeverity.ERROR);
        }
    }

    @Override
    public final void checkGlobal(Specification spec, AnnotationProblemReporter reporter) {
        // Do nothing.
    }
}
