/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler;

import com.google.common.collect.Iterables;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import lsat_graph.ActionTask;
import lsat_graph.DispatchGroupTask;
import lsat_graph.PeripheralActionTask;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.lsat.common.graph.directed.DirectedGraph;
import org.eclipse.lsat.common.graph.directed.Edge;
import org.eclipse.lsat.common.scheduler.graph.Dependency;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.graph.TaskDependencyGraph;
import org.eclipse.lsat.common.scheduler.graph.TimeStepData;
import org.eclipse.lsat.motioncalculator.MotionException;
import org.eclipse.lsat.timing.util.ITimingCalculator;
import org.eclipse.lsat.timing.util.MotionData;
import org.eclipse.lsat.timing.util.SpecificationException;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.ArrayExtensions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@FinalFieldsConstructor
@SuppressWarnings("all")
public class AddExecutionData {
  private static final Logger LOGGER = LoggerFactory.getLogger(AddExecutionData.class);

  @Extension
  private final ITimingCalculator timingCalculator;

  public <T extends Task> TaskDependencyGraph<T> transformModel(final TaskDependencyGraph<T> graph, final String[] filter) throws SpecificationException, MotionException {
    AddExecutionData.LOGGER.debug("Starting transformation");
    this.<T, Dependency>addExecutionData(graph, filter);
    AddExecutionData.LOGGER.debug("Finished transformation");
    return graph;
  }

  private <T extends Task, E extends Edge> void addExecutionData(final DirectedGraph<T, E> graph, final String[] filter) throws SpecificationException, MotionException {
    final Consumer<DirectedGraph<T, E>> _function = (DirectedGraph<T, E> it) -> {
      try {
        this.<T, E>addExecutionData(it, filter);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    graph.getSubGraphs().forEach(_function);
    final Function1<PeripheralActionTask, Boolean> _function_1 = (PeripheralActionTask it) -> {
      return Boolean.valueOf(((filter.length == 0) || ArrayExtensions.contains(filter, it.getAction().fqn())));
    };
    final Consumer<PeripheralActionTask> _function_2 = (PeripheralActionTask it) -> {
      try {
        this.doAddExecutionData(it);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    IterableExtensions.<PeripheralActionTask>filter(Iterables.<PeripheralActionTask>filter(graph.getNodes(), PeripheralActionTask.class), _function_1).forEach(_function_2);
  }

  private void _doAddExecutionData(final DispatchGroupTask task) {
    EMap<String, EList<TimeStepData>> _executionData = task.getExecutionData();
    boolean _tripleEquals = (_executionData == null);
    if (_tripleEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Expected execution data to be set for task: ");
      _builder.append(task);
      throw new IllegalStateException(_builder.toString());
    }
  }

  private void _doAddExecutionData(final ActionTask<?> task) throws SpecificationException, MotionException {
    Set<Map.Entry<String, MotionData>> _entrySet = this.timingCalculator.calculateMotionData(task.getAction()).entrySet();
    for (final Map.Entry<String, MotionData> entry : _entrySet) {
      {
        List<double[]> _timeData = entry.getValue().getTimeData();
        for (final double[] timeDataValues : _timeData) {
          {
            final BasicEList<BigDecimal> timeDataBigDecimal = new BasicEList<BigDecimal>();
            for (final double d : timeDataValues) {
              timeDataBigDecimal.add(BigDecimal.valueOf(d));
            }
            task.addExecutionData(entry.getKey(), timeDataBigDecimal);
          }
        }
        List<String> _parameterNames = entry.getValue().getParameterNames();
        for (final String paramName : _parameterNames) {
          boolean _contains = task.getExecutionDataParameters().contains(paramName);
          boolean _not = (!_contains);
          if (_not) {
            task.getExecutionDataParameters().add(paramName);
          }
        }
      }
    }
  }

  @XbaseGenerated
  private void doAddExecutionData(final Task task) throws SpecificationException, MotionException {
    if (task instanceof ActionTask) {
      _doAddExecutionData((ActionTask<?>)task);
      return;
    } else if (task instanceof DispatchGroupTask) {
      _doAddExecutionData((DispatchGroupTask)task);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(task).toString());
    }
  }

  public AddExecutionData(final ITimingCalculator timingCalculator) {
    super();
    this.timingCalculator = timingCalculator;
  }
}
