/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.external.api.model;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.lsat.common.util.NamingUtil;

public class LsatBase {

    private static final Map<String, String> FIELD_TRANSLATION = Map.of(
            "predecessorMove", "action",
            "successorMove", "action",
            "source", "task",
            "target", "task",
            "inTask", "task",
            "outTask", "task"
    );

    // Mark as transient to avoid exporting to JSON
    private transient boolean resolved = false;

    public boolean isResolved() {
        return resolved;
    }

    public void resolveReferences(LsatData lsatData) {
        resolveReferences(lsatData, null);
    }

    private void resolveReferences(LsatData lsatData, LsatBase container) {
        resolved = true;
        List<Field> fields = new ArrayList<>();
        Class<?> cls = this.getClass();
        while (cls != LsatBase.class) {
            var newFields = cls.getDeclaredFields();
            fields.addAll(Arrays.asList(newFields));
            cls = cls.getSuperclass();
        }
        var fieldNames = fields.stream().map(Field::getName).collect(Collectors.toSet());

        // # Round 1: find reference to all fields ending with "Id" or "Ids"
        for (var field: fields) {
            var fieldName = field.getName();
            if (fieldName.endsWith("Id") || fieldName.endsWith("Ids")) {
                var referenceFieldName = "";
                if (fieldName.endsWith("Id")) {
                    referenceFieldName = fieldName.substring(0, fieldName.length() - 2);
                } else {
                    referenceFieldName = fieldName.substring(0, fieldName.length() - 3);
                }
                var lookupName = NamingUtil.makePlural(translateField(referenceFieldName));
                Map<String, Object> lookupData;
                if (lookupName.equals("actions")) {
                    if (container instanceof Activity) {
                        lookupData = getMapAttr(container, "actions");
                    } else {
                        var activities = getMapAttr(container, "activities");
                        var activity = activities.get(getAttr(this, "activityId"));
                        lookupData = getMapAttr(activity, "actions");
                    }
                } else {
                    lookupData = getMapAttr(lsatData, lookupName);
                }
                if (fieldName.endsWith("Ids")) {
                    referenceFieldName = NamingUtil.makePlural(referenceFieldName);
                }

                var obj = getAttr(this, field);
                if (obj instanceof List listOfIds) {
                    @SuppressWarnings("unchecked")
                    var objList = (List<LsatBase>)getAttr(this, referenceFieldName);
                    for (var f: listOfIds) {
                        var data = (LsatBase)lookupData.get(f);
                        objList.add(data);
                    }
                    setAttr(this, referenceFieldName, objList);
                } else {
                    var data = (LsatBase)lookupData.get(obj);
                    setAttr(this, referenceFieldName, data);
                }
            }
        }

        // Round 2: iterate over all fields and recursively
        for (var field: fields) {
            var fieldName = field.getName();
            if (fieldNames.contains(fieldName + "Id") || fieldNames.contains(fieldName + "Ids")) {
                continue;
            }
            if (field.getType() == List.class) {
                var objList = (List<?>)getAttr(this, field);
                for (var obj: objList) {
                    if (obj instanceof LsatBase lsatBase) {
                        if (!lsatBase.isResolved()) {
                            lsatBase.resolveReferences(lsatData, this);
                        }
                    } else {
                        break; // if one object in the list is not of type LsatBase, none are
                    }
                }
            } else if (field.getType() == Map.class) {
                var objMap = (Map<?, ?>)getAttr(this, field);
                for (var obj: objMap.values()) {
                    if (obj instanceof LsatBase lsatBase) {
                        if (!lsatBase.isResolved()) {
                            lsatBase.resolveReferences(lsatData, this);
                        }
                    } else {
                        break; // if one object in the map is not of type LsatBase, none are
                    }
                }
            } else if (LsatBase.class.isAssignableFrom(field.getType())) {
                var lsatBase = (LsatBase)getAttr(this, field);
                if (lsatBase != null && !lsatBase.isResolved()) {
                    lsatBase.resolveReferences(lsatData, this);
                }
            }
        }
    }

    private String translateField(String fieldName) {
        if (this.getClass().getSimpleName().equals("ActivityDependency")
                && (fieldName.equals("source") || fieldName.equals("target")))
        {
            return "action";
        } else {
            return FIELD_TRANSLATION.containsKey(fieldName) ? FIELD_TRANSLATION.get(fieldName) : fieldName;
        }
    }

    private Map<String, Object> getMapAttr(Object container, String fieldName) {
        Map<String, Object> dataMap;
        var attr = getAttr(container, fieldName);
        if (attr instanceof Map<?, ?>) {
            @SuppressWarnings("unchecked")
            var cast = (Map<String, Object>)attr;
            dataMap = cast;
        } else if (attr == null) {
            dataMap = null;
        } else {
            throw new IllegalStateException(
                    "Expected a Map<String,Object> for " + fieldName + " but got: " + attr.getClass());
        }
        return dataMap;
    }

    private Object getAttr(Object container, String fieldName) {
        Field field;
        try {
            field = container.getClass().getDeclaredField(fieldName);
        } catch (Exception e) {
            return null;
        }
        return getAttr(container, field);
    }

    private Object getAttr(Object container, Field field) {
        Object obj = null;
        boolean accessibility = false;
        try {
            accessibility = field.canAccess(container);
        } catch (Exception e) {
            return obj;
        }
        try {
            field.setAccessible(true);
            obj = field.get(container);
        } catch (Exception e) {
            e.printStackTrace();
        }
        // Reset accessibility to its original value
        field.setAccessible(accessibility);
        return obj;
    }

    private void setAttr(Object container, String fieldName, Object value) {
        Class<?> cls = this.getClass();
        while (cls != LsatBase.class) {
            Field field = null;
            try {
                field = cls.getDeclaredField(fieldName);
            } catch (Exception e) {
            }

            if (field != null) {
                setAttr(container, field, value);
                break;
            } else {
                cls = cls.getSuperclass();
            }
        }
    }

    private void setAttr(Object container, Field field, Object value) {
        boolean accessibility = false;
        try {
            accessibility = field.canAccess(container);
        } catch (Exception e) {
            return;
        }
        try {
            field.setAccessible(true);
            field.set(container, value);
        } catch (Exception e) {
            e.printStackTrace();
        }
        // Reset accessibility to its original value
        field.setAccessible(accessibility);
    }
}
