/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.datastore;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.zip.CRC32;
import org.eclipse.rdf4j.common.io.ByteArrayUtil;
import org.eclipse.rdf4j.sail.nativerdf.NativeStore;
import org.eclipse.rdf4j.sail.nativerdf.ValueStore;
import org.eclipse.rdf4j.sail.nativerdf.datastore.DataFile;
import org.eclipse.rdf4j.sail.nativerdf.datastore.HashFile;
import org.eclipse.rdf4j.sail.nativerdf.datastore.IDFile;
import org.eclipse.rdf4j.sail.nativerdf.datastore.RecoveredDataException;
import org.eclipse.rdf4j.sail.nativerdf.model.NativeValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataStore
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(DataStore.class);
    private final DataFile dataFile;
    private final IDFile idFile;
    private final HashFile hashFile;
    private ValueStore valueStore;

    public DataStore(File dataDir, String filePrefix) throws IOException {
        this(dataDir, filePrefix, false);
    }

    public DataStore(File dataDir, String filePrefix, boolean forceSync) throws IOException {
        this.dataFile = new DataFile(new File(dataDir, filePrefix + ".dat"), forceSync);
        this.idFile = new IDFile(new File(dataDir, filePrefix + ".id"), forceSync);
        this.hashFile = new HashFile(new File(dataDir, filePrefix + ".hash"), forceSync);
    }

    public DataStore(File dataDir, String filePrefix, boolean forceSync, ValueStore valueStore) throws IOException {
        this(dataDir, filePrefix, forceSync);
        this.valueStore = valueStore;
    }

    public byte[] getData(int id) throws IOException {
        assert (id > 0) : "id must be larger than 0, is: " + id;
        long offset = this.idFile.getOffset(id);
        if (offset != 0L) {
            byte[] data = this.dataFile.getData(offset);
            assert (data != null) : "data must not be null";
            if (logger.isDebugEnabled()) {
                logger.debug("getData thread={} id={} offset={} resultLength={}", new Object[]{DataStore.threadName(), id, offset, data.length});
            }
            if (data.length == 0 && NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES) {
                data = this.attemptToRecoverCorruptData(id, offset, data);
            }
            return data;
        }
        return null;
    }

    private byte[] attemptToRecoverCorruptData(int id, long offset, byte[] data) throws IOException {
        try {
            long offsetNoCache = this.idFile.getOffsetNoCache(id);
            if (offset != offsetNoCache) {
                logger.error("IDFile cache mismatch for id {}: cached={}, raw={}. Using raw.", new Object[]{id, offset, offsetNoCache});
                offset = offsetNoCache;
                data = this.dataFile.getData(offset);
            }
        }
        catch (IOException offsetNoCache) {
            // empty catch block
        }
        long startData = offset + 4L;
        for (int prev = id - 1; prev >= 1; --prev) {
            long po = this.idFile.getOffset(prev);
            try {
                long poRaw = this.idFile.getOffsetNoCache(prev);
                if (po != poRaw) {
                    logger.error("IDFile cache mismatch for prev id {}: cached={}, raw={}. Using raw.", new Object[]{prev, po, poRaw});
                    po = poRaw;
                }
            }
            catch (IOException poRaw) {
                // empty catch block
            }
            if (po <= 0L) continue;
            try {
                byte[] prevData = this.dataFile.getData(po);
                if (prevData == null || prevData.length <= 0) continue;
                try {
                    if (this.valueStore != null && Thread.currentThread().getStackTrace().length < 512) {
                        NativeValue nativeValue = this.valueStore.data2value(prev, prevData);
                        logger.warn("Data in previous ID ({}) is: {}", (Object)prev, (Object)nativeValue);
                    } else {
                        logger.warn("Data in previous ID ({}) is: {}", (Object)prev, (Object)new String(prevData, StandardCharsets.UTF_8));
                    }
                }
                catch (Exception nativeValue) {
                    // empty catch block
                }
                startData = po + 4L + (long)prevData.length;
                break;
            }
            catch (Exception prevData) {
                // empty catch block
            }
        }
        long endOffset = 0L;
        int maxId = this.idFile.getMaxID();
        for (int next = id + 1; next <= maxId; ++next) {
            long no = this.idFile.getOffset(next);
            try {
                long noRaw = this.idFile.getOffsetNoCache(next);
                if (no != noRaw) {
                    logger.error("IDFile cache mismatch for next id {}: cached={}, raw={}. Using raw.", new Object[]{next, no, noRaw});
                    no = noRaw;
                }
            }
            catch (IOException noRaw) {
                // empty catch block
            }
            if (no <= 0L) continue;
            try {
                byte[] nextData = this.dataFile.getData(no);
                if (nextData == null || nextData.length <= 0) continue;
                try {
                    if (this.valueStore != null && Thread.currentThread().getStackTrace().length < 512) {
                        NativeValue nativeValue = this.valueStore.data2value(next, nextData);
                        logger.warn("Data in next ID ({}) is: {}", (Object)next, (Object)nativeValue);
                    } else {
                        logger.warn("Data in next ID ({}) is: {}", (Object)next, (Object)new String(nextData, StandardCharsets.UTF_8));
                    }
                }
                catch (Exception exception) {
                    // empty catch block
                }
                endOffset = no;
                break;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (endOffset == 0L) {
            endOffset = this.dataFile.getFileSize();
        }
        if (endOffset > startData) {
            byte[] recovered = this.dataFile.tryRecoverBetweenOffsets(Math.max(0L, startData - 4L), endOffset);
            throw new RecoveredDataException(id, recovered);
        }
        assert (data.length == 0);
        return data;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getID(byte[] queryData) throws IOException {
        int id;
        assert (queryData != null) : "queryData must not be null";
        int hash = this.getDataHash(queryData);
        if (logger.isDebugEnabled()) {
            logger.debug("getID start thread={} hash={} summary={}", new Object[]{DataStore.threadName(), hash, DataStore.summarize(queryData)});
        }
        try (HashFile.IDIterator iter = this.hashFile.getIDIterator(hash);){
            while ((id = iter.next()) >= 0) {
                long offset = this.idFile.getOffset(id);
                byte[] data = this.dataFile.getData(offset);
                boolean match = Arrays.equals(queryData, data);
                if (logger.isDebugEnabled()) {
                    logger.debug("getID candidate thread={} hash={} candidateId={} offset={} match={} candidateSummary={}", new Object[]{DataStore.threadName(), hash, id, offset, match, DataStore.summarize(data)});
                }
                if (!match) continue;
                break;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("getID result thread={} hash={} id={}", new Object[]{DataStore.threadName(), hash, id});
        }
        return id;
    }

    public int getMaxID() {
        return this.idFile.getMaxID();
    }

    public synchronized int storeData(byte[] data) throws IOException {
        int id;
        assert (data != null) : "data must not be null";
        if (logger.isDebugEnabled()) {
            int hash = this.getDataHash(data);
            logger.debug("storeData start thread={} hash={} summary={}", new Object[]{DataStore.threadName(), hash, DataStore.summarize(data)});
        }
        if ((id = this.getID(data)) == -1) {
            int hash = this.getDataHash(data);
            long offset = this.dataFile.storeData(data);
            id = this.idFile.storeOffset(offset);
            this.hashFile.storeID(hash, id);
            if (logger.isDebugEnabled()) {
                logger.debug("storeData stored thread={} hash={} id={} offset={} summary={}", new Object[]{DataStore.threadName(), hash, id, offset, DataStore.summarize(data)});
            }
        } else if (logger.isDebugEnabled()) {
            int hash = this.getDataHash(data);
            logger.debug("storeData reuse thread={} hash={} existingId={} summary={}", new Object[]{DataStore.threadName(), hash, id, DataStore.summarize(data)});
        }
        return id;
    }

    public void sync() throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("sync thread={} invoked", (Object)DataStore.threadName());
        }
        this.hashFile.sync();
        this.idFile.sync();
        this.dataFile.sync();
    }

    public void clear() throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("clear thread={} invoked", (Object)DataStore.threadName());
        }
        try {
            this.hashFile.clear();
        }
        finally {
            try {
                this.idFile.clear();
            }
            finally {
                this.dataFile.clear();
            }
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.hashFile.close();
        }
        finally {
            try {
                this.idFile.close();
            }
            finally {
                this.dataFile.close();
            }
        }
    }

    private static String summarize(byte[] data) {
        if (data == null) {
            return "null";
        }
        return "len=" + data.length + ",hash=" + Arrays.hashCode(data);
    }

    private static String threadName() {
        return Thread.currentThread().getName();
    }

    private int getDataHash(byte[] data) {
        CRC32 crc32 = new CRC32();
        crc32.update(data);
        return (int)crc32.getValue();
    }

    public static void main(String[] args) throws Exception {
        if (args.length < 2) {
            System.err.println("Usage: java org.eclipse.rdf4j.sesame.sailimpl.nativerdf.datastore.DataStore <data-dir> <file-prefix>");
            return;
        }
        System.out.println("Dumping DataStore contents...");
        File dataDir = new File(args[0]);
        try (DataStore dataStore = new DataStore(dataDir, args[1]);){
            DataFile.DataIterator iter = dataStore.dataFile.iterator();
            while (iter.hasNext()) {
                byte[] data = iter.next();
                System.out.println(ByteArrayUtil.toHexString(data));
            }
        }
    }
}

