/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import java.util.Arrays;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * A `BreakpointMode` is provided as a option when setting breakpoints on sources or instructions.
 * <p>
 * Since 1.65
 */
@SuppressWarnings("all")
public class BreakpointMode {
  /**
   * The internal ID of the mode. This value is passed to the `setBreakpoints` request.
   */
  @NonNull
  private String mode;

  /**
   * The name of the breakpoint mode. This is shown in the UI.
   */
  @NonNull
  private String label;

  /**
   * A help text providing additional information about the breakpoint mode.
   * This string is typically shown as a hover and can be translated.
   * <p>
   * This is an optional property.
   */
  private String description;

  /**
   * Describes one or more type of breakpoint this mode applies to.
   * <p>
   * Possible values include - but not limited to those defined in {@link BreakpointModeApplicability}
   */
  @NonNull
  private String[] appliesTo;

  /**
   * The internal ID of the mode. This value is passed to the `setBreakpoints` request.
   */
  @NonNull
  public String getMode() {
    return this.mode;
  }

  /**
   * The internal ID of the mode. This value is passed to the `setBreakpoints` request.
   */
  public void setMode(@NonNull final String mode) {
    this.mode = Preconditions.checkNotNull(mode, "mode");
  }

  /**
   * The name of the breakpoint mode. This is shown in the UI.
   */
  @NonNull
  public String getLabel() {
    return this.label;
  }

  /**
   * The name of the breakpoint mode. This is shown in the UI.
   */
  public void setLabel(@NonNull final String label) {
    this.label = Preconditions.checkNotNull(label, "label");
  }

  /**
   * A help text providing additional information about the breakpoint mode.
   * This string is typically shown as a hover and can be translated.
   * <p>
   * This is an optional property.
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * A help text providing additional information about the breakpoint mode.
   * This string is typically shown as a hover and can be translated.
   * <p>
   * This is an optional property.
   */
  public void setDescription(final String description) {
    this.description = description;
  }

  /**
   * Describes one or more type of breakpoint this mode applies to.
   * <p>
   * Possible values include - but not limited to those defined in {@link BreakpointModeApplicability}
   */
  @NonNull
  public String[] getAppliesTo() {
    return this.appliesTo;
  }

  /**
   * Describes one or more type of breakpoint this mode applies to.
   * <p>
   * Possible values include - but not limited to those defined in {@link BreakpointModeApplicability}
   */
  public void setAppliesTo(@NonNull final String[] appliesTo) {
    this.appliesTo = Preconditions.checkNotNull(appliesTo, "appliesTo");
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("mode", this.mode);
    b.add("label", this.label);
    b.add("description", this.description);
    b.add("appliesTo", this.appliesTo);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    BreakpointMode other = (BreakpointMode) obj;
    if (this.mode == null) {
      if (other.mode != null)
        return false;
    } else if (!this.mode.equals(other.mode))
      return false;
    if (this.label == null) {
      if (other.label != null)
        return false;
    } else if (!this.label.equals(other.label))
      return false;
    if (this.description == null) {
      if (other.description != null)
        return false;
    } else if (!this.description.equals(other.description))
      return false;
    if (this.appliesTo == null) {
      if (other.appliesTo != null)
        return false;
    } else if (!Arrays.deepEquals(this.appliesTo, other.appliesTo))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.mode== null) ? 0 : this.mode.hashCode());
    result = prime * result + ((this.label== null) ? 0 : this.label.hashCode());
    result = prime * result + ((this.description== null) ? 0 : this.description.hashCode());
    return prime * result + ((this.appliesTo== null) ? 0 : Arrays.deepHashCode(this.appliesTo));
  }
}
