/*
 * Copyright 2015-2026 the original author or authors.
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v2.0 which
 * accompanies this distribution and is available at
 *
 * https://www.eclipse.org/legal/epl-v20.html
 */

package org.junit.platform.commons.util;

import java.util.ServiceLoader;
import java.util.function.Predicate;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.apiguardian.api.API;

/**
 * Collection of utilities for working with {@link ServiceLoader}.
 *
 * <h2>DISCLAIMER</h2>
 *
 * <p>These utilities are intended solely for usage within the JUnit framework
 * itself. <strong>Any usage by external parties is not supported.</strong>
 * Use at your own risk!
 *
 * @since 1.11
 */
@API(status = API.Status.INTERNAL, since = "1.11")
public class ServiceLoaderUtils {

	private ServiceLoaderUtils() {
		/* no-op */
	}

	/**
	 * Filters the supplied service loader using the supplied predicate.
	 *
	 * @param <T> the type of the service
	 * @param serviceLoader the service loader to be filtered
	 * @param providerPredicate the predicate to filter the loaded services
	 * @return a stream of loaded services that match the predicate
	 */
	public static <T> Stream<T> filter(ServiceLoader<T> serviceLoader,
			Predicate<? super Class<? extends T>> providerPredicate) {
		return StreamSupport.stream(serviceLoader.spliterator(), false).filter(it -> {
			@SuppressWarnings("unchecked")
			Class<? extends T> type = (Class<? extends T>) it.getClass();
			return providerPredicate.test(type);
		});
	}

}
