/*
 *  Copyright (C) 2018 Team Gateship-One
 *  (Hendrik Borghorst & Frederik Luetkes)
 *
 *  The AUTHORS.md file contains a detailed contributors list:
 *  <https://gitlab.com/matthias-weiss/musicala/blob/master/AUTHORS.md>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package at.weiss.matthias.musicala.application.utils;

import android.view.KeyEvent;

import at.weiss.matthias.musicala.mpdservice.handlers.serverhandler.MPDCommandHandler;

import java.util.Timer;
import java.util.TimerTask;

public class HardwareKeyHandler {
    /**
     * Interval for repeating the volume events in ms.
     */
    private final static int VOLUME_CONTROL_REPEAT_PERIOD = 200;

    private static HardwareKeyHandler mInstance;

    private int mVolumeStepSize = 1;

    private Timer mRepeatTimer;

    /**
     * Singleton pattern
     * @return The singleton instance for this handler.
     */
    public static HardwareKeyHandler getInstance() {
        if (mInstance == null) {
            // Create singleton instance
            mInstance = new HardwareKeyHandler();
        }
        return mInstance;
    }

    /**
     * Can be called from the {@link android.app.Activity}s. that catches the key event.
     * This method ensures consistent behavior for button handling in all {@link android.app.Activity}s.
     * @param event
     * @return
     */
    public boolean handleKeyEvent(KeyEvent event, boolean volumeControl) {
        int action = event.getAction();
        int keyCode = event.getKeyCode();
        switch (keyCode) {
            case KeyEvent.KEYCODE_VOLUME_UP:
                if (!volumeControl) {
                    return false;
                }
                if (action == KeyEvent.ACTION_DOWN) {
                    MPDCommandHandler.increaseVolume(mVolumeStepSize);
                    // If this event is emitted the first time start an timer to repeat this action
                    if (mRepeatTimer == null) {
                        mRepeatTimer = new Timer();
                        mRepeatTimer.scheduleAtFixedRate(new IncreaseVolumeTask(), VOLUME_CONTROL_REPEAT_PERIOD, VOLUME_CONTROL_REPEAT_PERIOD);
                    }
                } else {
                    // Key is released. Stop running timer.
                    if (null != mRepeatTimer) {
                        mRepeatTimer.cancel();
                        mRepeatTimer.purge();
                        mRepeatTimer = null;
                    }
                }
                return true;
            case KeyEvent.KEYCODE_VOLUME_DOWN:
                if (!volumeControl) {
                    return false;
                }
                if (action == KeyEvent.ACTION_DOWN) {
                    MPDCommandHandler.decreaseVolume(mVolumeStepSize);
                    // If this event is emitted the first time start an timer to repeat this action
                    if (mRepeatTimer == null) {
                        mRepeatTimer = new Timer();
                        mRepeatTimer.scheduleAtFixedRate(new DecreaseVolumeTask(), VOLUME_CONTROL_REPEAT_PERIOD, VOLUME_CONTROL_REPEAT_PERIOD);
                    }
                } else {
                    // Key is released. Stop running timmer.
                    if (null != mRepeatTimer) {
                        mRepeatTimer.cancel();
                        mRepeatTimer.purge();
                        mRepeatTimer = null;
                    }
                }
                return true;
            case KeyEvent.KEYCODE_MEDIA_PLAY: {
                if (action == KeyEvent.ACTION_UP) {
                    MPDCommandHandler.play();
                }
                return true;
            }
            case KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE: {
                if (action == KeyEvent.ACTION_UP) {
                    MPDCommandHandler.togglePause();
                }
                return true;
            }
            case KeyEvent.KEYCODE_MEDIA_PAUSE: {
                if (action == KeyEvent.ACTION_UP) {
                    MPDCommandHandler.pause();
                }
                return true;
            }
            case KeyEvent.KEYCODE_MEDIA_STOP: {
                if (action == KeyEvent.ACTION_UP) {
                    MPDCommandHandler.stop();
                }
                return true;
            }
            case KeyEvent.KEYCODE_MEDIA_NEXT: {
                if (action == KeyEvent.ACTION_UP) {
                    MPDCommandHandler.nextSong();
                }
                return true;
            }
            case KeyEvent.KEYCODE_MEDIA_PREVIOUS: {
                if (action == KeyEvent.ACTION_UP) {
                    MPDCommandHandler.previousSong();
                }
                return true;
            }
            default:
                return false;
        }
    }

    public void setVolumeStepSize(int stepSize) {
        mVolumeStepSize = stepSize;
    }

    /**
     * Simple class to repeatably increase the volume.
     */
    private class IncreaseVolumeTask extends TimerTask {

        @Override
        public void run() {
            MPDCommandHandler.increaseVolume(mVolumeStepSize);
        }
    }


    /**
     * Simple class to repeatably decrease the volume.
     */
    private class DecreaseVolumeTask extends TimerTask {

        @Override
        public void run() {
            MPDCommandHandler.decreaseVolume(mVolumeStepSize);
        }
    }
}
