import 'package:flutter/material.dart';

import 'package:spartathlon_app/models/AppData.dart';
import 'package:spartathlon_app/models/Athlete.dart';
import 'package:spartathlon_app/models/BlogEntry.dart';
import 'package:spartathlon_app/providers/AthleteProvider.dart';
import 'package:spartathlon_app/providers/BlogProvider.dart';
import 'package:spartathlon_app/util/ApiCache.dart';
import 'package:spartathlon_app/util/DistanceUnit.dart';
import 'package:spartathlon_app/util/Themes.dart';
import 'package:spartathlon_app/util/Utils.dart';

// See https://ericwindmill.com/posts/inherited_widget/ for an intro to InheritedWidget

/// Root widget for the Spartathlon app.
///
/// It holds the data that needs to be
/// shared across the app (like the list of athletes) in its state, which can
/// be accessed from throughout the app since its an InheritedWidget.
class AppStateContainer extends StatefulWidget {
  final Widget child;

  AppStateContainer({@required this.child});

  @override
  AppStateContainerState createState() => AppStateContainerState();

  static AppStateContainerState of(BuildContext context) {
    return (context.inheritFromWidgetOfExactType(_InheritedAppStateContainer)
            as _InheritedAppStateContainer)
        .data;
  }
}

class AppStateContainerState extends State<AppStateContainer> {
  AppData data;
  ThemeData theme;

  @override
  void initState() {
    super.initState();
    data = AppData(); // Contains empty lists per default
    loadData(false);
  }

  @override
  Widget build(BuildContext context) {
    return _InheritedAppStateContainer(data: this, child: widget.child);
  }

  /// Evicts the cache no matter of the current network connection state
  Future<void> evictCache() async {
    print('Ignoring network and clearing cache...\n');
    await AthleteCacheManager().emptyCache();
    await BlogCacheManager().emptyCache();
    setState(() {
      data.athletes = null;
      data.blogEntries = null;
      data.favoriteAthletes = null;
      loadData(false);
    });
  }

  Future<ThemeData> loadTheme() async {
    AppTheme appTheme = await getThemeFromPrefs();
    theme = getThemeData(appTheme);
    return theme;
  }

  /// Load all data for the app (athletes, blog entries, etc)
  ///
  /// If and only if a network connection is available, [forceReload] will empty
  /// the cache and load all data from the API again.
  /// Otherwise [forceReload] has no effect.
  Future<void> loadData(bool forceReload) async {
    // Set connectivity status
    bool hasConn = await hasConnectivity();

    data.distanceUnit = await getDistanceUnit();
    setState(() {
      data.isLoading = true;
      data.hasError = false;
      data.hasConnection = hasConn;
    });

    // Evict caches if requested and reasonable
    if (hasConn && forceReload) {
      print('Clearing cache...\n');
      await AthleteCacheManager().emptyCache();
      await BlogCacheManager().emptyCache();
    }

    // Try and fetch all data for the app
    try {
      List<Athlete> athletes = await AthleteProvider.getAthletes();
      List<Athlete> favoriteAthletes =
          await AthleteProvider.getFavoriteAthletes(athletes);
      List<BlogEntry> blogEntries = await BlogProvider.getBlogEntries();

      setState(() {
        data.athletes = athletes;
        data.favoriteAthletes = favoriteAthletes;
        data.blogEntries = blogEntries;
      });
    } catch (e) {
      print('Error loading data: ' + e.toString());
      // Tell Flutter to rebuild and tell the widget that an error occurred.
      setState(() => data.hasError = true);
    }

    print('Finished loading data. Got:');
    print(data);

    // Tell Flutter to rebuild and tell the widget that loading has finished.
    setState(() => data.isLoading = false);
  }

  /// Helper function without arguments to call [loadData], forcing all data to
  /// be loaded again
  void reloadAfterError() {
    loadData(true);
  }

  /// Sets the apps current theme at runtime to newTheme.
  ///
  /// Updates both SharedPreferences and InheritedWidget
  void setTheme(AppTheme newTheme) {
    setThemeIdToPrefs(newTheme);

    ThemeData newThemeData = getThemeData(newTheme);
    setState(() {
      theme = newThemeData;
    });
  }

  void setAthleteList(List<Athlete> athletes) {
    setState(() {
      data.athletes = athletes;
    });
  }

  void setFavoriteAthletes(List<Athlete> favoriteAthletes) {
    data.favoriteAthletes = favoriteAthletes;
  }

  void setToggleFavoriteAthlete(String id) {
    AthleteProvider.setToggleAthleteFavoriteToPrefs(id);
    Athlete a = AthleteProvider.getAthleteById(data.athletes, id);

    if (a != null) {
      if (data.favoriteAthletes.contains(a)) {
        setState(() {
          data.favoriteAthletes.remove(a);
        });
      } else {
        setState(() {
          data.favoriteAthletes.add(a);
        });
      }
    }
    data.favoriteAthletes.sort();
  }

  void setBlogEntryList(List<BlogEntry> blogEntries) {
    setState(() {
      data.blogEntries = blogEntries;
    });
  }

  void setDistanceUnit(DistanceUnit unit){
    // Assumes this is written to the SharedPreferences elsewhere
    setState(() {
      data.distanceUnit = unit;
    });
  }
}

class _InheritedAppStateContainer extends InheritedWidget {
  final AppStateContainerState data;

  _InheritedAppStateContainer({
    Key key,
    @required this.data,
    @required Widget child,
  }) : super(key: key, child: child);

  // TODO Be more clever about checking whether the state has actually changed
  // and widget reliant on us need to be notified to rebuild
  @override
  bool updateShouldNotify(_InheritedAppStateContainer old) => true;
}
