import 'dart:async';
import 'dart:convert';

import 'package:html/parser.dart' show parse;
import 'package:html/dom.dart';

import 'package:spartathlon_app/models/BlogEntry.dart';
import 'package:spartathlon_app/util/ApiCache.dart';

/// Provides a list of blog entries, mostly race reports
///
/// For the former, it uses a combination of parsing the blog entries on the
/// website and using the data from the JSON provided by the API.
/// This is because the API a) only returns few articles (and their storyIds are
/// not always linearly increasing, because why would they?) and because b) the
/// API doesn't return links to the image.
///

// TODO The API redirects to unencrypted HTTP. Talk to ISA to get that fixed.

class BlogProvider {
  // TODO Work with ISA to make HTML nicer or run own backend. These HTML hacks are awful.
  static const String baseUrl = 'https://spartathlon.gr';
  static List<BlogEntry> blogEntries;

  /// Returns a list of BlogEntries, where fullText might still be null!
  static Future<List<BlogEntry>> getBlogEntries() async {
    blogEntries = List<BlogEntry>();
    print("Getting blog entries");

    var blogOverviewHtml = await BlogCacheManager().getSingleFile(
        'https://spartathlon.gr/en/newsroom-en/web-news-en.html');

    String htmlContent = await blogOverviewHtml.readAsString();
    Document blogOverviewDom = parse(htmlContent);

    // Extract blog articles from DOM object using helper method
    List<Node> articles =
        _traverseDomTree(blogOverviewDom, blogOverviewDom.children[0]);
    for (Node n in articles) {
      BlogEntry i = await _getNewsItemFromArticleTag(n);
      blogEntries.add(i);
    }
    return blogEntries;
  }

  /// The following methods are only for internal use in this class.

  /// Returns a list of all 'article' nodes in the DOM tree
  static List<Node> _traverseDomTree(Document dom, Node start) {
    List<Node> articles = new List<Node>();
    Node node = start;
    if (node.children != null) {
      for (Node child in node.children) {
        if (child.toString() == '<html article>') {
          articles.add(child);
        }
        articles.addAll(_traverseDomTree(dom, child));
      }
    }
    return articles;
  }

  static Future<BlogEntry> _getNewsItemFromArticleTag(Node item) async {
    Node a = item.children[0];

    String url = baseUrl + a.attributes.entries.elementAt(0).value;

    int storyId;
    // Some storyIds >= 100
    try {
      storyId = int.parse(url.substring(50, 53));
    } catch (e) {
      storyId = int.parse(url.substring(50, 52));
    }

    String imageUrl =
        baseUrl + a.children[0].attributes.entries.elementAt(0).value;

    String apiRequestUrl =
        BASE_URL_API_ISA + 'getStory&storyID=' + storyId.toString();

    String headline;
    String teaserText;
    String fullText;

    var response = await BlogCacheManager().getSingleFile(apiRequestUrl);
    String responseAsString = response.readAsStringSync();

    Map storyAsJson = json.decode(responseAsString)['message'] as Map;
    headline = storyAsJson['title'];
    var tmp = parse(storyAsJson['short']);
    teaserText = parse(tmp.body.text).documentElement.text;
    fullText = storyAsJson['full'];

    // Build blog entry object
    BlogEntry i = BlogEntry(
      url: url,
      storyId: storyId,
      headline: headline,
      teaserText: teaserText,
      fullText: fullText,
      imageUrl: imageUrl,
    );
    return i;
  }
}
